/**
 * bpmn-plugins - Geasy BPMN Editor Plugins - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.petalsbpm.server.plugin.bpmn.descriptive;

import java.io.File;
import java.net.URL;
import java.util.UUID;

import com.ebmwebsourcing.easybox.api.XmlContextFactory;
import com.ebmwebsourcing.easybox.api.XmlObjectReader;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Definitions;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.infrastructure.DefinitionsBean;
import com.ebmwebsourcing.petalsbpm.business.domain.di.api.IBPMNDiagram;
import com.ebmwebsourcing.petalsbpm.client.plugin.bpmn.BPMNExportFormat;
import com.ebmwebsourcing.petalsbpm.client.plugin.bpmn.BPMNProjectInstance;
import com.ebmwebsourcing.petalsbpm.server.plugin.bpmn.BPMNFileHelper;
import com.ebmwebsourcing.petalsbpm.server.service.bpmn2.serverToClient.ServerToClient;
import com.ebmwebsourcing.petalsbpm.server.service.bpmndi.serverToClient.BPMNDiagramParser;
import com.ebmwebsourcing.petalsbpm.utils.server.FileHelper;
import com.ebmwebsourcing.petalsbpm.utils.server.ZipHelper;
import com.ebmwebsourcing.petalsbpm.xpdl.XPDLAdapter;
import com.ebmwebsourcing.webeditor.api.domain.project.IProjectInstanceFormat;
import com.ebmwebsourcing.webeditor.impl.domain.exception.ServiceException;
import com.ebmwebsourcing.webeditor.server.impl.service.user.UserServiceImpl;
import com.ebmwebsourcing.webeditor.server.impl.service.webeditor.WebEditorService;

public class DescriptiveBPMNFileReader {

    private BPMNProjectInstance instance;
    
    public void readFile(File projectFile, IProjectInstanceFormat projectFilesFormat, BPMNProjectInstance instance) throws ServiceException{
        this.instance = instance;
        if(this.instance.getId()==null) {
            this.instance.setId(UUID.randomUUID().toString());
        }
        try{
            switch ((BPMNExportFormat)projectFilesFormat) {
            case BPMN:
                readFile(projectFile);
                break;

            case XPDL:
                readXPDLFile(projectFile);
                break;
            }
        }
        catch(Exception e){
            e.printStackTrace();
            throw new ServiceException("An error occured during the file reading : "+e.getMessage());
        }
    }

    
    private void readXPDLFile(File file) throws Exception{
        XPDLAdapter adapter = new XPDLAdapter();
        File bpmnFile = adapter.getBPMN2FromXPDL(file.getAbsolutePath(),file.getParent());
        
        readBPMNFiles(new File[]{bpmnFile});
    }

    
    protected void readFile(File file) throws Exception {
        if(ZipHelper.getInstance().isZipFile(file)) {
            readZipFile(file);
        }
        else {
            readBPMNFiles(new File[]{file});
        }
    }
    

    private void readZipFile(File file) throws Exception {
        File unzipDir = BPMNFileHelper.getOrCreateUnzipDirectory(file);
        ZipHelper.getInstance().unzipFile(file,unzipDir);
        
        //Read the bpmn definitions and diagram from the files and fill the instance
        readBPMNFiles(unzipDir.listFiles());
        
        //copy the possible attached files into the proper directory
        File attachedFilesDir = WebEditorService.getInstance().getAttachedFilesDirectory(UserServiceImpl.getInstance().getLoggedUser(), instance);
        File bpmnFile = BPMNFileHelper.getBPMNFileInList(unzipDir.listFiles());
        File importedAttachmentsFile = new File(unzipDir,WebEditorService.ATTACHMENTS_DIR);
        if(importedAttachmentsFile.exists()) {
            for(File f : importedAttachmentsFile.listFiles()) {
                FileHelper.copyFile(f, attachedFilesDir);
            }
        }
        //normally it is not executed if the files were produced by the editor
        for(File f : unzipDir.listFiles()) {
            if(!f.equals(bpmnFile) && !f.equals(importedAttachmentsFile)) {
                FileHelper.copyFile(f, attachedFilesDir);
            }
        }
    }
    
    
    private void readBPMNFiles(File[] files) throws Exception{
        Definitions defs = null;
        DefinitionsBean res = null;
        
        try {
            File bpmnFile = BPMNFileHelper.getBPMNFileInList(files);
            assert(bpmnFile!=null);
            
            URL url       = bpmnFile.toURI().toURL();

            System.out.println("URL of BPMN file:"+url);
            
            XmlObjectReader reader = new XmlContextFactory().newContext().createReader();
            
            defs = reader.readDocument(url, Definitions.class);
            
        } catch (Exception e) { 
            e.printStackTrace();
            throw new Exception("Could not read BPMN file - make sure the bpmn file name follow the pattern BPMN_(...).xml or (...).bpmn  "+e.getLocalizedMessage());
        }
        
        try{
            res = ServerToClient.adapt(defs);
        } catch(Exception e){
            e.printStackTrace();
            throw new Exception("Could not bind server to client beans:"+e.getLocalizedMessage());
        }
        
        IBPMNDiagram resDiagram = BPMNDiagramParser.parseBPMNDiagram(defs.getBPMNDiagram()[0], res);
        
        instance.setDefinitions(res);
        instance.setBpmndiagram(resDiagram);
        instance.setName(res.getName());
    }
    

}
