/**
 * sa-manager - A Web based designer for BPMN 2.0 standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.ebmwebsourcing.jbi.bpel.sagenerator;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.QName;

import org.petalslink.abslayer.service.api.Endpoint;
import org.petalslink.abslayer.service.api.Interface;
import org.petalslink.abslayer.service.api.PartnerLinkType;

import com.ebmwebsourcing.easybpel.model.bpel.api.BPELProcess;
import com.ebmwebsourcing.easybpel.model.bpel.api.partnerLink.PartnerLink;
import com.ebmwebsourcing.easybpel.model.bpel.api.wsdlImports.Import;
import com.ebmwebsourcing.easybpel.model.bpel.impl.BPELFactoryImpl;
import com.ebmwebsourcing.jbi.sugenerator.SuGenerator;
import com.ebmwebsourcing.jbi.util.SOAAddress;
import com.ebmwebsourcing.jbi.util.Util;
import com.ebmwebsourcing.petals.services.generation.JbiUtils;

/**
 * Handle BPEL project as a SA (BPEL provide, SOAP consume for BPEL,
 * and SOAP provide for external Web Services
 */
public class BPELSaGenerator {

    private File bpelFile = null;

    private Map<File, List<SOAAddress>> bpelDescriptionFiles = null;

    private Map<SOAAddress, File> wsdlPartners = null;

    private Map<String, File> wsdls = null;

    private File bpelProjectDirectory = null;



    /**
     * Initiate BPEL project from a directory containing all needed files
     */
    public BPELSaGenerator(File bpelFile) throws Exception{
        this.bpelFile = bpelFile;
        
        if(bpelFile.isDirectory()) throw new Exception(bpelProjectDirectory + " is not a bpel file but a directory.");
        
        this.bpelProjectDirectory = bpelFile.getParentFile();

        this.wsdls = new HashMap<String, File>();
        if(bpelProjectDirectory.isDirectory()){
            String[] l = bpelProjectDirectory.list();
            for(int i=0 ; i < l.length ; i++){
                if(l[i].endsWith(".wsdl")||l[i].endsWith(".WSDL")||l[i].endsWith(".xsd")||l[i].endsWith(".XSD")){
                    this.wsdls.put(l[i], new File(bpelProjectDirectory, l[i]));
                }
            }
        }

        prepareServiceAssemblyArchive();
    }

    private  void addWsdlSOAAddress(File wsdl, SOAAddress soa){
        List<SOAAddress> value = bpelDescriptionFiles.get(wsdl);
        if(value != null){
            if(!value.contains(soa)){
                value.add(soa);
            }
        }else{
            List<SOAAddress> temp = new ArrayList<SOAAddress>();
            temp.add(soa);
            bpelDescriptionFiles.put(wsdl, temp);
        }
    }
    
    /**
     * BPEL Project parsing in order to identify partners, wsdls for these partners, ...
     * This function MUST be called before the sa Generation.
     */
    private void prepareServiceAssemblyArchive() throws Exception{
        this.bpelDescriptionFiles = new HashMap<File, List<SOAAddress>>();
        this.wsdlPartners = new HashMap<SOAAddress, File>();

        System.out.println("[SA generation] -- begin");
        BPELProcess bpel = BPELFactoryImpl.getInstance().newBPELReader().readBPEL(this.bpelFile.toURI());

        /*
         * Retrieve wsdls of BPEL
         */
        for(PartnerLink partnerlink : bpel.getPartnerLinks()) {
            PartnerLinkType plt = bpel.getImports().getPartnerLinkType(partnerlink.getPartnerLinkType());
            if(plt==null) continue;
            /*
             * Retrieve MyRole interfaces (wsdls of bpel)
             */
            if(partnerlink.getMyRole() != null){
                QName partnerInterface = plt.getRole(partnerlink.getMyRole()).getInterfaceQName();
                System.out.println("[SA generation] -- found a myrole interface (soap consume) "+partnerInterface);

                for(Import currentImport : bpel.getImports().getBPELImports()) {
                    if(currentImport.getNamespace().compareTo(URI.create(partnerInterface.getNamespaceURI())) == 0){
                        Interface interf = currentImport.getDescription().findInterface(partnerInterface);
                        if(interf != null){
                            Endpoint endpoint = Util.findEndpoint(interf, currentImport.getDescription().getServices());
                            if(endpoint != null){
                                SOAAddress adr = new SOAAddress(endpoint.getName(),
                                        endpoint.getService().getQName(),
                                        interf.getQName());

                                //TODO there must be a cleaner way to retrieve the imported file
                                File importedFile = new File(bpelProjectDirectory,currentImport.getLocation().toString());
                                addWsdlSOAAddress(importedFile, adr);

                                System.out.println("[SA generation] -- "+partnerInterface+" ok");
                            }
                            else {
                                throw new Exception("The process cannot be deployed because one of its exposed interface is abstract.");
                            }
                        }
                    }
                }
            }else {
//                QName partnerInterface = plt.getRole(partnerlink.getPartnerRole()).getInterfaceQName();
//                System.out.println("[SA generation] -- found a partnerrole interface (soap provide) "+partnerInterface);
//                for(Import currentImport : bpel.getImports().getBPELImports()) {
//                    System.out.println("[SA generation] list imports -- "+ currentImport.getLocation());
//                    if(currentImport.getNamespace().compareTo(URI.create(partnerInterface.getNamespaceURI())) == 0){
//                        if(currentImport.getDescription() != null){
//                            Interface interf = currentImport.getDescription().findInterface((QName.valueOf(partnerInterface.toString())));
//                            if(interf != null){
//                                Endpoint endpoint = Util.findEndpoint(interf, currentImport.getDescription().getServices());
//                                if(endpoint != null){
//                                    SOAAddress adr = new SOAAddress(endpoint.getName(), endpoint.getService().getQName(), interf.getQName());
//                                    adr.setAddress(endpoint.getAddress());
//                                    this.wsdlPartners.put(adr, new File(currentImport.getLocation().toString()));
//                                    System.out.println("[SA generation] -- "+partnerInterface+" ok");
//                                }
//                                else {
//                                    throw new Exception("The process cannot be deployed because one of its partner interface is abstract.");
//                                }
//                            }
//                        }
//                    }
//                }
            }
        }
    }


    public File generateZip(String saName, File outputDir) throws Exception {
        File res = null;

        if(this.bpelFile != null || this.wsdlPartners != null || bpelDescriptionFiles != null) {
            // Create the SA
            Map<String, String> suMap = new HashMap<String, String>();
            Map<String,File> suNameToSuFile = new HashMap<String,File>();

            File suBPEL = null;
            // Generate BPEL provide
            try {
                System.out.println("Install : " +this.bpelProjectDirectory.getAbsolutePath());
                suBPEL = SuGenerator.generateBPELProvide(this.bpelDescriptionFiles, this.bpelFile, this.wsdls.values(), outputDir);

                suMap.put(Util.removeExtension(suBPEL.getName()), SuGenerator.bpelComponentName);
                System.out.println("================== "+suBPEL.getName());
                suNameToSuFile.put( suBPEL.getName(), suBPEL );
            } catch (IOException e) {
                throw new Exception("su-BPEL generation failed ..."+e.getMessage());	
            }

            // Generate SOAP provides for partners
            Map<SOAAddress, File> suSOAPProvides = new HashMap<SOAAddress, File>();
            for(SOAAddress adr : this.wsdlPartners.keySet()) {

                File su = SuGenerator.generateDefaultSoapProvide(adr.getAddress(),
                        adr.getEndpoint(),
                        adr.get_interface(),
                        adr.getService(),
                        this.wsdlPartners.get(adr).getName(),
                        this.wsdls.values(),
                        outputDir);

                suSOAPProvides.put(adr, su);
                suMap.put(Util.removeExtension(su.getName()),SuGenerator.soapComponentName);
                System.out.println("===============SOAP======"+su.getName());
                suNameToSuFile.put( su.getName(), su );
            }

            // Generate SOAP consumes for BPEL 
            //TODO
            Map<SOAAddress, File> suSOAPConsumes = new HashMap<SOAAddress, File>();
            for(List<SOAAddress> adrs : bpelDescriptionFiles.values()) {
                for(SOAAddress adr : adrs){
                    File su = SuGenerator.generateSOAPConsume(adr.getEndpoint(),
                            adr.get_interface(),
                            adr.getService(),
                            null,
                            null,
                            adr.getEndpoint(),
                            outputDir);
                    suSOAPConsumes.put(adr, su);
                    suMap.put(Util.removeExtension(su.getName()), SuGenerator.soapComponentName);
                    suNameToSuFile.put( su.getName(), su );
                }
            }

            String saJbiXmlContent = JbiUtils.generateJbiXmlForSA(saName, suMap);

            res = new File(outputDir.getAbsolutePath(), saName + ".zip");

            System.out.println("SA file : "+res.getAbsolutePath());
            res = JbiUtils.createJbiArchive( res.getAbsolutePath(), saJbiXmlContent, suNameToSuFile );
        }

        return res;
    }







    public List<File> generateZips(String saName, File outputDir) {
        List<File> res = new ArrayList<File>();

        if(this.bpelFile != null || this.wsdlPartners != null || bpelDescriptionFiles != null) {
            // Create the SAs
            Map<String, String> suMap = new HashMap<String, String>();
            Map<String,File> suNameToSuFile = new HashMap<String,File>();

            File suBPEL = null;
            // Generate BPEL provide SU AND SA
            try {
                System.out.println("Install : " +this.bpelProjectDirectory.getAbsolutePath());
                //GENERATE BPEL SU
                suBPEL = SuGenerator.generateBPELProvide(this.bpelDescriptionFiles, this.bpelFile, this.wsdls.values(), outputDir);

                suMap.put(suBPEL.getName().substring(0, suBPEL.getName().length()-4), SuGenerator.bpelComponentName);
                System.out.println("================== "+suBPEL.getName());
                suNameToSuFile.put( suBPEL.getName(), suBPEL );

                //GENERATE BPEL SA
                String saBPELFileName = "sa-BPEL-"+suBPEL.getName();
                String saJbiXmlContent = JbiUtils.generateJbiXmlForSA(saBPELFileName, suMap);
                File saBPELFile = new File(this.bpelProjectDirectory, saBPELFileName);

                System.out.println("SA file to be installed: "+saBPELFile.getName());
                saBPELFile = JbiUtils.createJbiArchive( saBPELFile.getAbsolutePath(), saJbiXmlContent, suNameToSuFile );
                res.add(saBPELFile);
                suMap = new HashMap<String, String>();
                suNameToSuFile = new HashMap<String,File>();
            } catch (Exception e) {
                e.printStackTrace();
            }

            // Generate SOAP provides for partners
            Map<SOAAddress, File> suSOAPProvides = new HashMap<SOAAddress, File>();
            for(SOAAddress adr : this.wsdlPartners.keySet()) {
                try{
                    File su = SuGenerator.generateDefaultSoapProvide(adr.getAddress(),
                            adr.getEndpoint(),
                            adr.get_interface(),
                            adr.getService(),
                            this.wsdlPartners.get(adr).getName(),
                            this.wsdls.values(),
                            outputDir);

                    suSOAPProvides.put(adr, su);
                    suMap.put(su.getName().substring(0, su.getName().length()-4),SuGenerator.soapComponentName);
                    System.out.println("===============SOAP======"+su.getName());
                    suNameToSuFile.put( su.getName(), su );

                    //GENERATE SOAP Provide SA
                    String saSOAPProvideFileName = "sa-SOAP-"+su.getName();
                    String saJbiXmlContent = JbiUtils.generateJbiXmlForSA(saSOAPProvideFileName, suMap);
                    File saSOAPFile = new File(this.bpelProjectDirectory, saSOAPProvideFileName);

                    System.out.println("SA file to be installed: "+saSOAPFile.getName());
                    saSOAPFile = JbiUtils.createJbiArchive( saSOAPFile.getAbsolutePath(), saJbiXmlContent, suNameToSuFile );

                    res.add(saSOAPFile);

                }catch (Exception e) {
                    e.printStackTrace();
                }

                suMap = new HashMap<String, String>();
                suNameToSuFile = new HashMap<String,File>();

            }

            // Generate SOAP consumes for BPEL 
            //TODO
            Map<SOAAddress, File> suSOAPConsumes = new HashMap<SOAAddress, File>();
            for(List<SOAAddress> adrs : bpelDescriptionFiles.values()) {
                for(SOAAddress adr : adrs){
                    try{
                        File su = SuGenerator.generateSOAPConsume(adr.getEndpoint(),
                                adr.get_interface(),
                                adr.getService(),
                                adr.getOperation(),
                                adr.getMep(),
                                adr.getEndpoint(),
                                outputDir);

                        suSOAPConsumes.put(adr, su);
                        suMap.put(su.getName().substring(0, su.getName().length()-4), SuGenerator.soapComponentName);
                        suNameToSuFile.put( su.getName(), su );

                        //GENERATE SOAP Provide SA
                        String saSOAPConsumeFileName = "sa-SOAP-"+su.getName();
                        String saJbiXmlContent = JbiUtils.generateJbiXmlForSA(saSOAPConsumeFileName, suMap);
                        File saSOAPFile = new File(this.bpelProjectDirectory, saSOAPConsumeFileName);

                        System.out.println("SA file to be installed: "+saSOAPFile.getName());
                        saSOAPFile = JbiUtils.createJbiArchive( saSOAPFile.getAbsolutePath(), saJbiXmlContent, suNameToSuFile );

                        res.add(saSOAPFile);

                    } catch (Exception e) {
                        e.printStackTrace();
                    }

                    suMap = new HashMap<String, String>();
                    suNameToSuFile = new HashMap<String,File>();
                }
            }
        }
        return res;
    }

}