/**
 * petalsbpm-domain - Domain Objects for BPMN standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.collaboration;

import java.util.ArrayList;
import java.util.List;

import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.collaboration.ILaneBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.collaboration.ILaneSetBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IFlowElementBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.IProcessBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.activity.ICallActivityBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.activity.ISubProcessBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.activity.ITaskBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IDataObjectBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IEndEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IIntermediateCatchEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IIntermediateThrowEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IStartEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.gateway.IGatewayBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.mock.IdGenerator;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.foundation.BaseElementBean;



public class LaneBean extends BaseElementBean implements ILaneBean {

	private String name;
	private ILaneSetBean childLaneSet;
	private List<IStartEventBean> startEvents;
	private List<IIntermediateCatchEventBean> intermediateCatchEvents;
	private List<IIntermediateThrowEventBean> intermediateThrowEvents;
	private List<IEndEventBean> endEvents;
	private List<ITaskBean> tasks;
	private List<IGatewayBean> gateways;
	private List<IDataObjectBean> dataObjects;
	private List<ISubProcessBean> subprocesses;
	private List<ICallActivityBean> callActivities;
	
	//not part of specification
	private ILaneSetBean parentLaneSet;
	private ILaneBean parentLane;
	private IProcessBean process;
	
	public LaneBean(){
		this(IdGenerator.createUniqueId());
	}
	
	public LaneBean(String id){
		super(id);
		startEvents = new ArrayList<IStartEventBean>();
        endEvents = new ArrayList<IEndEventBean>();
        tasks = new ArrayList<ITaskBean>();
        gateways = new ArrayList<IGatewayBean>();
        intermediateCatchEvents = new ArrayList<IIntermediateCatchEventBean>();
        intermediateThrowEvents = new ArrayList<IIntermediateThrowEventBean>();
        dataObjects = new ArrayList<IDataObjectBean>();
        subprocesses = new ArrayList<ISubProcessBean>();
        callActivities = new ArrayList<ICallActivityBean>();
		name = "";
		childLaneSet	= new LaneSetBean();
	}
	
	public void addCallActivity(ICallActivityBean a){
        if (callActivities.contains(a)==false) callActivities.add(a);
        if (process!=null) process.addCallActivity(a);
    }
    
    public void removeCallActivity(ICallActivityBean a){
        callActivities.remove(a);
        if (process!=null) process.removeCallActivity(a);
    }
	
	public void addSubProcess(ISubProcessBean spb){
	    if (subprocesses.contains(spb)==false) subprocesses.add(spb);
	    if (process!=null) process.addSubProcess(spb);
	}

	public void removeSubProcess(ISubProcessBean spb){
	    subprocesses.remove(spb);
	    if (process!=null) process.removeSubProcess(spb);
	}
	
	public void addStartEvent(IStartEventBean s){
		if (startEvents.contains(s)==false) startEvents.add(s);
		if (process!=null) process.addStartEvent(s);
	}
	
	public void addEndEvent(IEndEventBean e){
		if (endEvents.contains(e)==false) endEvents.add(e);
		if (process!=null) process.addEndEvent(e);
	}
	
	public void addTask(ITaskBean t){
		if (tasks.contains(t)==false) tasks.add(t);
		if (process!=null) process.addTask(t);
	}
	
	public void addGateway(IGatewayBean g){
		if (gateways.contains(g)==false) gateways.add(g);
		if (process!=null) process.addGateway(g);
	}
	
	public void addIntermediateCatchEvent(IIntermediateCatchEventBean e){
		if (intermediateCatchEvents.contains(e)==false) intermediateCatchEvents.add(e);
		if (process!=null) process.addIntermediateCatchEvent(e);
	}
	
	public void addIntermediateThrowEvent(IIntermediateThrowEventBean e){
		if (intermediateThrowEvents.contains(e)==false) intermediateThrowEvents.add(e);
		if (process!=null) process.addIntermediateThrowEvent(e);
	}
	
	public void addDataObject(IDataObjectBean d){
		if (dataObjects.contains(d)==false) dataObjects.add(d);
		if (process!=null) process.addDataObject(d);
	}
	
	public void removeStartEvent(IStartEventBean s){
        startEvents.remove(s);
        if (process!=null) process.removeStartEvent(s);
    }

    public void removeEndEvent(IEndEventBean e){
        endEvents.remove(e);
        if (process!=null) process.removeEndEvent(e);
    }

    public void removeTask(ITaskBean t){
        tasks.remove(t);
        if (process!=null) process.removeTask(t);
    }

    public void removeGateway(IGatewayBean g){
        gateways.remove(g);
        if (process!=null) process.removeGateway(g);
    }

    public void removeIntermediateCatchEvent(IIntermediateCatchEventBean e){
        intermediateCatchEvents.remove(e);
        if (process!=null) process.removeIntermediateCatchEvent(e);
    }

    public void removeIntermediateThrowEvent(IIntermediateThrowEventBean e){
        intermediateThrowEvents.remove(e);
        if (process!=null) process.removeIntermediateThrowEvent(e);
    }

    public void removeDataObject(IDataObjectBean d){
        dataObjects.remove(d);
        if (process!=null) process.removeDataObject(d);
    }
	
	public ITaskBean getTaskById(String id){
		return searchListById(tasks,id);
	}
	
	public IGatewayBean getGatewayById(String id){
		return searchListById(gateways,id);
	}
	
	public IStartEventBean getStartEventById(String id){
		return searchListById(startEvents,id);
	}
	
	public IEndEventBean getEndEventById(String id){
		return searchListById(endEvents,id);
	}
	
	public IIntermediateCatchEventBean getIntermediateCatchEventById(String id){
		return searchListById(intermediateCatchEvents,id);
	}
	
	public IIntermediateThrowEventBean getIntermediateThrowEventById(String id){
		return searchListById(intermediateThrowEvents, id);
	}
	
	public IDataObjectBean getDataObjectById(String id){
		return searchListById(dataObjects, id);
	}
	
	private <T extends IFlowElementBean> T searchListById(List<T> l, String id){
		for(T t : l){
			if(t.getId().equals(id)){
				return t;
			}
		}
		return null;
	}
	
	@Override
	public void addFlowElement(IFlowElementBean flowElement) {

		
		if (flowElement instanceof IEndEventBean){
			addEndEvent((IEndEventBean) flowElement);
		}else if (flowElement instanceof ITaskBean){
			addTask((ITaskBean) flowElement);
		}else if (flowElement instanceof IStartEventBean){
			addStartEvent((IStartEventBean) flowElement);
		}else if (flowElement instanceof IGatewayBean){
			addGateway((IGatewayBean) flowElement);
		}else if (flowElement instanceof IIntermediateCatchEventBean){
			addIntermediateCatchEvent((IIntermediateCatchEventBean) flowElement);
		}else if (flowElement instanceof IIntermediateThrowEventBean){
			addIntermediateThrowEvent((IIntermediateThrowEventBean) flowElement);
		}else if (flowElement instanceof IDataObjectBean){
			addDataObject((IDataObjectBean) flowElement);
		}else if (flowElement instanceof ICallActivityBean){
			addCallActivity((ICallActivityBean) flowElement);
		}
		
	}
	
	public List<? extends IFlowElementBean> getFlowNodes() {
		List<IFlowElementBean> nodes = null;
		nodes = new ArrayList<IFlowElementBean>();
		nodes.addAll(getEndEvents());
		nodes.addAll(getStartEvents());
		nodes.addAll(getTasks());
		nodes.addAll(getGateways());
		nodes.addAll(getIntermediateCatchEvents());
		nodes.addAll(getIntermediateThrowEvents());
		nodes.addAll(getDataObjects());
		if(getChildLaneSet()!=null && getChildLaneSet().getFlowNodes().size()>0){
			nodes.addAll(getChildLaneSet().getFlowNodes());
		}
		nodes.addAll(getCallActivities());
		return nodes;
	}
	
	/**
	 * Removes feb which is either a task, an event or a gateway from the lane
	 * @param feb
	 * @return true if feb was in the lane
	 */
	public boolean removeFlowElement(IFlowElementBean feb){
		if(tasks.contains(feb)){
		    return tasks.remove(feb);
		}
		if(gateways.contains(feb)){
		    return gateways.remove(feb);
		}
		if(endEvents.contains(feb)){
		    return endEvents.remove(feb);
		}
		if(startEvents.contains(feb)){
		    return startEvents.remove(feb);
		}
		if(intermediateCatchEvents.contains(feb)){
		    return intermediateCatchEvents.remove(feb);
		}
		if(intermediateThrowEvents.contains(feb)){
		    return intermediateThrowEvents.remove(feb);
		}
		if(dataObjects.contains(feb)){
		    return dataObjects.remove(feb);
		}
		if(callActivities.contains(feb)){
		    return callActivities.remove(feb);
		}
		return false;
	}
	////////////////////////////////////////////
	/////                                  /////
	/////       GETTERS AND SETTERS        /////
	/////                                  /////
	////////////////////////////////////////////
	public String getName() {
		return name;
	}
	public void setName(String name) {
		this.name = name;
	}
    public ILaneSetBean getChildLaneSet() {
        return childLaneSet;
    }
    public void setChildLaneSet(ILaneSetBean childLaneSet) {
        this.childLaneSet = childLaneSet;
    }
    public List<IStartEventBean> getStartEvents() {
        return startEvents;
    }
    public void setStartEvents(List<IStartEventBean> startEvents) {
        this.startEvents = startEvents;
    }
    public List<IIntermediateCatchEventBean> getIntermediateCatchEvents() {
        return intermediateCatchEvents;
    }
    public void setIntermediateCatchEvents(List<IIntermediateCatchEventBean> intermediateCatchEvents) {
        this.intermediateCatchEvents = intermediateCatchEvents;
    }
    public List<IIntermediateThrowEventBean> getIntermediateThrowEvents() {
        return intermediateThrowEvents;
    }
    public void setIntermediateThrowEvents(List<IIntermediateThrowEventBean> intermediateThrowEvents) {
        this.intermediateThrowEvents = intermediateThrowEvents;
    }
    public List<IEndEventBean> getEndEvents() {
        return endEvents;
    }
    public void setEndEvents(List<IEndEventBean> endEvents) {
        this.endEvents = endEvents;
    }
    public List<ITaskBean> getTasks() {
        return tasks;
    }
    public void setTasks(List<ITaskBean> tasks) {
        this.tasks = tasks;
    }
    public List<IGatewayBean> getGateways() {
        return gateways;
    }
    public void setGateways(List<IGatewayBean> gateways) {
        this.gateways = gateways;
    }
    public List<IDataObjectBean> getDataObjects() {
        return dataObjects;
    }
    public void setDataObjects(List<IDataObjectBean> dataObjects) {
        this.dataObjects = dataObjects;
    }
    public List<ISubProcessBean> getSubprocesses() {
        return subprocesses;
    }
    public void setSubprocesses(List<ISubProcessBean> subprocesses) {
        this.subprocesses = subprocesses;
    }
    public ILaneSetBean getParentLaneSet() {
        return parentLaneSet;
    }
    public void setParentLaneSet(ILaneSetBean parentLaneSet) {
        this.parentLaneSet = parentLaneSet;
    }
    public ILaneBean getParentLane() {
        return parentLane;
    }
    public void setParentLane(ILaneBean parentLane) {
        this.parentLane = parentLane;
    }
    public List<ICallActivityBean> getCallActivities() {
        return callActivities;
    }
    public void setCallActivities(List<ICallActivityBean> callActivities) {
        this.callActivities = callActivities;
    }

	@Override
	public IProcessBean getProcess() {
		return process;
	}

	@Override
	public void setProcess(IProcessBean process) {
		this.process = process;
	}

	@Override
	public void clearFlowElements() {
		startEvents.clear();
        endEvents.clear();
        tasks.clear();
        gateways.clear();
        intermediateCatchEvents.clear();
        intermediateThrowEvents.clear();
        dataObjects.clear();
        subprocesses.clear();
        callActivities.clear();
	}
	
}
