/**
 * petalsbpm-domain - Domain Objects for BPMN standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process;

import java.util.ArrayList;
import java.util.List;

import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.Constants.ProcessTypes;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.collaboration.ICollaborationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.collaboration.ILaneBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IFlowElementBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.artifact.IArtifactBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.artifact.IAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.artifact.ITextAnnotationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.infrastructure.IDefinitionsBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.IProcessBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.activity.IActivityBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.activity.ICallActivityBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.activity.ISubProcessBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.activity.ITaskBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IDataInputBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IDataObjectBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IDataOutputBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IIOSpecificationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IInputOutputBinding;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IItemAwareElementBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.ICatchEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IEndEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IIntermediateCatchEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IIntermediateThrowEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IStartEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IThrowEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.gateway.IGatewayBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.gateway.ISequenceFlowBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.service.IInterfaceBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.mock.IdGenerator;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.common.artifact.AssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.common.artifact.TextAnnotationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.foundation.BaseElementBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.data.IOSpecificationBean;



public class ProcessBean extends BaseElementBean implements IProcessBean {

	private String name;
	private ProcessTypes type;
	private boolean isClosed;
	private boolean executable;
	private IIOSpecificationBean ioSpecification;
	private List<IInputOutputBinding> iOBindings;
	private List<ISequenceFlowBean> sequenceFlows;
	private List<IStartEventBean> startEvents;
	private List<IIntermediateCatchEventBean> intermediateCatchEvents;
	private List<IIntermediateThrowEventBean> intermediateThrowEvents;
	private List<IEndEventBean> endEvents;
	private List<ITaskBean> tasks;
	private List<IGatewayBean> gateways;
	private List<IDataObjectBean> dataObjects;
	private List<ISubProcessBean> subprocesses;
	private List<IArtifactBean> artifacts;
	private List<ICallActivityBean> callActivities;
	private ICollaborationBean definitionalCollaboration;

	private List<IInterfaceBean> supportedInterfaces;
	
	//not part of specification
	private List<ILaneBean> lanes;
	private IDefinitionsBean parentDefinitions;
	
	public ProcessBean(){
		this(IdGenerator.createUniqueId());
	}
	
	public ProcessBean(String id){
		super(id);
		isClosed = false;
		type = ProcessTypes.NONE;
        sequenceFlows = new ArrayList<ISequenceFlowBean>();
        startEvents = new ArrayList<IStartEventBean>();
        endEvents = new ArrayList<IEndEventBean>();
        tasks = new ArrayList<ITaskBean>();
        gateways = new ArrayList<IGatewayBean>();
        intermediateCatchEvents = new ArrayList<IIntermediateCatchEventBean>();
        intermediateThrowEvents = new ArrayList<IIntermediateThrowEventBean>();
        dataObjects = new ArrayList<IDataObjectBean>();
        subprocesses = new ArrayList<ISubProcessBean>();
        callActivities = new ArrayList<ICallActivityBean>();
        artifacts = new ArrayList<IArtifactBean>();
        iOBindings = new ArrayList<IInputOutputBinding>();
        supportedInterfaces = new ArrayList<IInterfaceBean>();
        ioSpecification			= new IOSpecificationBean(IdGenerator.createUniqueId());
        lanes = new ArrayList<ILaneBean>();
	}
	
	
	
	public void addLane(ILaneBean lane){
		if (lanes.contains(lane)==false) this.lanes.add(lane);
		lane.setProcess(this);
		this.synchronizeWith(lane);
	}
	
	public void removeLane(ILaneBean lane) {
	    lanes.remove(lane);
	    lane.setProcess(null);
	}
	
	public void addCallActivity(ICallActivityBean a){
        if (callActivities.contains(a)==false) callActivities.add(a);
        a.setProcess(this);
        synchronizeWith(a);
    }
    
    public void removeCallActivity(ICallActivityBean a){
        callActivities.remove(a);
        a.setProcess(null);
    }
	
	public void addArtifact(IArtifactBean a){
        if (artifacts.contains(a)==false) artifacts.add(a);
    }
    
    public void removeArtifact(IArtifactBean a){
        artifacts.remove(a);
    }
	
	public void addSubProcess(ISubProcessBean spb){
	    if (subprocesses.contains(spb)==false) subprocesses.add(spb);
	    spb.setProcess(this);
	    synchronizeWith(spb);
	}
	
	public void removeSubProcess(ISubProcessBean spb){
        subprocesses.remove(spb);
        spb.setProcess(null);
    }
	
	public void addSequenceFlow(ISequenceFlowBean s){
		if (sequenceFlows.contains(s)==false) sequenceFlows.add(s);
		s.setProcess(this);
	}
	
	public void removeSequenceFlow(ISequenceFlowBean sf){
		sequenceFlows.remove(sf);
		sf.setProcess(null);
	}
	
	public void addStartEvent(IStartEventBean s){
		if (startEvents.contains(s)==false) startEvents.add(s);
		s.setProcess(this);
		this.synchronizeWithCatchEvent(s);
	}
	
	public void addEndEvent(IEndEventBean e){
		if (endEvents.contains(e)==false) endEvents.add(e);
		e.setProcess(this);
		this.synchroniWithThrowEvent(e);
	}
	
	public void addTask(ITaskBean t){
		if (tasks.contains(t)==false) tasks.add(t);
		t.setProcess(this);
		synchronizeWith(t);
	}
	
	public void addGateway(IGatewayBean g){
		if (gateways.contains(g)==false) gateways.add(g);
		g.setProcess(this);
	}
	
	public void addIntermediateCatchEvent(IIntermediateCatchEventBean e){
		if (intermediateCatchEvents.contains(e)==false) intermediateCatchEvents.add(e);
		e.setProcess(this);
		this.synchronizeWithCatchEvent(e);
	}
	
	public void addIntermediateThrowEvent(IIntermediateThrowEventBean e){
		if (intermediateThrowEvents.contains(e)==false) intermediateThrowEvents.add(e);
		e.setProcess(this);
		this.synchroniWithThrowEvent(e);
	}
	
	public void addDataObject(IDataObjectBean d){
		if (dataObjects.contains(d)==false) dataObjects.add(d);
		d.setProcess(this);
	}
	
	public void removeStartEvent(IStartEventBean s){
	    if (startEvents.contains(s)==false) startEvents.remove(s);
	    s.setProcess(null);
	}

	public void removeEndEvent(IEndEventBean e){
	    endEvents.remove(e);
	    e.setProcess(null);
	}

	public void removeTask(ITaskBean t){
	    tasks.remove(t);
	    t.setProcess(null);
	}

	public void removeGateway(IGatewayBean g){
	    gateways.remove(g);
	    g.setProcess(null);
	}

	public void removeIntermediateCatchEvent(IIntermediateCatchEventBean e){
	    intermediateCatchEvents.remove(e);
	    e.setProcess(null);
	}

	public void removeIntermediateThrowEvent(IIntermediateThrowEventBean e){
	    intermediateThrowEvents.remove(e);
	    e.setProcess(null);
	}

	public void removeDataObject(IDataObjectBean d){
	    dataObjects.remove(d);
	    d.setProcess(null);
	}
	
	public ITaskBean getTaskById(String id){
		return searchListById(tasks,id);
	}
	
	public IGatewayBean getGatewayById(String id){
		return searchListById(gateways,id);
	}
	
	public IStartEventBean getStartEventById(String id){
		return searchListById(startEvents,id);
	}
	
	public IEndEventBean getEndEventById(String id){
		return searchListById(endEvents,id);
	}
	
	public IIntermediateCatchEventBean getIntermediateCatchEventById(String id){
		return searchListById(intermediateCatchEvents,id);
	}
	
	public IIntermediateThrowEventBean getIntermediateThrowEventById(String id){
		return searchListById(intermediateThrowEvents, id);
	}
	
	public IDataObjectBean getDataObjectById(String id){
		return searchListById(dataObjects, id);
	}
	
	private <T extends IFlowElementBean> T searchListById(List<T> l, String id){
		for(T t : l){
			if(t.getId().equals(id)){
				return t;
			}
		}
		return null;
	}
	
	public List<? extends IFlowElementBean> getFlowNodes() {
		List<IFlowElementBean> nodes = null;
		nodes = new ArrayList<IFlowElementBean>();
		nodes.addAll(getEndEvents());
		nodes.addAll(getStartEvents());
		nodes.addAll(getTasks());
		nodes.addAll(getGateways());
		nodes.addAll(getIntermediateCatchEvents());
		nodes.addAll(getIntermediateThrowEvents());
		nodes.addAll(getDataObjects());
		nodes.addAll(getCallActivities());
		return nodes;
	}
	
	private void synchronizeWith(ILaneBean lane){
		for(IFlowElementBean flowElement:lane.getFlowNodes()){
			addFlowElement(flowElement);
		}
	}
	
	private void synchronizeWith(IActivityBean activity){
		//consider data object / data input /data output
		for(IItemAwareElementBean data:activity.getDataInputs()){
			//if its a dataObject add it to current process
			if (data instanceof IDataObjectBean){
				IDataObjectBean dataObject = (IDataObjectBean) data;
				this.addDataObject(dataObject);
			//if its a dataInput/dataOuput add it to current processes ioSpecification	
			}else if (data instanceof IDataInputBean){
				IDataInputBean dinput = (IDataInputBean) data;
				this.getIoSpecification().addDataInput(dinput);
			}
		}
		
		for(IItemAwareElementBean data:activity.getDataOutputs()){
			//if its a dataObject add it to current process
			if (data instanceof IDataObjectBean){
				IDataObjectBean dataObject = (IDataObjectBean) data;
				this.addDataObject(dataObject);
			//if its a dataInput/dataOuput add it to current processes ioSpecification	
			}else if(data instanceof IDataOutputBean){
				IDataOutputBean doutput = (IDataOutputBean) data;
				this.getIoSpecification().addDataOutput(doutput);
			}			
		}		
	}
	
	private void synchroniWithThrowEvent(IThrowEventBean e) {
		for(IItemAwareElementBean data : e.getIncomingItemAwareElements()){
			//if its a dataObject add it to current process
			if (data instanceof IDataObjectBean){
				this.addDataObject((IDataObjectBean) data);
			//if its a dataInput/dataOuput add it to current processes ioSpecification	
			}else if (data instanceof IDataInputBean){
				this.getIoSpecification().addDataInput((IDataInputBean) data);
			}
		}
	}
	
	private void synchronizeWithCatchEvent(ICatchEventBean e) {
		for(IItemAwareElementBean data : e.getOutgoingItemAwareElements()) {
			//if its a dataObject add it to current process
			if (data instanceof IDataObjectBean) {
				this.addDataObject((IDataObjectBean) data);
			//if its a dataInput/dataOuput add it to current processes ioSpecification	
			}else if(data instanceof IDataOutputBean) {
				this.getIoSpecification().addDataOutput((IDataOutputBean) data);
			}			
		}		
	}
	
	/**
	 * Removes feb which is either a task, an event or a gateway from the lane
	 * @param feb
	 * @return true if feb was in the lane
	 */
	public boolean removeFlowElement(IFlowElementBean feb){
		if(tasks.contains(feb)){
		    return tasks.remove(feb);
		}
		if(gateways.contains(feb)){
			return gateways.remove(feb);
		}
		if(endEvents.contains(feb)){
			return endEvents.remove(feb);
		}
		if(startEvents.contains(feb)){
			return startEvents.remove(feb);
		}
		if(intermediateCatchEvents.contains(feb)){
			return intermediateCatchEvents.remove(feb);
		}
		if(intermediateThrowEvents.contains(feb)){
			return intermediateThrowEvents.remove(feb);
		}
		if(dataObjects.contains(feb)){
			return dataObjects.remove(feb);
		}
		return false;
	}
	
	////////////////////////////////////////////
	/////                                  /////
	/////       GETTERS AND SETTERS        /////
	/////                                  /////
	////////////////////////////////////////////	
	public String getName() {
		return name;
	}
	public void setName(String name) {
		this.name = name;
	}
    @Override
    public List<IAssociationBean> getAssociations() {
        List<IAssociationBean> res = new ArrayList<IAssociationBean>();
        for(IArtifactBean a : artifacts){
            if(a instanceof AssociationBean){
                res.add((AssociationBean)a);
            }
        }
        return res;
    }
    @Override
    public List<ITextAnnotationBean> getTextAnnotations() {
        List<ITextAnnotationBean> res = new ArrayList<ITextAnnotationBean>();
        for(IArtifactBean a : artifacts){
            if(a instanceof TextAnnotationBean){
                res.add((TextAnnotationBean)a);
            }
        }
        return res;
    }
    public ProcessTypes getType() {
        return type;
    }
    public void setType(ProcessTypes type) {
        this.type = type;
    }
    public boolean isClosed() {
        return isClosed;
    }
    public void setIsClosed(boolean isClosed) {
        this.isClosed = isClosed;
    }
    public IIOSpecificationBean getIoSpecification() {
        return ioSpecification;
    }
    public void setIoSpecification(IIOSpecificationBean ioSpecification) {
        this.ioSpecification = ioSpecification;
    }
    public List<ISequenceFlowBean> getSequenceFlows() {
        return sequenceFlows;
    }
    public void setSequenceFlows(List<ISequenceFlowBean> sequenceFlows) {
        this.sequenceFlows = sequenceFlows;
    }
    public List<IStartEventBean> getStartEvents() {
        return startEvents;
    }
    public void setStartEvents(List<IStartEventBean> startEvents) {
        this.startEvents = startEvents;
    }
    public List<IIntermediateCatchEventBean> getIntermediateCatchEvents() {
        return intermediateCatchEvents;
    }
    public void setIntermediateCatchEvents(List<IIntermediateCatchEventBean> intermediateCatchEvents) {
        this.intermediateCatchEvents = intermediateCatchEvents;
    }
    public List<IIntermediateThrowEventBean> getIntermediateThrowEvents() {
        return intermediateThrowEvents;
    }
    public void setIntermediateThrowEvents(List<IIntermediateThrowEventBean> intermediateThrowEvents) {
        this.intermediateThrowEvents = intermediateThrowEvents;
    }
    public List<IEndEventBean> getEndEvents() {
        return endEvents;
    }
    public void setEndEvents(List<IEndEventBean> endEvents) {
        this.endEvents = endEvents;
    }
    public List<ITaskBean> getTasks() {
        return tasks;
    }
    public void setTasks(List<ITaskBean> tasks) {
        this.tasks = tasks;
    }
    public List<IGatewayBean> getGateways() {
        return gateways;
    }
    public void setGateways(List<IGatewayBean> gateways) {
        this.gateways = gateways;
    }
    public List<IDataObjectBean> getDataObjects() {
        return dataObjects;
    }
    public void setDataObjects(List<IDataObjectBean> dataObjects) {
        this.dataObjects = dataObjects;
    }
    public List<ISubProcessBean> getSubprocesses() {
        return subprocesses;
    }
    public void setSubprocesses(List<ISubProcessBean> subprocesses) {
        this.subprocesses = subprocesses;
    }
    public List<IArtifactBean> getArtifacts() {
        return artifacts;
    }
    public void setArtifacts(List<IArtifactBean> artifacts) {
        this.artifacts = artifacts;
    }
    public List<ICallActivityBean> getCallActivities() {
        return callActivities;
    }
    public void setCallActivities(List<ICallActivityBean> callActivities) {
        this.callActivities = callActivities;
    }
    public ICollaborationBean getDefinitionalCollaboration() {
        return definitionalCollaboration;
    }
    public void setDefinitionalCollaboration(ICollaborationBean definitionalCollaboration) {
        this.definitionalCollaboration = definitionalCollaboration;
    }
    @Override
    public IDefinitionsBean getParentDefinitions() {
        return parentDefinitions;
    }
    @Override
    public void setParentDefinitions(IDefinitionsBean defs) {
        this.parentDefinitions = defs;
    }
	@Override
	public void addIOBinding(IInputOutputBinding ioBinding) {
		this.iOBindings.add(ioBinding);
	}
	@Override
	public List<IInputOutputBinding> getIOBindings() {
		return iOBindings;
	}
	@Override
	public void setIOBindings(List<IInputOutputBinding> ioBindings) {
		this.iOBindings = ioBindings;
	}
	@Override
	public boolean isExecutable() {
		return executable;
	}
	@Override
	public void setExecutable(boolean arg0) {
	    this.executable = arg0;
	}
	@Override
	public void addSupportedInterface(IInterfaceBean interfaceBean) {
	    supportedInterfaces.add(interfaceBean);
	}
	@Override
	public List<IInterfaceBean> getSupportedIntefaces() {
		return supportedInterfaces;
	}
	@Override
	public void setSupportedInterfaces(List<IInterfaceBean> interfaces) {
		this.supportedInterfaces = interfaces;
	}

	@Override
	public String toString() {
		return this.getId()+":"+this.getName();
	}

	@Override
	public List<ILaneBean> getLanes() {
		return lanes;
	}
	public void setLanes(List<ILaneBean> lanes) {
	    this.lanes = lanes;
	}

	@Override
	public void addFlowElement(IFlowElementBean flowElement) {

		
		if (flowElement instanceof IEndEventBean){
			addEndEvent((IEndEventBean) flowElement);
		}else if (flowElement instanceof ITaskBean){
			addTask((ITaskBean) flowElement);
		}else if (flowElement instanceof IStartEventBean){
			addStartEvent((IStartEventBean) flowElement);
		}else if (flowElement instanceof IGatewayBean){
			addGateway((IGatewayBean) flowElement);
		}else if (flowElement instanceof IIntermediateCatchEventBean){
			addIntermediateCatchEvent((IIntermediateCatchEventBean) flowElement);
		}else if (flowElement instanceof IIntermediateThrowEventBean){
			addIntermediateThrowEvent((IIntermediateThrowEventBean) flowElement);
		}else if (flowElement instanceof IDataObjectBean){
			addDataObject((IDataObjectBean) flowElement);
		}else if (flowElement instanceof ICallActivityBean){
			addCallActivity((ICallActivityBean) flowElement);
		}
		
	}

	@Override
	public void clearFlowElements() {
		
		endEvents.clear();
		startEvents.clear();
		tasks.clear();
		gateways.clear();
		intermediateCatchEvents.clear();
		intermediateThrowEvents.clear();
		dataObjects.clear();
		callActivities.clear();
		subprocesses.clear();
		sequenceFlows.clear();
		
		//also clean every lanes
		for(ILaneBean lane:getLanes()){
			
			lane.getEndEvents().clear();
			lane.getStartEvents().clear();
			lane.getTasks().clear();
			lane.getGateways().clear();
			lane.getIntermediateCatchEvents().clear();
			lane.getIntermediateThrowEvents().clear();
			lane.getDataObjects().clear();
			lane.getCallActivities().clear();
			lane.getSubprocesses().clear();
		
		}
	
	}

	@Override
	public void clearIOSpec() {
		ioSpecification.getDataInputs().clear();
		ioSpecification.getDataOutputs().clear();
	}
	
}
