/**
 * petalsbpm-service - BPMN Editor service - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.petalsbpm.server.service.bpmn2.clientToServer;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Stack;

import javax.xml.namespace.QName;

import com.ebmwebsourcing.easybox.api.XmlContext;
import com.ebmwebsourcing.easybox.api.XmlContextFactory;
import com.ebmwebsourcing.easybox.api.XmlObject;
import com.ebmwebsourcing.easybox.api.XmlObjectNode;
import com.ebmwebsourcing.easybox.api.XmlObjectText;
import com.ebmwebsourcing.easybox.api.with.WithOtherAttributes;
import com.ebmwebsourcing.easybpmn.bpmn20.api.BPMNException;
import com.ebmwebsourcing.easybpmn.bpmn20.api.DefinitionsHelper;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Assignment;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Association;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.BaseElement;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.BusinessRuleTask;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.CallActivity;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Choreography;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ChoreographyTask;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Collaboration;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ConditionalEventDefinition;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.DataAssociation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.DataInput;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.DataInputAssociation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.DataObject;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.DataOutput;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.DataOutputAssociation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Definitions;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Documentation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.EndEvent;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.EndPoint;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Error;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Escalation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.EventBasedGateway;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.EventDefinition;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ExclusiveGateway;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Expression;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ExtensionElements;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.FlowNode;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Import;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.InputOutputBinding;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.InputOutputSpecification;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.InputSet;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Interface;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.IntermediateCatchEvent;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.IntermediateThrowEvent;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ItemDefinition;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Lane;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.LaneSet;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ManualTask;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Message;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.MessageEventDefinition;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.MessageFlow;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Operation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.OutputSet;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ParallelGateway;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Participant;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.PartnerRole;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Process;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ReceiveTask;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Script;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ScriptTask;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.SendTask;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.SequenceFlow;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ServiceTask;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Signal;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.StartEvent;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.SubChoreography;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.SubProcess;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Task;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.TerminateEventDefinition;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Text;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.TextAnnotation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.TimerEventDefinition;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.UserTask;
import com.ebmwebsourcing.easybpmn.bpmn20.api.type.TAssociationDirection;
import com.ebmwebsourcing.easybpmn.bpmn20.api.type.TChoreographyLoopType;
import com.ebmwebsourcing.easybpmn.bpmn20.api.type.TEventBasedGatewayType;
import com.ebmwebsourcing.easybpmn.bpmn20.api.type.TItemKind;
import com.ebmwebsourcing.easybpmn.bpmn20.api.type.TProcessType;
import com.ebmwebsourcing.easybpmn.bpmn20.api.with.ItemAwareElement;
import com.ebmwebsourcing.easybpmn.bpmn20.api.with.WithArtifact;
import com.ebmwebsourcing.easybpmn.bpmn20.api.with.WithDataInput;
import com.ebmwebsourcing.easybpmn.bpmn20.api.with.WithDataInputAssociation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.with.WithDataOutput;
import com.ebmwebsourcing.easybpmn.bpmn20.api.with.WithDataOutputAssociation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.with.WithDefaultSequenceFlow;
import com.ebmwebsourcing.easybpmn.bpmn20.api.with.WithFlowElements;
import com.ebmwebsourcing.easybpmn.bpmn20.api.with.WithIOSpecification;
import com.ebmwebsourcing.easybpmn.bpmn20.api.with.WithLaneSet;
import com.ebmwebsourcing.easycommons.lang.UncheckedException;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.extension.client.AttributeExtension;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.extension.client.ObjectExtension;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.extension.client.WithAttributeExtension;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.Constants;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.Constants.AssociationDirection;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.Constants.ChoreographyLoopType;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.Constants.EventGatewayType;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.Constants.ItemKind;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.Constants.ProcessTypes;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.choreography.IChoreographyBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.choreography.IChoreographyTaskBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.choreography.ISubChoreographyBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.collaboration.ICollaborationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.collaboration.IInteractionNodeBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.collaboration.ILaneBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.collaboration.ILaneSetBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IErrorBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IEscalationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IFlowElementBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IFlowNodeBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IItemDefinitionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IMessageBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IMessageFlowBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IParticipantBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IPartnerRoleBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.artifact.IAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.artifact.ITextAnnotationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.foundation.IBaseElementBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.infrastructure.IDefinitionsBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.infrastructure.IImportBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.infrastructure.INamespaceDeclaration;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.IProcessBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.activity.ICallActivityBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.activity.ISubProcessBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.activity.ITaskBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IAssignmentBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IDataAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IDataInputBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IDataObjectBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IDataOutputBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IExpressionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IIOSpecificationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IInputOutputBinding;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IItemAwareElementBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.WithDataInputAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.WithDataInputBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.WithDataOutputAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.WithDataOutputBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IEndEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IIntermediateCatchEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IIntermediateThrowEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.ISignalBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IStartEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.definition.IEventDefinitionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.gateway.IGatewayBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.gateway.ISequenceFlowBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.service.IEndPointBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.service.IInterfaceBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.service.IOperationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.DefinitionsBeanVisitor;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.infrastructure.DefinitionsBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.activity.BusinessRuleTaskBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.activity.ManualTaskBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.activity.ReceiveTaskBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.activity.ScriptTaskBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.activity.SendTaskBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.activity.ServiceTaskBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.activity.TaskBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.activity.UserTaskBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.event.definition.ConditionalEventDefinitionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.event.definition.MessageEventDefinitionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.event.definition.TerminateEventDefinitionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.event.definition.TimerEventDefinitionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.gateway.EventBasedGatewayBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.gateway.ExclusiveGatewayBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.gateway.ParallelGatewayBean;
import com.ebmwebsourcing.petalsbpm.server.service.extension.ExtensionBindingManager;

public class BeanModelAdapter extends DefinitionsBeanVisitor{
	
	private XmlContext context;
	private ExtensionBindingManager bindingManager;
	
	private Definitions defs;
	
	private Interface currentInterface;
	private Operation currentOperation;
	
	private Stack<WithFlowElements> currentFlowElementContainer;
	private Stack<WithLaneSet> currentLaneSetContainer;
	private Stack<WithArtifact> currentArtifactContainer;
	private Stack<LaneSet> currentLaneSet;
	
	private Process currentProcess;
	private Lane currentLane;
	private Collaboration currentCollaboration;
	private MessageFlow currentMessageFlow;
	private Participant currentParticipant;
	private PartnerRole currentPartnerRole;
	private WithDataInput currentWithDataInput;
	private InputSet currentInputSet;
	private WithDataOutput currentWithDataOutput;
	private OutputSet currentOutputSet;
	private WithDataInputAssociation currentDataInputAssociationOwner;
	private WithDataOutputAssociation currentDataOutputAssociationOwner;
	private Stack<WithIOSpecification> currentIOSpecOwner;
	
	private DefinitionsBean defsBean;
	private Map<String, WithDefaultSequenceFlow> defaultFlows;
	private static Map<DataAssociation, IItemAwareElementBean> dataAssociationTargets;
    private static Map<DataAssociation, List<IItemAwareElementBean>> dataAssociationSources;
    private static Map<IItemAwareElementBean, ItemAwareElement> itemAwareElements;
	
	
	public BeanModelAdapter(DefinitionsBean defs) {
		 super(defs);
		 defsBean = defs;
		 bindingManager = new ExtensionBindingManager();
		 currentFlowElementContainer = new Stack<WithFlowElements>();
		 currentLaneSetContainer = new Stack<WithLaneSet>();
		 currentArtifactContainer = new Stack<WithArtifact>();
		 currentLaneSet = new Stack<LaneSet>();
		 currentIOSpecOwner = new Stack<WithIOSpecification>();
	}
	
	public Definitions getDefinitions(){
		return this.defs;
	}
	
	
	@Override
	public void visitDefinitionsByPools() {
		defaultFlows = new HashMap<String, WithDefaultSequenceFlow>();
		super.visitDefinitionsByPools();
	}
	
	
	@Override	
	public void visitDefinitions(IDefinitionsBean defsBean) {
		defs = newInstance(Definitions.class);
		
		defs.setId(defsBean.getId());
		defs.setTargetNamespace(defsBean.getTargetNamespace());
		defs.setTypeLanguage(defsBean.getTypeLanguage());
		defs.setExpressionLanguage(defsBean.getExpressionLanguage());
		defs.setExporter(Constants.EXPORTER);
		
		adaptAttributeExtensions(defs, defsBean);
	}
	
	@Override
	public void visitImport(IImportBean importBean) {
		Import i = newInstance(Import.class);
		i.setImportType(importBean.getImportType());
		i.setLocation(importBean.getLocation());
		i.setNamespace(importBean.getNamespace());
		defs.addImport(i);
	}
	
	@Override
	public void visitSignal(ISignalBean sb) {
	    Signal s = newInstance(Signal.class);
	    s.setId(sb.getId());
	    s.setName(sb.getName());
	    setDocumentationAndExtensions(s, sb);
	    if(sb.getStructureRef()!=null){
	        s.setStructureRef(getBaseElementRef(sb.getStructureRef()));
	    }
	    defs.addRootElement(s);
	}
	
	@Override
	public void visitEscalation(IEscalationBean eb) {
        Escalation e = newInstance(Escalation.class);
        e.setId(eb.getId());
        e.setName(eb.getName());
        e.setEscalationCode(eb.getEscalationCode());
        setDocumentationAndExtensions(e, eb);
        if(eb.getStructureRef()!=null){
            e.setStructureRef(getBaseElementRef(eb.getStructureRef()));
        }
        defs.addRootElement(e);
    }
	
	@Override
    public void visitError(IErrorBean eb) {
	    Error e = newInstance(Error.class);
        e.setId(eb.getId());
        e.setName(eb.getName());
        e.setErrorCode(eb.getErrorCode());
        setDocumentationAndExtensions(e, eb);
        if(eb.getStructureRef()!=null){
            e.setStructureRef(getBaseElementRef(eb.getStructureRef()));
        }
        defs.addRootElement(e);
	}
	
	@Override
	public void visitEndPoint(IEndPointBean epb) {
	    EndPoint ep = newInstance(EndPoint.class);
	    ep.setId(epb.getId());
	    setDocumentationAndExtensions(ep, epb);
	    defs.addRootElement(ep);
	}
	
	@Override
	public void visitItemDefinition(IItemDefinitionBean idb) {
		ItemDefinition id = newInstance(ItemDefinition.class);
		id.setIsCollection(idb.getCollection());
		setDocumentationAndExtensions(id, idb);
		id.setId(idb.getId());
		id.setItemKind(retrieveItemKind(idb.getItemKind()));
		if(idb.getStructureRef()!=null){
		    id.setStructureRef(getQNameFromString(idb.getStructureRef()));
		}
		
		defs.addRootElement(id);
	}
	
	@Override
	public void visitMessage(IMessageBean messageBean) {
		Message m = newInstance(Message.class);
		setDocumentationAndExtensions(m, messageBean);
		m.setId(messageBean.getId());
		m.setName(messageBean.getName());
		if(messageBean.getItemDefinition()!=null){
		    m.setItemRef(getBaseElementRef(messageBean.getItemDefinition()));
		}
		
		defs.addRootElement(m);
	}
	
	@Override
	public void visitRootEventDefinition(IEventDefinitionBean edb) {
	    defs.addRootElement(buildEventDefinition(edb));
	}
	
	@Override
	public void visitPartnerRole(IPartnerRoleBean prBean) {
		PartnerRole pr = newInstance(PartnerRole.class);
		pr.setId(prBean.getId());
		setDocumentationAndExtensions(pr, prBean);
		pr.setName(prBean.getName());
		defs.addRootElement(pr);
		currentPartnerRole = pr;
	}
	
	@Override
	public void visitPartnerRoleParticipant(IParticipantBean p, IPartnerRoleBean pr) {
		currentPartnerRole.addParticipantRef(getBaseElementRef(p));
	}
	
	@Override
	public void visitInterface(IInterfaceBean itfb) {
		Interface itf = newInstance(Interface.class);
		itf.setId(itfb.getId());
		setDocumentationAndExtensions(itf, itfb);
		itf.setName(itfb.getName());
		if(itfb.getWsdlImplementationRef()!=null){
		    itf.setImplementationRef(getQNameFromString(itfb.getWsdlImplementationRef()));
        }
		defs.addRootElement(itf);
		currentInterface = itf;
	}
	
	
	@Override
	public void visitOperation(IOperationBean opBean) {
		Operation op = buildOperation(opBean);
		currentInterface.addOperation(op);

		currentOperation = op;
	}
	
	
	private Operation buildOperation(IOperationBean opBean){
		Operation op = newInstance(Operation.class);
		setDocumentationAndExtensions(op, opBean);
		op.setName(opBean.getName());
		op.setId(opBean.getId());
		
		if(opBean.getWsdlImplementationRef()!=null && !opBean.getWsdlImplementationRef().isEmpty()) {
		    op.setImplementationRef(getQNameFromString(opBean.getWsdlImplementationRef()));
		}
        
		return op;
	}
	
	
	@Override
	public void visitOperationMessageIn(IMessageBean msg, IOperationBean opBean) {
		currentOperation.setInMessageRef(getBaseElementRef(msg));
	}
	
	@Override
	public void visitOperationMessageOut(IMessageBean msg, IOperationBean opBean) {
		currentOperation.setOutMessageRef(getBaseElementRef(msg));
	}
	
	@Override
	public void visitOperationError(IErrorBean msg, IOperationBean opBean) {
		currentOperation.addErrorRef(getBaseElementRef(msg));
	}
	
	@Override
    public void visitChoreography(IChoreographyBean chor){
	    Choreography c = newInstance(Choreography.class);
	    setDocumentationAndExtensions(c, chor);
	    c.setId(chor.getId());
	    c.setName(chor.getName());
	    c.setIsClosed(chor.isClosed());
	    
	    defs.addRootElement(c);
	    
	    currentFlowElementContainer.push(c);
	    currentArtifactContainer.push(c);
	    currentCollaboration = c;
	}
	
	@Override
    public void endChoreographyVisit(IChoreographyBean chor){
	    currentFlowElementContainer.pop();
	    currentArtifactContainer.pop();
	}
	
	@Override
    public void visitSubChoreography(ISubChoreographyBean subChor){
	    SubChoreography sc = newInstance(SubChoreography.class);
	    sc.setId(subChor.getId());
	    setDocumentationAndExtensions(sc, subChor);
	    sc.setLoopType(retrieveChoreographyLoopType(subChor.getChoreographyLoopType()));
	    sc.setInitiatingParticipantRef(new QName(subChor.getInitiatingParticipant().getId()));
	    sc.setName(subChor.getName());
	    if(subChor.getParticipants()!=null){
            for(IParticipantBean p : subChor.getParticipants()){
                sc.addParticipantRef(new QName(p.getId()));
            }
        }
	    
	    currentFlowElementContainer.peek().addFlowElement(sc);
	    currentFlowElementContainer.push(sc);
	    currentArtifactContainer.push(sc);
	}
    
    @Override
    public void endSubChoreographyVisit(ISubChoreographyBean chor){
        currentFlowElementContainer.pop();
        currentArtifactContainer.pop();
    }
	
	@Override
    public void visitChoreographyTask(IChoreographyTaskBean cTask){
	    ChoreographyTask ct = newInstance(ChoreographyTask.class);
	    ct.setId(cTask.getId());
	    ct.setName(cTask.getName());
	    setDocumentationAndExtensions(ct, cTask);
	    ct.setLoopType(retrieveChoreographyLoopType(cTask.getChoreographyLoopType()));
	    ct.setInitiatingParticipantRef(new QName(cTask.getInitiatingParticipant().getId()));
	    if(cTask.getParticipants()!=null){
	        for(IParticipantBean p : cTask.getParticipants()){
	            ct.addParticipantRef(new QName(p.getId()));
	        }
	    }
	    if(cTask.getInitiatingMessageFlow()!=null){
	        ct.addMessageFlowRef(new QName(cTask.getInitiatingMessageFlow().getId()));
	    }
	    if(cTask.getReturnMessageFlow()!=null){
            ct.addMessageFlowRef(new QName(cTask.getReturnMessageFlow().getId()));
        }
	    
	    currentFlowElementContainer.peek().addFlowElement(ct);
	}
	
	@Override
	public void visitProcess(IProcessBean processBean) {
		Process process = newInstance(Process.class);
		setDocumentationAndExtensions(process,processBean);
		process.setId(processBean.getId());
		process.setName(processBean.getName());
		process.setProcessType(retrieveProcessType(processBean.getType()));
		process.setIsClosed(processBean.isClosed());
		process.setIsExecutable(processBean.isExecutable());
		
		if(processBean.getDefinitionalCollaboration()!=null){
		    process.setDefinitionalCollaborationRef(getBaseElementRef(processBean.getDefinitionalCollaboration()));
		}
		if(processBean.getSupportedIntefaces()!=null) {
		    for(IInterfaceBean itfb : processBean.getSupportedIntefaces()) {
		        process.addSupportedInterfaceRef(getBaseElementRef(itfb));
		    }
		}
		
		defs.addRootElement(process);
		
		LaneSet defaultLS = newInstance(LaneSet.class);
		process.addLaneSet(defaultLS);
		currentLaneSet.push(defaultLS);
		
		currentProcess = process;
		currentFlowElementContainer.push(process);
		currentLaneSetContainer.push(process);
		currentArtifactContainer.push(process);
		currentIOSpecOwner.push(process);
		itemAwareElements = new HashMap<IItemAwareElementBean, ItemAwareElement>();
		dataAssociationSources = new HashMap<DataAssociation, List<IItemAwareElementBean>>();
		dataAssociationTargets = new HashMap<DataAssociation, IItemAwareElementBean>();
	}
	
	@Override
	public void endProcessVisit(IProcessBean p){
	    currentFlowElementContainer.pop();
	    currentLaneSetContainer.pop();
	    currentArtifactContainer.pop();
	    currentLaneSet.pop();
	    
	    for(DataAssociation da : dataAssociationSources.keySet()){
	        for(IItemAwareElementBean iae : dataAssociationSources.get(da)){
	            da.addSourceRef(itemAwareElements.get(iae));
	        }
	    }
	    for(DataAssociation da : dataAssociationTargets.keySet()){
	        da.setTargetRef(itemAwareElements.get(dataAssociationTargets.get(da)));
	    }
	}
	
	@Override
	public void visitSubprocessLaneSet(ILaneSetBean laneSetBean) {
		LaneSet laneSet = newInstance(LaneSet.class);
		currentLaneSetContainer.peek().addLaneSet(laneSet);
		laneSet.setId(laneSetBean.getId());
		setDocumentationAndExtensions(laneSet, laneSetBean);
		
		currentLaneSet.push(laneSet);
	}
	
	@Override
    public void endLaneSetVisit(ILaneSetBean laneSet) {
        currentLaneSet.pop();
    }
	
	@Override
	public void visitLane(ILaneBean laneBean) {
		Lane lane = newInstance(Lane.class);
		currentLaneSet.peek().addLane(lane);
		setDocumentationAndExtensions(lane,laneBean);
		lane.setName(laneBean.getName());
		lane.setId(laneBean.getId());
		
		for(IFlowElementBean fe : laneBean.getFlowNodes()){
			if(fe instanceof IFlowNodeBean) {
				lane.addFlowNodeRef((FlowNode) currentFlowElementContainer.peek().getFlowElementById(fe.getId()));
			}
		}
		
		currentLane = lane;
	}
	
	@Override
	public void visitChildLaneSet(ILaneSetBean lsb) {
		LaneSet laneSet = newInstance(LaneSet.class);
		currentLane.setChildLaneSet(laneSet);
		laneSet.setId(lsb.getId());
		setDocumentationAndExtensions(laneSet, lsb);
		
		currentLaneSet.push(laneSet);
	}
	
	
	@Override
	public void visitSubProcess(ISubProcessBean spBean){
	    SubProcess sp = newInstance(SubProcess.class);
	    sp.setId(spBean.getId());
	    sp.setName(sp.getName());
	    setDocumentationAndExtensions(sp, spBean);
	    
	    currentFlowElementContainer.peek().addFlowElement(sp);
	    currentFlowElementContainer.push(sp);
	    currentLaneSetContainer.push(sp);
	    currentArtifactContainer.push(sp);
	    currentIOSpecOwner.push(sp);
	}
	
	@Override
	public void endSubProcessVisit(ISubProcessBean proc) {
	    currentFlowElementContainer.pop();
	    currentLaneSetContainer.pop();
	    currentArtifactContainer.pop();
	}
	
	@Override
	public void visitStartEvent(IStartEventBean seb) {
		StartEvent startEvent = newInstance(StartEvent.class);
		setDocumentationAndExtensions(startEvent, seb);
		startEvent.setId(seb.getId());
		startEvent.setName(seb.getName());
		if(seb.getTriggers()!=null){
			for(IEventDefinitionBean edb : seb.getTriggers()){
			    startEvent.addEventDefinitionRef(adaptEventDefinitionRef(edb));
			}
		}
		currentFlowElementContainer.peek().addFlowElement(startEvent);
		currentDataOutputAssociationOwner = startEvent;
		currentWithDataOutput = startEvent;
		if(seb.getDataOutputs()!=null && !seb.getDataOutputs().isEmpty()){
		    OutputSet os = newInstance(OutputSet.class);
		    startEvent.setOutputSet(os);
		    currentOutputSet = os;
		}
	}
	
	@Override
	public void visitIntermediateThrowEvent(IIntermediateThrowEventBean e) {
		IntermediateThrowEvent ie = newInstance(IntermediateThrowEvent.class);
		setDocumentationAndExtensions(ie, e);
		ie.setId(e.getId());
		ie.setName(e.getName());
		if(e.getResults()!=null){
			for(IEventDefinitionBean edb : e.getResults()){
				ie.addEventDefinitionRef(adaptEventDefinitionRef(edb));
			}
		}
		currentFlowElementContainer.peek().addFlowElement(ie);
		currentDataInputAssociationOwner = ie;
		currentWithDataInput = ie;
		if(e.getDataInputs()!=null && !e.getDataInputs().isEmpty()){
		    InputSet is  = newInstance(InputSet.class);
		    ie.setInputSet(is);
		    currentInputSet = is;
		}
	}
	
	@Override
	public void visitIntermediateCatchEvent(IIntermediateCatchEventBean e) {
		IntermediateCatchEvent ie = newInstance(IntermediateCatchEvent.class);
		setDocumentationAndExtensions(ie, e);
		ie.setId(e.getId());
		ie.setName(e.getName());
		if(e.getTriggers()!=null){
			for(IEventDefinitionBean edb : e.getTriggers()){
				ie.addEventDefinitionRef(adaptEventDefinitionRef(edb));
			}
		}
		currentFlowElementContainer.peek().addFlowElement(ie);
		currentDataOutputAssociationOwner = ie;
		currentWithDataOutput = ie;
		if(e.getDataOutputs()!=null && !e.getDataOutputs().isEmpty()){
            OutputSet os = newInstance(OutputSet.class);
            ie.setOutputSet(os);
            currentOutputSet = os;
        }
	}
	
	@Override
	public void visitTask(ITaskBean taskBean) {
	    if(taskBean instanceof ReceiveTaskBean){
	        adaptReceiveTask((ReceiveTaskBean) taskBean);
	    }
	    else if(taskBean instanceof SendTaskBean){
	        adaptSendTask((SendTaskBean) taskBean);
	    }
	    else if(taskBean instanceof ServiceTaskBean){
	        adaptServiceTask((ServiceTaskBean) taskBean);
	    }
	    else if(taskBean instanceof ManualTaskBean){
	        adaptManualTask((ManualTaskBean) taskBean);
	    }
	    else if(taskBean instanceof UserTaskBean) {
	    	adaptUserTask((UserTaskBean)taskBean);
	    }
	    else if(taskBean instanceof ScriptTaskBean) {
	    	adaptScriptTask((ScriptTaskBean) taskBean);
	    }
	    else if(taskBean instanceof BusinessRuleTaskBean) {
	    	adaptBusinessRuleTask((BusinessRuleTaskBean) taskBean);
	    }
	    else {
	        adaptSimpleTask((TaskBean)taskBean, newInstance(Task.class));
	    }
	}
	
	@Override
	public void visitCallActivity(ICallActivityBean cab){
	   CallActivity ca = newInstance(CallActivity.class);
	   ca.setId(cab.getId());
	   setDocumentationAndExtensions(ca, cab);
	   ca.setName(cab.getName());
	   if(cab.getCalledElement()!=null){
	       ca.setCalledElement(getBaseElementRef(cab.getCalledElement()));
	   }
	   
	   currentFlowElementContainer.peek().addFlowElement(ca);
	   currentDataInputAssociationOwner = ca;
	   currentDataOutputAssociationOwner = ca;
	   currentIOSpecOwner.push(ca);
	}
	
	@Override
	public void visitDataObject(IDataObjectBean dobj) {
		DataObject result = newInstance(DataObject.class);
		setDocumentationAndExtensions(result, dobj);
		result.setId(dobj.getId());
		result.setIsCollection(dobj.isCollection());
		result.setName(dobj.getName());
		if(dobj.getItemSubject()!=null){
			result.setItemSubjectRef(getBaseElementRef(dobj.getItemSubject()));
		}
		
		currentFlowElementContainer.peek().addFlowElement(result);
		itemAwareElements.put(dobj, result);
	}
	
	@Override
	public void visitIOSpecification(IIOSpecificationBean ios) {
		InputOutputSpecification result = null;
		
		boolean hasDataInput = ios.getDataInputs()!=null && !ios.getDataInputs().isEmpty();
		boolean hasDataOutput = ios.getDataOutputs()!=null && !ios.getDataOutputs().isEmpty();
		if(hasDataInput || hasDataOutput) {
			result = newInstance(InputOutputSpecification.class);
			result.setId(ios.getId());
			setDocumentationAndExtensions(result, ios);
			InputSet is = newInstance(InputSet.class);
			result.addInputSet(is);
			OutputSet os = newInstance(OutputSet.class);
			result.addOutputSet(os);

			currentWithDataInput = result;
			currentInputSet = is;
			currentWithDataOutput = result;
			currentOutputSet = os;
		}
		
		currentIOSpecOwner.pop().setIoSpecification(result);
	}
	
	@Override
	public void visitIOBinding(IInputOutputBinding iobinding) {
	    InputOutputBinding iob = newInstance(InputOutputBinding.class);
	    iob.setId(iobinding.getId());
	    if(iobinding.getOperation()!=null) {
	        iob.setOperationRef(getBaseElementRef(iobinding.getOperation()));
	    }
	    if(iobinding.getInputData()!=null) {
	        try {
                iob.setInputDataRef(DefinitionsHelper.findObjectInDefinitions(defs, iobinding.getId(), DataInput.class));
            } catch (BPMNException e) {
                e.printStackTrace();
                throw new UncheckedException(e);
            }
	    }
	    if(iobinding.getOutputData()!=null) {
	        try {
                iob.setOutputDataRef(DefinitionsHelper.findObjectInDefinitions(defs, iobinding.getId(), DataOutput.class));
            } catch (BPMNException e) {
                e.printStackTrace();
                throw new UncheckedException(e);
            }
	    }
	    
	    currentProcess.addIoBinding(iob);
	}
	
	@Override
	public void visitDataInput(IDataInputBean dib, WithDataInputBean owner) {
		DataInput di = newInstance(DataInput.class);
		di.setId(dib.getId());
		di.setIsCollection(dib.isCollection());
		di.setName(dib.getName());
		setDocumentationAndExtensions(di, dib);
		if(dib.getItemSubject()!=null){
		    di.setItemSubjectRef(getBaseElementRef(dib.getItemSubject()));
		}
		
		currentWithDataInput.addDataInput(di);
		currentInputSet.addDataInputRef(di);
		itemAwareElements.put(dib, di);
	}
	
	@Override
	public void visitDataOutput(IDataOutputBean dob, WithDataOutputBean owner) {
	    DataOutput do_ = newInstance(DataOutput.class);
        do_.setId(dob.getId());
        do_.setIsCollection(dob.isCollection());
        do_.setName(dob.getName());
        setDocumentationAndExtensions(do_, dob);
        if(dob.getItemSubject()!=null){
            do_.setItemSubjectRef(getBaseElementRef(dob.getItemSubject()));
        }
        
        currentWithDataOutput.addDataOutput(do_);
        currentOutputSet.addDataOutputRef(do_);
        itemAwareElements.put(dob, do_);
	}
	
	@Override
	public void visitDataInputAssociation(IDataAssociationBean diaBean, WithDataInputAssociationBean owner){
	    DataInputAssociation dia = newInstance(DataInputAssociation.class);
        setDocumentationAndExtensions(dia, diaBean);
        dia.setId(diaBean.getId());
        if(diaBean.getTarget()!=null){
            dataAssociationTargets.put(dia, diaBean.getTarget());
        }
        if(diaBean.getSources()!=null){
            dataAssociationSources.put(dia, diaBean.getSources());
        }
        for(IAssignmentBean assignementBean : diaBean.getAssignements()){
            Assignment assignement = newInstance(Assignment.class);
            setDocumentationAndExtensions(assignement, assignementBean);
            assignement.setId(assignementBean.getId());
            buildExpression(assignement, assignementBean);
            dia.addAssignment(assignement);
        }
        if(diaBean.getTransformation()!=null){
            dia.setTransformation(buildExpression(diaBean.getTransformation().getContent(), diaBean.getTransformation().getId()));
        }
        
        currentDataInputAssociationOwner.addDataInputAssociation(dia);
	}
	
	@Override
	public void visitDataOutputAssociation(IDataAssociationBean doaBean, WithDataOutputAssociationBean owner){
	    DataOutputAssociation doa = newInstance(DataOutputAssociation.class);
        setDocumentationAndExtensions(doa, doaBean);
        doa.setId(doaBean.getId());
        if(doaBean.getTarget()!=null){
            dataAssociationTargets.put(doa, doaBean.getTarget());
        }
        if(doaBean.getSources()!=null){
            dataAssociationSources.put(doa, doaBean.getSources());
        }
        for(IAssignmentBean ab : doaBean.getAssignements()){
            Assignment a = newInstance(Assignment.class);
            setDocumentationAndExtensions(a, ab);
            a.setId(ab.getId());
            buildExpression(a, ab);
            doa.addAssignment(a);
        }
        if(doaBean.getTransformation()!=null){
            doa.setTransformation(buildExpression(doaBean.getTransformation().getContent(), doaBean.getTransformation().getId()));
        }
        currentDataOutputAssociationOwner.addDataOutputAssociation(doa);
	}
	
	@Override
	public void visitEndEvent(IEndEventBean endEventBean) {
		EndEvent endEvent = newInstance(EndEvent.class);
		setDocumentationAndExtensions(endEvent, endEventBean);
		endEvent.setName(endEventBean.getName());
		endEvent.setId(endEventBean.getId());
		if(endEventBean.getResults()!=null){
			for(IEventDefinitionBean edb : endEventBean.getResults()){
				endEvent.addEventDefinitionRef(adaptEventDefinitionRef(edb));
			}
		}
		currentFlowElementContainer.peek().addFlowElement(endEvent);
		currentDataInputAssociationOwner = endEvent;
		currentWithDataInput = endEvent;
		if(endEventBean.getDataInputs()!=null && !endEventBean.getDataInputs().isEmpty()) {
            InputSet is  = newInstance(InputSet.class);
            endEvent.setInputSet(is);
            currentInputSet = is;
        }
	}
	
	@Override
	public void visitGateway(IGatewayBean gb) {
		if(gb instanceof ExclusiveGatewayBean){
			adaptExclusiveGateway((ExclusiveGatewayBean) gb);
		}
		else if(gb instanceof ParallelGatewayBean){
			adaptParallelGateway((ParallelGatewayBean) gb);
		}
		else if(gb instanceof EventBasedGatewayBean){
		    adaptEventBasedGateway((EventBasedGatewayBean) gb);
		}
		else{
		    throw new IllegalArgumentException(gb.getClass()+" are not supported yet as gateways.");
		}
	}
	
    @Override
	public void visitSequenceFlow(ISequenceFlowBean sequenceFlowBean) {
		String id = sequenceFlowBean.getId();  
		SequenceFlow sequenceFlow = newInstance(SequenceFlow.class);
		setDocumentationAndExtensions(sequenceFlow, sequenceFlowBean);
		sequenceFlow.setId(id);
		sequenceFlow.setName(sequenceFlowBean.getName());
		
		if(sequenceFlowBean.getExpression()!=null
		        && sequenceFlowBean.getExpression().getContent()!=null
		        && !sequenceFlowBean.getExpression().getContent().isEmpty()){
			Expression exp = newInstance(Expression.class);
			setDocumentation(exp, sequenceFlowBean.getExpression());
			exp.setId(sequenceFlowBean.getExpression().getId());
			exp.setTextContent(sequenceFlowBean.getExpression().getContent());
			sequenceFlow.setConditionExpression(exp);
		}
		
		if(sequenceFlowBean.getSourceNode()!=null) {
			FlowNode source = findFlowElementById(currentFlowElementContainer.peek(),sequenceFlowBean.getSourceNode().getId());
			sequenceFlow.setSourceRef(source);
		}
		if(sequenceFlowBean.getTargetNode()!=null) {
			FlowNode target = findFlowElementById(currentFlowElementContainer.peek(), sequenceFlowBean.getTargetNode().getId());
			sequenceFlow.setTargetRef(target);
		}
		
		currentFlowElementContainer.peek().addFlowElement(sequenceFlow);
		
		if(defaultFlows.containsKey(id)){
            defaultFlows.get(id).setDefault(sequenceFlow);
            defaultFlows.remove(id);
        }
	}
	
	@Override
	public void visitCollaboration(ICollaborationBean cBean) {
		Collaboration c = newInstance(Collaboration.class);
		c.setName(cBean.getName());
		c.setId(cBean.getId());
		c.setIsClosed(cBean.isClosed());
		setDocumentationAndExtensions(c, cBean);
		defs.addRootElement(c);
		
		currentCollaboration = c;
		currentArtifactContainer.push(c);
	}
	
	@Override
	public void visitMessageFlow(IMessageFlowBean msgFlowBean) {
		MessageFlow msgFlow = newInstance(MessageFlow.class);
		msgFlow.setId(msgFlowBean.getId());
		msgFlow.setName(msgFlowBean.getName());
		setDocumentationAndExtensions(msgFlow, msgFlowBean);
		currentCollaboration.addMessageFlow(msgFlow);
		currentMessageFlow = msgFlow;
	}
	
	@Override
	public void visitMessageFlowMessage(IMessageBean mb) {
		currentMessageFlow.setMessageRef(getBaseElementRef(mb)); 
	}
	
	@Override
	public void visitMessageFlowSource(IInteractionNodeBean node) {
		currentMessageFlow.setSourceRef(new QName(node.getId()));
	}
	
	@Override
	public void visitMessageFlowTarget(IInteractionNodeBean node) {
		currentMessageFlow.setTargetRef(new QName(node.getId()));
	}
	
	@Override
	public void visitParticipant(IParticipantBean participantBean) {
		Participant participant = newInstance(Participant.class);
		participant.setId(participantBean.getId());
		participant.setName(participantBean.getName());
		if(participantBean.getProcess()!=null){
			participant.setProcessRef(getBaseElementRef(participantBean.getProcess()));
		}
		
		setDocumentationAndExtensions(participant, participantBean);
		currentCollaboration.addParticipant(participant);
		
		currentParticipant = participant;
	}
	
	@Override
    public void visitParticipantEndPoint(IEndPointBean epb, IParticipantBean pb) {
        currentParticipant.addEndpointRef(getBaseElementRef(epb));
    }
	
	@Override
	public void visitParticipantInterface(IInterfaceBean interface1) {
		currentParticipant.addInterfaceRef(getBaseElementRef(interface1));
	}
	
	@Override
	public void visitTextAnnotation(ITextAnnotationBean tab) {
	   TextAnnotation ta = newInstance(TextAnnotation.class);
	   ta.setId(tab.getId());
	   ta.setTextFormat(tab.getTextFormat());
	   setDocumentationAndExtensions(ta, tab);
	   if(tab.getText()!=null && !tab.getText().isEmpty()){
	       Text t = newInstance(Text.class);
	       XmlObjectText content = newInstance(XmlObjectText.class);
           content.setText(tab.getText());
	       t.addText(content);
	       ta.setText(t);
	   }
	   
	   currentArtifactContainer.peek().addArtifact(ta);
	}
	
	@Override
    public void visitAssociation(IAssociationBean ab) {
	    Association a = newInstance(Association.class);
	    a.setId(ab.getId());
	    setDocumentationAndExtensions(a, ab);
	    a.setAssociationDirection(retrieveAssociationDirection(ab.getDirection()));
	    if(ab.getSource()!=null){
	        a.setSourceRef(new QName(ab.getSource().getId()));
	    }
	    if(ab.getTarget()!=null){
            a.setTargetRef(new QName(ab.getTarget().getId()));
        }
	    
	    currentArtifactContainer.peek().addArtifact(a);
	}
	
	
    private static FlowNode findFlowElementById(WithFlowElements wfe, String id) {
		return (FlowNode) wfe.getFlowElementById(id);
	}

	
	private void adaptParallelGateway(ParallelGatewayBean gatewayBean) {
		ParallelGateway pg = newInstance(ParallelGateway.class);
		setDocumentationAndExtensions(pg, gatewayBean);
		pg.setId(gatewayBean.getId());
		pg.setName(gatewayBean.getName());
		currentFlowElementContainer.peek().addFlowElement(pg);
	}

	private void adaptExclusiveGateway(ExclusiveGatewayBean gatewayBean) {
		ExclusiveGateway eg = newInstance(ExclusiveGateway.class);
		setDocumentationAndExtensions(eg, gatewayBean);
		eg.setId(gatewayBean.getId());
		eg.setName(gatewayBean.getName());
		if(gatewayBean.getDefaultSequenceFlow()!=null){
			defaultFlows.put(gatewayBean.getDefaultSequenceFlow().getId(), eg);
		}
		currentFlowElementContainer.peek().addFlowElement(eg);
	}
	
	private void adaptEventBasedGateway(EventBasedGatewayBean gb) {
	    EventBasedGateway eg = newInstance(EventBasedGateway.class);
	    setDocumentationAndExtensions(eg, gb);
	    eg.setId(gb.getId());
	    eg.setName(gb.getName());
	    eg.setInstantiate(gb.isInstantiate());
	    eg.setEventGatewayType(retrieveEventGatewayType(gb.getType()));
	    currentFlowElementContainer.peek().addFlowElement(eg);
    }
	
    private void adaptSimpleTask(TaskBean taskBean, Task t) {
		setDocumentationAndExtensions(t, taskBean);
		t.setId(taskBean.getId());
		t.setName(taskBean.getName());
		if(taskBean.getDefaultSequenceFlow()!=null) {
            defaultFlows.put(taskBean.getDefaultSequenceFlow().getId(), t);
        }
		currentFlowElementContainer.peek().addFlowElement(t);
		currentDataInputAssociationOwner = t;
		currentDataOutputAssociationOwner = t;
		currentIOSpecOwner.push(t);
	}
    
    private void adaptManualTask(ManualTaskBean taskBean) {
        ManualTask t = newInstance(ManualTask.class);
        adaptSimpleTask(taskBean, t);
    }
    
    private void adaptUserTask(UserTaskBean taskBean) {
        UserTask t = newInstance(UserTask.class);
        adaptSimpleTask(taskBean, t);
    }
    
    private void adaptScriptTask(ScriptTaskBean taskBean) {
        ScriptTask t = newInstance(ScriptTask.class);
        adaptSimpleTask(taskBean, t);
        t.setScriptFormat(taskBean.getScriptFormat());
        if(taskBean.getScript()!=null && !taskBean.getScript().isEmpty()) {
        	Script script = newInstance(Script.class);
        	XmlObjectText content = newInstance(XmlObjectText.class);
            content.setText(taskBean.getScript());
        	script.addText(content);
        	t.setScript(script);
        }
    }
    
    private void adaptBusinessRuleTask(BusinessRuleTaskBean taskBean) {
        BusinessRuleTask t = newInstance(BusinessRuleTask.class);
        adaptSimpleTask(taskBean, t);
    }

	private void adaptServiceTask(ServiceTaskBean taskBean) {
		ServiceTask st = newInstance(ServiceTask.class);
		adaptSimpleTask(taskBean, st);
		if(taskBean.getOperation()!=null){
			st.setOperationRef(getBaseElementRef(taskBean.getOperation()));
		}
	}

	private void adaptSendTask(SendTaskBean taskBean) {
		SendTask st = newInstance(SendTask.class);
		adaptSimpleTask(taskBean, st);
		if(taskBean.getOperation()!=null){
			st.setOperationRef(getBaseElementRef(taskBean.getOperation()));
		}
		if(taskBean.getMessage()!=null){
			st.setMessageRef(getBaseElementRef(taskBean.getMessage()));
		}
	}

	private void adaptReceiveTask(ReceiveTaskBean taskBean) {
		ReceiveTask rt = newInstance(ReceiveTask.class);
		adaptSimpleTask(taskBean, rt);
		if(taskBean.getOperation()!=null){
			rt.setOperationRef(getBaseElementRef(taskBean.getOperation()));
		}
		if(taskBean.getMessage()!=null){
			rt.setMessageRef(getBaseElementRef(taskBean.getMessage()));
		}
	}
	
	
	private void buildExpression(Assignment a, IAssignmentBean ab){
		Expression expFrom = newInstance(Expression.class);
		setDocumentation(expFrom, ab.getFrom());
		expFrom.setId(ab.getFrom().getId());
		expFrom.setTextContent(ab.getFrom().getContent());
		Expression expTo = newInstance(Expression.class);
		setDocumentation(expTo, ab.getTo());
		expTo.setId(ab.getTo().getId());
		expTo.setTextContent(ab.getTo().getContent());
		
		a.setFrom(expFrom);
		a.setTo(expTo);
	}
	
	//A duplicate of the method for base elements because Expression are BaseElementWithMixedContent
	private void setDocumentation(Expression element, IExpressionBean bean){
		if(bean!=null && bean.getDocumentation()!=null && !bean.getDocumentation().trim().isEmpty()){
			Documentation doc = newInstance(Documentation.class);
			XmlObjectText content = newInstance(XmlObjectText.class);
			content.setText(bean.getDocumentation());
			doc.addDocumentationContent(content);
			element.addDocumentation(doc);
		}
		
		adaptAttributeExtensions(element, bean);
		
		if(!bean.getObjectExtensions().isEmpty()){
			ExtensionElements ee = newInstance(ExtensionElements.class);
			for(ObjectExtension obj : bean.getObjectExtensions()){
				XmlObject ext = bindingManager.clientToServer(obj);
				if(ext!=null) ee.addAnyXmlObject(ext);
			}
			element.setExtensionElements(ee);
		}
	}
	
	private void setDocumentationAndExtensions(BaseElement element, IBaseElementBean bean) {
		if(bean!=null && bean.getDocumentation()!=null && !bean.getDocumentation().trim().isEmpty()){
		    Documentation doc = newInstance(Documentation.class);
            XmlObjectText content = newInstance(XmlObjectText.class);
            content.setText(bean.getDocumentation());
            doc.addDocumentationContent(content);
            element.addDocumentation(doc);
		}
		
		adaptAttributeExtensions(element, bean);
		
		if(!bean.getObjectExtensions().isEmpty()){
			ExtensionElements ee = newInstance(ExtensionElements.class);
			for(ObjectExtension obj : bean.getObjectExtensions()){
				XmlObject ext = bindingManager.clientToServer(obj);
				if(ext!=null) {
					ee.addAnyXmlObject(ext);
					element.setExtensionElements(ee);
				}
			}
		}
	}
	
	private QName getQNameFromString(String s) {
	    //case 1 : {NSURI}localPart
	    try{
	        QName qn = QName.valueOf(s);
	        if(qn.getNamespaceURI()!=null && !qn.getNamespaceURI().isEmpty()) {
	            INamespaceDeclaration nsd = defsBean.getNsDeclarationByNamespace(qn.getNamespaceURI());
	            if(nsd!=null) {
	                return new QName(qn.getNamespaceURI(), qn.getLocalPart(), nsd.getPrefix());
	            }
	            
	            return qn;
	        }
	    }
	    catch(IllegalArgumentException iae) {}
	    
	    //case 2 : nsPrefix:localPart
        if (s.contains(":")) {
            String prefix = s.substring(0, s.indexOf(':'));
            String localpart = s.substring(s.indexOf(':') + 1);
            INamespaceDeclaration nsd = defsBean.getNsDeclarationByPrefix(prefix);
            if(nsd!=null) {
                return new QName(nsd.getURI(), localpart, nsd.getPrefix());
            }
        }
        
        //case 3 : localPart 
        return new QName(s);
	}
	
	private void adaptAttributeExtensions(WithOtherAttributes element, WithAttributeExtension bean){
		if (bean.getAttributeExtensions()!=null){
			for(AttributeExtension att : bean.getAttributeExtensions()){
				element.addOtherAttribute(new QName(att.getAttributeQNameNS(), att.getAttributeQNameLocalPart()), att.getAttributeValue());
			}
		}
	}
	
	private TEventBasedGatewayType retrieveEventGatewayType(EventGatewayType type) {
	    TEventBasedGatewayType result = null;
        switch(type){
            case Exclusive:
                result = TEventBasedGatewayType.Exclusive;
                break;
            case Parallel:
                result = TEventBasedGatewayType.Parallel;
                break;
        }
        return result;
    }
	
	private TItemKind retrieveItemKind(ItemKind itemKind) {
		TItemKind result = null;
		if(itemKind!=null){
			switch (itemKind) {
			case INFORMATION:
				result = TItemKind.Information;
				break;
			case PHYSICAL:
				result = TItemKind.Physical;
				break;
			default:
				break;
			}
		}
		return result;
	}
	
	private TProcessType retrieveProcessType(ProcessTypes type) {
		TProcessType result = null;
		if(type!=null){
			switch (type) {
			case PRIVATE:
				result = TProcessType.Private;
				break;
			case NONE:
				result = TProcessType.None;
				break;
			case PUBLIC:
				result = TProcessType.Public;
				break;
			default:
				break;
			}
		}
		return result;
	}
	
	private TChoreographyLoopType retrieveChoreographyLoopType(ChoreographyLoopType type){
	    TChoreographyLoopType result = null;
	    if(type!=null){
	        switch (type) {
                case MultiInstanceParallel:
                    result = TChoreographyLoopType.MultiInstanceParallel;
                    break;
                case MultiInstanceSequential:
                    result = TChoreographyLoopType.MultiInstanceSequential;
                    break;
                case None:
                    result = TChoreographyLoopType.None;
                    break;
                case Standard:
                    result = TChoreographyLoopType.Standard;
                    break;
            }
	    }
	    return result;
	}
	
	private TAssociationDirection retrieveAssociationDirection(AssociationDirection dir) {
	    TAssociationDirection result = null;
	    if(dir!=null){
	        switch(dir){
	            case Both:
	                result = TAssociationDirection.Both;
	                break;
	            case None:
	                result = TAssociationDirection.None;
	                break;
	            case One:
	                result = TAssociationDirection.One;
	                break;
	        }
	    }
        return result;
    }
	
	private QName adaptEventDefinitionRef(IEventDefinitionBean edb) {
	    if(defs.getEventDefinitionById(edb.getId())==null) {
	        defs.addRootElement(buildEventDefinition(edb));
	    }
        return new QName(edb.getId());
	}
	
	private EventDefinition buildEventDefinition(IEventDefinitionBean edb){
		EventDefinition result = null;
		if(edb instanceof MessageEventDefinitionBean){
			result = newInstance(MessageEventDefinition.class);
			if(((MessageEventDefinitionBean)edb).getMessage()!=null){
				((MessageEventDefinition)result).setMessageRef(getBaseElementRef(((MessageEventDefinitionBean)edb).getMessage()));
			}
			if(((MessageEventDefinitionBean)edb).getOperation()!=null) {
			    ((MessageEventDefinition)result).setOperationRef(getBaseElementRef(((MessageEventDefinitionBean)edb).getOperation()));
			}
		}
		else if(edb instanceof ConditionalEventDefinitionBean){
			result = newInstance(ConditionalEventDefinition.class);
			String cond = ((ConditionalEventDefinitionBean)edb).getCondition();
			Expression exp = buildExpression(cond, edb.getId()+"_condition");
			((ConditionalEventDefinition)result).setCondition(exp);
		}
		else if(edb instanceof TerminateEventDefinitionBean){
			result = newInstance(TerminateEventDefinition.class);
		}
		else if(edb instanceof TimerEventDefinitionBean){
			TimerEventDefinitionBean ted = (TimerEventDefinitionBean) edb;
			result = newInstance(TimerEventDefinition.class);
			((TimerEventDefinition)result).setTimeCycle(buildExpression(ted.getTimeCycle(), edb.getId()+"_cycle"));
			((TimerEventDefinition)result).setTimeDate(buildExpression(ted.getTimeDate(), edb.getId()+"_date"));
			((TimerEventDefinition)result).setTimeDuration(buildExpression(ted.getTimeDuration(), edb.getId()+"_duration"));
		}
		else{
		    throw new IllegalArgumentException(edb.getClass()+" are not supported yet as event defintions.");
		}
		
		if(result!=null){
			result.setId(edb.getId());
			setDocumentationAndExtensions(result, edb);
		}
		
		return result;
	}
	
	
	private Expression buildExpression(String content, String id){
		if(content!=null){
			Expression exp = newInstance(Expression.class);
			exp.setId(id);
			exp.setTextContent(content);
			return exp;
		}
		return null;
	}
	
	
	private QName getBaseElementRef(IBaseElementBean bean) {
	    if(bean.getNSDeclaration()!=null) {
	        return new QName(bean.getNSDeclaration().getURI(), bean.getId(), bean.getNSDeclaration().getPrefix());
	    }
	    return new QName(bean.getId()); 
	}
	
	private <T extends XmlObjectNode> T newInstance(Class<? extends T> c){
		if(context==null){
			context = new XmlContextFactory().newContext();
		}
		return context.getXmlObjectFactory().create(c);
	}
	
}
