/**
 * petalsbpm-service - BPMN Editor service - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.petalsbpm.server.service.bpmn2.serverToClient;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import javax.xml.namespace.QName;

import com.ebmwebsourcing.easybox.api.XmlObject;
import com.ebmwebsourcing.easybox.api.with.WithOtherAttributes;
import com.ebmwebsourcing.easybpmn.bpmn20.api.BPMNException;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Activity;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Artifact;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Assignment;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Association;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.BaseElement;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.CallActivity;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Choreography;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ChoreographyTask;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Collaboration;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ConditionalEventDefinition;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.DataInput;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.DataInputAssociation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.DataObject;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.DataOutput;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.DataOutputAssociation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Definitions;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Documentation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.EndEvent;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.EndPoint;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Error;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Escalation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.EventBasedGateway;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.EventDefinition;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ExclusiveGateway;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Expression;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ExtensionElements;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.FlowNode;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Gateway;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Import;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.InputOutputBinding;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.InputOutputSpecification;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Interface;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.IntermediateCatchEvent;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.IntermediateThrowEvent;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ItemDefinition;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Lane;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.LaneSet;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ManualTask;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Message;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.MessageEventDefinition;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.MessageFlow;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Operation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ParallelGateway;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Participant;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.PartnerRole;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Process;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ReceiveTask;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.SendTask;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.SequenceFlow;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ServiceTask;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Signal;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.StartEvent;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.SubChoreography;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.SubProcess;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Task;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.TerminateEventDefinition;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.TextAnnotation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.TimerEventDefinition;
import com.ebmwebsourcing.easybpmn.bpmn20.api.type.TAssociationDirection;
import com.ebmwebsourcing.easybpmn.bpmn20.api.type.TChoreographyLoopType;
import com.ebmwebsourcing.easybpmn.bpmn20.api.type.TEventBasedGatewayType;
import com.ebmwebsourcing.easybpmn.bpmn20.api.type.TItemKind;
import com.ebmwebsourcing.easybpmn.bpmn20.api.type.TProcessType;
import com.ebmwebsourcing.easybpmn.bpmn20.api.with.ItemAwareElement;
import com.ebmwebsourcing.easybpmn.bpmn20.api.with.WithArtifact;
import com.ebmwebsourcing.easybpmn.bpmn20.api.with.WithDataInput;
import com.ebmwebsourcing.easybpmn.bpmn20.api.with.WithDataInputAssociation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.with.WithDataOutput;
import com.ebmwebsourcing.easybpmn.bpmn20.api.with.WithDataOutputAssociation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.with.WithFlowElements;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.extension.client.AttributeExtension;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.extension.client.ObjectExtension;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.extension.client.WithAttributeExtension;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.WithDefaultSequenceFlow;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.WithFlowElementsBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.WithGatewaysBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.WithSequenceFlowsBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.Constants.AssociationDirection;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.Constants.ChoreographyLoopType;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.Constants.EventGatewayType;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.Constants.ItemKind;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.Constants.ProcessTypes;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.choreography.WithChoreographyFlowElementsBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.collaboration.IInteractionNodeBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.ICallableElementBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IErrorBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IParticipantBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.artifact.WithArtifactsBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.foundation.IBaseElementBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.infrastructure.INamespaceDeclaration;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.IProcessBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IDataInputBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IDataOutputBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IItemAwareElementBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.WithDataInputAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.WithDataInputBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.WithDataOutputAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.WithDataOutputBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.service.IEndPointBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.service.IInterfaceBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.mock.IdGenerator;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.DefinitionsHelper;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.choreography.ChoreographyBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.choreography.ChoreographyTaskBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.choreography.SubChoreographyBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.collaboration.CollaborationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.collaboration.LaneBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.collaboration.LaneSetBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.common.ErrorBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.common.EscalationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.common.FlowElementBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.common.ItemDefinitionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.common.MessageBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.common.MessageFlowBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.common.ParticipantBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.common.PartnerRoleBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.common.SignalBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.common.artifact.AssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.common.artifact.TextAnnotationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.foundation.BaseElementBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.infrastructure.DefinitionsBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.infrastructure.ImportBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.infrastructure.NamespaceDeclaration;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.ProcessBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.activity.ActivityBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.activity.CallActivityBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.activity.ManualTaskBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.activity.ReceiveTaskBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.activity.SendTaskBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.activity.ServiceTaskBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.activity.SubProcessBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.activity.TaskBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.data.AssignmentBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.data.DataAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.data.DataInputAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.data.DataInputBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.data.DataObjectBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.data.DataOutputAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.data.DataOutputBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.data.ExpressionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.data.IOSpecificationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.data.InputOutputBindingBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.event.EndEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.event.IntermediateCatchEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.event.IntermediateThrowEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.event.StartEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.event.definition.ConditionalEventDefinitionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.event.definition.EventDefinitionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.event.definition.MessageEventDefinitionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.event.definition.TerminateEventDefinitionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.event.definition.TimerEventDefinitionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.gateway.EventBasedGatewayBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.gateway.ExclusiveGatewayBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.gateway.GatewayBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.gateway.ParallelGatewayBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.gateway.SequenceFlowBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.service.EndPointBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.service.InterfaceBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.service.OperationBean;
import com.ebmwebsourcing.petalsbpm.server.service.extension.ExtensionBindingManager;


public class ServerToClient {
	
	private static ExtensionBindingManager bindingManager;
	private static Map<String, WithDefaultSequenceFlow> defaultFlows;
	private static Map<DataAssociationBean, ItemAwareElement> dataAssociationTargets;
	private static Map<DataAssociationBean, ItemAwareElement[]> dataAssociationSources;
	private static Map<ItemAwareElement, IItemAwareElementBean> itemAwareElements;
	private static Map<CallActivityBean, QName> callActivityProcesses;
	private static INamespaceDeclaration nsDeclaration;
	
	public static DefinitionsBean adapt(Definitions def, INamespaceDeclaration nsd) throws BPMNException{
	    nsDeclaration = nsd;
	    DefinitionsBean result = adapt(def);
	    nsDeclaration = null;
	    return result;
	}
	
	public static DefinitionsBean adapt(Definitions def) throws BPMNException{
		bindingManager = new ExtensionBindingManager();
		
		DefinitionsBean result = new DefinitionsBean(def.getId());
		
		result.setTargetNamespace(def.getTargetNamespace());
		result.setExpressionLanguage(def.getExpressionLanguage());
		result.setTypeLanguage(def.getTypeLanguage());
		
		adaptImports(result, def.getImports(), def);
		
		adaptItemDefinitions(result, def.getItemDefinitions());
		
		adaptSignals(result, def.getSignals());
		
		adaptErrors(result, def.getErrors());
		
		adaptEscalations(result, def.getEscalations());
		
		adaptEndPoints(result, def.getEndPoints());
		
		adaptMessages(result, def.getMessages());
		
		adaptInterfaces(result, def.getInterfaces());
		
		adaptProcesses(def.getProcesses(), result);
		
		adaptCollaborations(result,def.getCollaborations());
		
		adaptChoreographies(result,def.getChoreographies());
		
		adaptPartnerRoles(result, def.getPartnerRoles());
		
		adaptAttributeExtensions(result, def);
		
		return result;
	}

    private static void adaptSignals(DefinitionsBean defsBean, Signal[] signals) {
        if(signals!=null){
            for(Signal s : signals){
                SignalBean sb = new SignalBean(s.getId());
                setDocumentationAndExtensions(sb, s);
                sb.setName(s.getName());
                if(s.getStructureRef()!=null){
                    sb.setStructureRef(defsBean.getItemDefinitionById(s.getStructureRef().getLocalPart()));
                }
                defsBean.addSignal(sb);
            }
        }
    }
    
    private static void adaptErrors(DefinitionsBean defsBean, Error[] Errors) {
        if(Errors!=null){
            for(Error s : Errors){
                ErrorBean sb = new ErrorBean(s.getId());
                setDocumentationAndExtensions(sb, s);
                sb.setName(s.getName());
                sb.setErrorCode(s.getErrorCode());
                if(s.getStructureRef()!=null){
                    sb.setStructureRef(defsBean.getItemDefinitionById(s.getStructureRef().getLocalPart()));
                }
                defsBean.addError(sb);
            }
        }
    }
    
    private static void adaptEscalations(DefinitionsBean defsBean, Escalation[] Escalations) {
        if(Escalations!=null){
            for(Escalation s : Escalations){
                EscalationBean sb = new EscalationBean(s.getId());
                setDocumentationAndExtensions(sb, s);
                sb.setName(s.getName());
                sb.setEscalationCode(s.getEscalationCode());
                if(s.getStructureRef()!=null){
                    sb.setStructureRef(defsBean.getItemDefinitionById(s.getStructureRef().getLocalPart()));
                }
                defsBean.addEscalation(sb);
            }
        }
    }
    
    private static void adaptEndPoints(DefinitionsBean defsBean, EndPoint[] EndPoints) {
        if(EndPoints!=null){
            for(EndPoint ep : EndPoints){
                EndPointBean epb = new EndPointBean(ep.getId());
                setDocumentationAndExtensions(epb, ep);
                if(ep.hasEndPointRef()) {
                    epb.setWsdlImplementationRef(getStringFromQname(ep.getEndPointRef(), defsBean));
                }
                defsBean.addEndPoint(epb);
            }
        }
    }

    private static void adaptPartnerRoles(DefinitionsBean defs, PartnerRole[] partnerRoles) {
		if(partnerRoles!=null){
			for(PartnerRole pr : partnerRoles){
				PartnerRoleBean prb = new PartnerRoleBean(pr.getId());
				prb.setName(pr.getName());
				setDocumentationAndExtensions(prb, pr);
				prb.setParticipants(findParticipants(defs, pr.getParticipantRef()));
			}
		}
	}
	
	private static List<IParticipantBean> findParticipants(DefinitionsBean defs, QName[] ps){
		List<IParticipantBean> result = new ArrayList<IParticipantBean>();
		if(ps!=null){
			for(QName p : ps){
			    result.add(findObjectInDefinitions(defs, p, ParticipantBean.class));
			}
		}
		return result;
	}

	private static void adaptInterfaces(DefinitionsBean result, Interface[] interfaces) {
		if(interfaces!=null){
			for(Interface itf : interfaces){
				InterfaceBean itfBean = new InterfaceBean(itf.getId());
				setDocumentationAndExtensions(itfBean, itf);
				itfBean.setName(itf.getName());
				
				if(itf.getOperations()!=null){
					for(Operation op : itf.getOperations()){
						itfBean.addOperation(adaptOperation(result, op));
					}
				}
				if(itf.getImplementationRef()!=null){
				    QName implemRef = itf.getImplementationRef();
				    assert implemRef.getPrefix()!=null && !implemRef.getPrefix().isEmpty(); 
				    itfBean.setWsdlImplementationRef(implemRef.getPrefix()+":"+implemRef.getLocalPart());
				}
				
				result.addInterface(itfBean);
			}
		}
	}
	
	private static void adaptChoreographies(DefinitionsBean def, Choreography[] choreographies) {
        if(choreographies!=null){
            for(Choreography chor : choreographies){
                ChoreographyBean cBean = new ChoreographyBean(chor.getId());
                def.addChoreography(cBean);
                cBean.setParentDefinitions(def);
                cBean.setName(chor.getName());
                cBean.setClosed(chor.isIsClosed());
                setDocumentationAndExtensions(cBean, chor);
                
                Map<String,ParticipantBean> participants = new HashMap<String, ParticipantBean>();
                if(chor.getParticipant()!=null){
                    for(Participant p : chor.getParticipant()){
                        ParticipantBean pBean = adaptParticipant(p,cBean,def);
                        participants.put(pBean.getId(), pBean);
                    }
                }
                
                Map<String,MessageFlowBean> messageFlows = new HashMap<String, MessageFlowBean>();
                if(chor.getMessageFlow()!=null){
                    for(MessageFlow mf : chor.getMessageFlow()){
                        MessageFlowBean mfb = adaptMessageFlow(mf,cBean,def);
                        messageFlows.put(mfb.getId(), mfb);
                    }
                }
                
                Map<String, FlowElementBean> elements = new HashMap<String, FlowElementBean>();
                if(chor.getFlowElement()!=null){
                    fillChoreography(cBean,chor,elements,participants,messageFlows,def);
                }

                if(chor.getFlowElementsByClass(SequenceFlow.class)!=null){
                    for(SequenceFlow sf : chor.getFlowElementsByClass(SequenceFlow.class)){
                        adaptSequenceFlow(sf, cBean, elements, def);
                    }
                }
                
                Map<String, IBaseElementBean> baseElts = new HashMap<String, IBaseElementBean>();
                baseElts.putAll(participants);
                baseElts.putAll(messageFlows);
                baseElts.putAll(elements);
                adaptArtifacts(chor, cBean, baseElts);
            }
        }
    }
	
	private static void fillChoreography(WithChoreographyFlowElementsBean pBean, WithFlowElements wfe,
            Map<String, FlowElementBean> elements, Map<String,ParticipantBean> participants,
            Map<String, MessageFlowBean> messageFlows, DefinitionsBean defs) {
        
        for(FlowNode fn : wfe.getFlowNode()){
            //START EVENTS
            if(fn instanceof StartEvent){
                StartEvent se = (StartEvent) fn;
                StartEventBean seBean = new StartEventBean(se.getId());
                seBean.setName(se.getName());
                setDocumentationAndExtensions(seBean, se);
                if(se.getEventDefinition()!=null){
                    for(EventDefinition ed : se.getEventDefinition()){
                        EventDefinitionBean edb  = buildEventDefinition(ed,defs);
                        setDocumentationAndExtensions(edb, ed);
                        seBean.addTrigger(edb);
                    }
                }
                pBean.addStartEvent(seBean);
                elements.put(seBean.getId(), seBean);
            }
            //IntermediateCatch EVENTS
            else if(fn instanceof IntermediateCatchEvent){
                IntermediateCatchEvent se = (IntermediateCatchEvent) fn;
                IntermediateCatchEventBean seBean = new IntermediateCatchEventBean(se.getId());
                seBean.setName(se.getName());
                setDocumentationAndExtensions(seBean, se);
                if(se.getEventDefinition()!=null){
                    for(EventDefinition ed : se.getEventDefinition()){
                        EventDefinitionBean edb  = buildEventDefinition(ed,defs);
                        setDocumentationAndExtensions(edb, ed);
                        seBean.addTrigger(edb);
                    }
                }
                pBean.addIntermediateCatchEvent(seBean);
                elements.put(seBean.getId(), seBean);
            }
            //GATEWAYS
            else if(fn instanceof ParallelGateway){
                adaptParallelGateway(pBean,(ParallelGateway) fn,elements);
            }
            else if(fn instanceof ExclusiveGateway){
                adaptExclusiveGateway(pBean, (ExclusiveGateway) fn, elements);
            }
            else if(fn instanceof EventBasedGateway){
                adaptEventBasedGateway(pBean, (EventBasedGateway)fn, elements);
            }
            //IntermediateThrow EVENTS
            else if(fn instanceof IntermediateThrowEvent){
                IntermediateThrowEvent ee = (IntermediateThrowEvent) fn;
                IntermediateThrowEventBean eeb = new IntermediateThrowEventBean(ee.getId());
                eeb.setName(ee.getName());
                setDocumentationAndExtensions(eeb, ee);
                if(ee.getEventDefinition()!=null){
                    for(EventDefinition ed : ee.getEventDefinition()){
                        EventDefinitionBean edb = buildEventDefinition(ed,defs);
                        setDocumentationAndExtensions(edb, ed);
                        eeb.addResult(edb);
                    }
                }
                pBean.addIntermediateThrowEvent(eeb);
                elements.put(eeb.getId(), eeb);
            }
            //END EVENTS
            else if(fn instanceof EndEvent){
                EndEvent ee = (EndEvent) fn;
                EndEventBean eeb = new EndEventBean(ee.getId());
                eeb.setName(ee.getName());
                setDocumentationAndExtensions(eeb, ee);
                if(ee.getEventDefinition()!=null){
                    for(EventDefinition ed : ee.getEventDefinition()){
                        EventDefinitionBean edb = buildEventDefinition(ed,defs);
                        setDocumentationAndExtensions(edb, ed);
                        eeb.addResult(edb);
                    }
                }
                pBean.addEndEvent(eeb);
                elements.put(eeb.getId(), eeb);
            }
            //CHOREOGRAPHY TASKS
            else if(fn instanceof ChoreographyTask){
                ChoreographyTask ct = (ChoreographyTask) fn;
                ChoreographyTaskBean ctb = new ChoreographyTaskBean(ct.getId());
                setDocumentationAndExtensions(ctb, ct);
                if(ct.hasInitiatingParticipantRef()){
                    ctb.setInitiatingParticipant(participants.get(ct.getInitiatingParticipantRef().getLocalPart()));
                }
                if(ct.hasParticipantRef()){
                    for(QName qn : ct.getParticipantRef()){
                        ctb.addParticipantBean(participants.get(qn.getLocalPart()));
                    }
                }
                ctb.setChoreographyLoopType(retrieveChoreographyLoopType(ct.getLoopType()));
                if(ct.hasMessageFlowRef()){
                    ctb.setInitiatingMessageFlow(messageFlows.get(ct.getMessageFlowRef()[0].getLocalPart()));
                    if(ct.getMessageFlowRef().length==2){
                        ctb.setReturnMessageFlow(messageFlows.get(ct.getMessageFlowRef()[1].getLocalPart()));
                    }
                }
                pBean.addChoreographyTask(ctb);
                elements.put(ctb.getId(), ctb);
            }
            //SUB CHOREOGRAPHIES
            else if(fn instanceof SubChoreography){
                SubChoreography sc = (SubChoreography) fn;
                SubChoreographyBean scb = new SubChoreographyBean(sc.getId());
                setDocumentationAndExtensions(scb, sc);
                if(sc.hasInitiatingParticipantRef()){
                    scb.setInitiatingParticipant(participants.get(sc.getInitiatingParticipantRef().getLocalPart()));
                }
                if(sc.hasParticipantRef()){
                    for(QName qn : sc.getParticipantRef()){
                        scb.addParticipantBean(participants.get(qn.getLocalPart()));
                    }
                }
                scb.setChoreographyLoopType(retrieveChoreographyLoopType(sc.getLoopType()));
                
                fillChoreography(scb, sc, elements, participants, messageFlows, defs);
                
                //SEQUENCE FLOWS
                if(sc.getFlowElementsByClass(SequenceFlow.class)!=null){
                    for(SequenceFlow sf : sc.getFlowElementsByClass(SequenceFlow.class)){
                        adaptSequenceFlow(sf, scb, elements, defs);
                    }
                }
                
                Map<String, IBaseElementBean> baseElts = new HashMap<String, IBaseElementBean>();
                baseElts.putAll(participants);
                baseElts.putAll(messageFlows);
                baseElts.putAll(elements);
                adaptArtifacts(sc, scb, baseElts);
                
                pBean.addSubChoreography(scb);
                elements.put(scb.getId(), scb);
            }
            else{
                throw new IllegalArgumentException(fn.getClass()+" is not supported yet as a choreography element.");
            }
        }
    }

	private static List<CollaborationBean> adaptCollaborations(DefinitionsBean def, Collaboration[] collaborations) throws BPMNException {
		List<CollaborationBean> result = null;
		if(collaborations!=null){
			result = new ArrayList<CollaborationBean>();
			for(Collaboration collab : collaborations){
				CollaborationBean cBean = new CollaborationBean(collab.getId());
				def.addCollaboration(cBean);
				cBean.setParentDefinitions(def);
				cBean.setName(collab.getName());
				cBean.setClosed(collab.isIsClosed());
				setDocumentationAndExtensions(cBean, collab);
				
				if(collab.getParticipant()!=null){
					for(Participant p : collab.getParticipant()){
						adaptParticipant(p,cBean,def);
					}
				}
				
				if(collab.getMessageFlow()!=null){
					for(MessageFlow mf : collab.getMessageFlow()){
						adaptMessageFlow(mf,cBean,def);
					}
				}
				
				Map<String, IBaseElementBean> baseElts = new HashMap<String, IBaseElementBean>();
				baseElts.putAll(getMap(cBean.getParticipants()));
				baseElts.putAll(getMap(cBean.getMessageFlows()));
				adaptArtifacts(collab, cBean, baseElts);
			}
		}
		return result;
	}

	private static MessageFlowBean adaptMessageFlow(MessageFlow mf, CollaborationBean cBean, DefinitionsBean defs) {
		MessageFlowBean mfBean = new MessageFlowBean(mf.getId());
		mfBean.setName(mf.getName());
		setDocumentationAndExtensions(mfBean, mf);
		if(mf.getMessageRef()!=null){
			mfBean.setMessage(defs.getMessageById(mf.getMessageRef().getLocalPart()));
		}
		if(mf.getSourceRef()!=null){
			mfBean.setSource((IInteractionNodeBean) cBean.getFlowNode(mf.getSourceRef().getLocalPart()));
		}
		if(mf.getTargetRef()!=null){
			mfBean.setTarget((IInteractionNodeBean) cBean.getFlowNode(mf.getTargetRef().getLocalPart()));
		}
		
		cBean.addMessageFlow(mfBean);
		return mfBean;
	}

	private static void adaptProcesses(Process[] processes, DefinitionsBean def) throws BPMNException {
		if(processes!=null){
		    callActivityProcesses = new HashMap<CallActivityBean, QName>();
			for(Process process : processes){
				defaultFlows = new HashMap<String, WithDefaultSequenceFlow>();
				itemAwareElements = new HashMap<ItemAwareElement, IItemAwareElementBean>();
				dataAssociationTargets = new HashMap<DataAssociationBean, ItemAwareElement>();
				dataAssociationSources = new HashMap<DataAssociationBean, ItemAwareElement[]>();
				
				ProcessBean pBean = new ProcessBean(process.getId());
				setDocumentationAndExtensions(pBean, process);
				pBean.setParentDefinitions(def);
				pBean.setName(process.getName());
				pBean.setExecutable(process.isIsExecutable());
				pBean.setType(retrieveProcessType(process.getProcessType()));
				pBean.setClosed(process.isIsClosed());
				pBean.setIoSpecification(adaptIOSpecifications(process.getIoSpecification(), def));
				
				if(process.hasSupportedInterfaceRef()) {
				    for(QName itfRef : process.getSupportedInterfaceRef()) {
				        pBean.addSupportedInterface(findObjectInDefinitions(def, itfRef, InterfaceBean.class));
				    }
				}
				
				Map<String, FlowElementBean> elements = new HashMap<String, FlowElementBean>();
				if(process.getFlowElement()!=null){
					fillProcess(pBean,process,elements,def);
				}

				if(process.getLaneSets()!=null){
					for(LaneSet ls  : process.getLaneSets()){
						pBean.addLaneSet(adaptLaneSet(ls,elements,def));
					}
				}

				//SEQUENCE FLOWS
				if(process.getFlowElementsByClass(SequenceFlow.class)!=null){
				    for(SequenceFlow sf : process.getFlowElementsByClass(SequenceFlow.class)){
				        adaptSequenceFlow(sf, pBean, elements, def);
				    }
				}
				
				Map<String, IBaseElementBean> baseElts = new HashMap<String, IBaseElementBean>();
				baseElts.putAll(elements);
				baseElts.putAll(getMap(pBean.getLaneSets()));
				adaptArtifacts(process, pBean, baseElts);
				
				for(CallActivityBean ca : callActivityProcesses.keySet()){
				    QName calledEltRef = callActivityProcesses.get(ca);
				    if(calledEltRef!=null) {
				        BaseElementBean e = DefinitionsHelper.getInstance().getElementById(def, calledEltRef.getLocalPart(), calledEltRef.getNamespaceURI());
				        ca.setCalledElement((ICallableElementBean)e);
				        INamespaceDeclaration nsd = def.getNsDeclarationByNamespace(calledEltRef.getNamespaceURI());
				        if(calledEltRef.getPrefix()!=null && !calledEltRef.getPrefix().isEmpty()) {
				            nsd.setPrefix(calledEltRef.getPrefix());
				        }
				    }
	            }
	            for(DataAssociationBean da : dataAssociationTargets.keySet()){
	                da.setTarget(itemAwareElements.get(dataAssociationTargets.get(da)));
	            }
	            for(DataAssociationBean da : dataAssociationSources.keySet()){
	                for(ItemAwareElement iae : dataAssociationSources.get(da)){
	                    da.addSource(itemAwareElements.get(iae));
	                }
	            }
	            
	            for(InputOutputBinding iobinding : process.getIoBinding()) {
	                InputOutputBindingBean iobb = new InputOutputBindingBean(iobinding.getId());
	                if(iobinding.hasOperationRef()) {
	                    iobb.setOperation(findObjectInDefinitions(def, iobinding.getOperationRef(), OperationBean.class));
	                }
	                if(iobinding.hasInputDataRef()) {
	                    iobb.setInputData((IDataInputBean) itemAwareElements.get(iobinding.getInputDataRef()));
	                }
	                if(iobinding.hasOutputDataRef()) {
	                    iobb.setOutputData((IDataOutputBean) itemAwareElements.get(iobinding.getOutputDataRef()));
	                }
	                pBean.addIOBinding(iobb);
	            }
				
				def.addProcess(pBean);
			}
		}
	}
	
	private static void adaptSequenceFlow(SequenceFlow sf, WithSequenceFlowsBean wfe, Map<String, FlowElementBean> elements, DefinitionsBean def){
	    SequenceFlowBean sfb = new SequenceFlowBean(sf.getId());
        setDocumentationAndExtensions(sfb, sf);
        sfb.setName(sf.getName());
        if(sf.getConditionExpression()!=null){
            sfb.setExpression(adaptExpression(sf.getConditionExpression()));
        }
        if(elements.get(sf.getSourceRef().getId()).getClass().equals(TaskBean.class)){
            updateTask((TaskBean)elements.get(sf.getSourceRef().getId()),(Task)sf.getSourceRef(),elements, def);
        }
        if(elements.get(sf.getTargetRef().getId()).getClass().equals(TaskBean.class)){
            updateTask((TaskBean)elements.get(sf.getTargetRef().getId()), (Task)sf.getTargetRef(), elements, def);
        }
        sfb.setSourceNode(elements.get(sf.getSourceRef().getId())); 
        sfb.setTargetNode(elements.get(sf.getTargetRef().getId()));
        
        wfe.addSequenceFlow(sfb);
        elements.put(sfb.getId(), sfb);

        if(defaultFlows.containsKey(sfb.getId())){
            defaultFlows.get(sfb.getId()).setDefaultSequenceFlow(sfb);
        }
	}

	private static ParticipantBean adaptParticipant(Participant participant, CollaborationBean c, DefinitionsBean defs) {
		ParticipantBean result = new ParticipantBean(participant.getId());
		
		result.setName(participant.getName());
		result.setInterfaces(findInterfaces(defs, participant.getInterfaceRef()));
		result.setEndPoints(findEndPoints(defs, participant.getEndPointRef()));
		if(participant.getProcessRef()!=null){
		    IProcessBean pBean = findProcess(defs, participant.getProcessRef().getLocalPart());
		    result.setProcess(pBean);
		    pBean.setDefinitionalCollaboration(c);
		}
		setDocumentationAndExtensions(result, participant);
		
		c.addParticipant(result);
		
		return result;
	}
	
	private static IProcessBean findProcess(DefinitionsBean defs, String id){
		for(IProcessBean p : defs.getProcesses()){
			if(p.getId().equals(id)){
				return p;
			}
		}
		return null;
	}
	
	private static List<IInterfaceBean> findInterfaces(DefinitionsBean defs, QName[] ids){
		List<IInterfaceBean> result = new ArrayList<IInterfaceBean>();
		for(QName id : ids){
		    result.add(findObjectInDefinitions(defs, id, InterfaceBean.class));
		}
		return result;
	}
	
	private static List<IEndPointBean> findEndPoints(DefinitionsBean defs, QName[] ids){
        List<IEndPointBean> result = new ArrayList<IEndPointBean>();
        for(QName id : ids){
            result.add(findObjectInDefinitions(defs, id, EndPointBean.class));
        }
        return result;
    }
	
	private static LaneSetBean adaptLaneSet(LaneSet ls, Map<String, FlowElementBean> elements, DefinitionsBean defs) {
		LaneSetBean lsBean = null;
		if(ls!=null){
			lsBean = new LaneSetBean(ls.getId());
			setDocumentationAndExtensions(lsBean, ls);
			if(ls.getLanes()!=null){
				for(Lane l : ls.getLanes()){
					lsBean.addLane(adaptLane(l,elements,defs));
				}
			}
		}
		return lsBean;
	}

	private static LaneBean adaptLane(Lane l, Map<String, FlowElementBean> elements, DefinitionsBean defs) {
		LaneBean lBean = null;
		if(l!=null){
			lBean = new LaneBean(l.getId());
			lBean.setName(l.getName());
			setDocumentationAndExtensions(lBean, l);
			lBean.setChildLaneSet(adaptLaneSet(l.getChildLaneSet(),elements,defs));
			fillLane(lBean,l,elements,defs);
		}
		return lBean;
	}

	
	private static void fillProcess(WithFlowElementsBean pBean, WithFlowElements process,
			Map<String, FlowElementBean> elements, DefinitionsBean defs) {
		
		for(FlowNode fn : process.getFlowNode()){
			//START EVENTS
			if(fn instanceof StartEvent){
				StartEvent se = (StartEvent) fn;
				StartEventBean seBean = new StartEventBean(se.getId());
				seBean.setName(se.getName());
				setDocumentationAndExtensions(seBean, se);
				if(se.getEventDefinition()!=null){
					for(EventDefinition ed : se.getEventDefinition()){
						EventDefinitionBean edb  = buildEventDefinition(ed,defs);
						setDocumentationAndExtensions(edb, ed);
						seBean.addTrigger(edb);
					}
				}

				adaptDataOutputs(se, seBean, defs);
				addDataOutputAssociations(seBean,se);
				
				pBean.addStartEvent(seBean);
				elements.put(seBean.getId(), seBean);
			}
			//IntermediateCatch EVENTS
			else if(fn instanceof IntermediateCatchEvent){
				IntermediateCatchEvent se = (IntermediateCatchEvent) fn;
				IntermediateCatchEventBean seBean = new IntermediateCatchEventBean(se.getId());
				seBean.setName(se.getName());
				setDocumentationAndExtensions(seBean, se);
				if(se.getEventDefinition()!=null){
					for(EventDefinition ed : se.getEventDefinition()){
						EventDefinitionBean edb  = buildEventDefinition(ed,defs);
						setDocumentationAndExtensions(edb, ed);
						seBean.addTrigger(edb);
					}
				}

				adaptDataOutputs(se, seBean, defs);
				addDataOutputAssociations(seBean,se);
				
				pBean.addIntermediateCatchEvent(seBean);
				elements.put(seBean.getId(), seBean);
			}
			//TASKS
			else if(fn instanceof ReceiveTask){
				adaptReceiveTask(pBean,(ReceiveTask) fn, elements,defs);
			}
			else if(fn instanceof SendTask){
				adaptSendtask(pBean,(SendTask) fn, elements, defs);
			}
			else if(fn instanceof ServiceTask){
				adaptServiceTask(pBean,(ServiceTask) fn, elements, defs);
			}
			else if(fn instanceof ManualTask){
			    adaptManualTask(pBean,(ManualTask)fn,elements);
			}
			else if(fn instanceof Task){
				adaptSimpleTask(pBean,(Task)fn, elements, defs);
			}
			else if(fn instanceof CallActivity){
			    adaptCallActivity(pBean, (CallActivity)fn, elements, defs);
			}
			//GATEWAYS
			else if(fn instanceof ParallelGateway){
				adaptParallelGateway(pBean,(ParallelGateway) fn,elements);
			}
			else if(fn instanceof ExclusiveGateway){
				adaptExclusiveGateway(pBean, (ExclusiveGateway) fn, elements);
			}
			else if(fn instanceof EventBasedGateway){
			    adaptEventBasedGateway(pBean, (EventBasedGateway)fn, elements);
			}
			//IntermediateThrow EVENTS
			else if(fn instanceof IntermediateThrowEvent){
				IntermediateThrowEvent ee = (IntermediateThrowEvent) fn;
				IntermediateThrowEventBean eeb = new IntermediateThrowEventBean(ee.getId());
				eeb.setName(ee.getName());
				setDocumentationAndExtensions(eeb, ee);
				if(ee.getEventDefinition()!=null){
					for(EventDefinition ed : ee.getEventDefinition()){
						EventDefinitionBean edb = buildEventDefinition(ed,defs);
						setDocumentationAndExtensions(edb, ed);
						eeb.addResult(edb);
					}
				}
				
				adaptDataInputs(ee, eeb, defs);
				addDataInputAssociations(eeb,ee);
				
				pBean.addIntermediateThrowEvent(eeb);
				elements.put(eeb.getId(), eeb);
			}
			//END EVENTS
			else if(fn instanceof EndEvent){
				EndEvent ee = (EndEvent) fn;
				EndEventBean eeb = new EndEventBean(ee.getId());
				eeb.setName(ee.getName());
				setDocumentationAndExtensions(eeb, ee);
				if(ee.getEventDefinition()!=null){
					for(EventDefinition ed : ee.getEventDefinition()){
						EventDefinitionBean edb = buildEventDefinition(ed,defs);
						setDocumentationAndExtensions(edb, ed);
						eeb.addResult(edb);
					}
				}
				
				adaptDataInputs(ee, eeb, defs);
				addDataInputAssociations(eeb,ee);
				
				pBean.addEndEvent(eeb);
				elements.put(eeb.getId(), eeb);
			}
			else if(fn instanceof SubProcess){
			    SubProcess sp = (SubProcess) fn;
			    SubProcessBean spb = new SubProcessBean(sp.getId());
			    spb.setName(sp.getName());
			    spb.setIoSpecification(adaptIOSpecifications(sp.getIoSpecification(), defs));
			    addDataAssociations(spb, sp);
			    setDocumentationAndExtensions(spb, sp);
			   
			    fillProcess(spb, sp, elements, defs);
			    
			    if(sp.getLaneSets()!=null){
			        for(LaneSet ls : sp.getLaneSets()){
			            adaptLaneSet(ls, elements, defs);
			        }
			    }
			    
			    if(sp.getFlowElementsByClass(SequenceFlow.class)!=null){
			        for(SequenceFlow sf : sp.getFlowElementsByClass(SequenceFlow.class)){
			            adaptSequenceFlow(sf, spb, elements, defs);
			        }
			    }
			    
			    Map<String, IBaseElementBean> baseElts = new HashMap<String, IBaseElementBean>();
                baseElts.putAll(elements);
                baseElts.putAll(getMap(spb.getLaneSets()));
                adaptArtifacts(sp, spb, baseElts);
			    
                addDataAssociations(spb, sp);
                
			    pBean.addSubProcess(spb);
			    elements.put(spb.getId(), spb);
			}
			else{
				throw new IllegalArgumentException(fn.getClass()+" is not supported yet as a process element.");
			}
		}
		
		for(DataObject do_ : process.getFlowElementsByClass(DataObject.class)){
            pBean.addDataObject(adaptDataObject(do_,defs));
        }
		
	}
	
    private static DataObjectBean adaptDataObject(DataObject do1, DefinitionsBean defs) {
        DataObjectBean dob = new DataObjectBean(do1.getId());
        setDocumentationAndExtensions(dob, do1);
        dob.setName(do1.getName());
        if(do1.getItemSubjectRef()!=null) {
            dob.setItemSubject(defs.getItemDefinitionById(do1.getItemSubjectRef().getLocalPart()));
        }
        
        itemAwareElements.put(do1, dob);
        
        return dob;
    }

    private static void fillLane(LaneBean lBean, Lane l, Map<String, FlowElementBean> elements, DefinitionsBean defs) {
		if(l.getFlowNodeRef()!=null){
			for(FlowNode fn : l.getFlowNodeRef()){
				//START EVENTS
				if(fn instanceof StartEvent){
					lBean.addStartEvent((StartEventBean)elements.get(fn.getId()));
				}
				//IntermediateCatch EVENTS
				else if(fn instanceof IntermediateCatchEvent){
					lBean.addIntermediateCatchEvent((IntermediateCatchEventBean)elements.get(fn.getId()));
				}
				//TASKS
				else if(fn instanceof Task){
					lBean.addTask((TaskBean)elements.get(fn.getId()));
				}
				//GATEWAYS
				else if(fn instanceof Gateway){
					lBean.addGateway((GatewayBean)elements.get(fn.getId()));
				}
				//IntermediateThrow EVENTS
				else if(fn instanceof IntermediateThrowEvent){
					lBean.addIntermediateThrowEvent((IntermediateThrowEventBean)elements.get(fn.getId()));
				}
				//END EVENTS
				else if(fn instanceof EndEvent){
					lBean.addEndEvent((EndEventBean)elements.get(fn.getId()));
				}
				//SUB PROCESSES
				else if(fn instanceof SubProcess){
				    lBean.addSubProcess((SubProcessBean)elements.get(fn.getId()));
				}
			}
		}
		
	}
	


	private static void updateTask(TaskBean taskBean, Task t, Map<String,FlowElementBean> elements, DefinitionsBean defs) {
		if(taskBean instanceof ReceiveTaskBean){
			ReceiveTaskBean rtb = (ReceiveTaskBean) taskBean;
			ReceiveTask rt = (ReceiveTask) t;
			rtb.setInstanciate(rt.isInstantiate());
			if(rt.getOperationRef()!=null){
				rtb.setOperation(defs.getOperationById(rt.getOperationRef().getLocalPart()));
			}
			if(rt.getMessageRef()!=null){
				rtb.setMessage(defs.getMessageById(rt.getMessageRef().getLocalPart()));
			}
			elements.put(rtb.getId(), rtb);
		}
		else if(taskBean instanceof SendTaskBean){
			SendTaskBean stb = (SendTaskBean) taskBean;
			SendTask st = (SendTask) t;
			if(st.getOperationRef()!=null){
				stb.setOperation(defs.getOperationById(st.getOperationRef().getLocalPart()));
			}
			if(st.getMessageRef()!=null){
				stb.setMessage(defs.getMessageById(st.getMessageRef().getLocalPart()));
			}
			elements.put(stb.getId(), stb);
		}
		else if(taskBean instanceof ServiceTaskBean){
			ServiceTaskBean stb = (ServiceTaskBean) taskBean;
			ServiceTask st = (ServiceTask) t;
			if(st.getOperationRef()!=null){
				stb.setOperation(defs.getOperationById(st.getOperationRef().getLocalPart()));
			}
			elements.put(stb.getId(), stb);
		}
	}

	private static void adaptExclusiveGateway(WithGatewaysBean pBean, ExclusiveGateway eg, Map<String, FlowElementBean> elements) {
		ExclusiveGatewayBean egb = new ExclusiveGatewayBean(eg.getId());
		egb.setName(eg.getName());
		setDocumentationAndExtensions(egb, eg);
		if(eg.getDefault()!=null){
			defaultFlows.put(eg.getDefault().getId(), egb);
		}
		pBean.addGateway(egb);
		elements.put(egb.getId(), egb);
	}

	private static void adaptParallelGateway(WithGatewaysBean pBean, ParallelGateway pg, Map<String, FlowElementBean> elements) {
		ParallelGatewayBean pgb = new ParallelGatewayBean(pg.getId());
		pgb.setName(pg.getName());
		setDocumentationAndExtensions(pgb, pg);
		pBean.addGateway(pgb);
		elements.put(pgb.getId(), pgb);
	}
	
	private static void adaptEventBasedGateway(WithGatewaysBean pBean, EventBasedGateway eg,
            Map<String, FlowElementBean> elements) {
	    EventBasedGatewayBean pgb = new EventBasedGatewayBean(eg.getId());
        pgb.setName(eg.getName());
        setDocumentationAndExtensions(pgb, eg);
        pgb.setInstantiate(eg.isInstantiate());
        pgb.setType(retrieveEventGatewayType(eg.getEventGatewayType()));
        pBean.addGateway(pgb);
        elements.put(pgb.getId(), pgb);
    }
	
	private static void adaptManualTask(WithFlowElementsBean pBean, ManualTask t, Map<String, FlowElementBean> elements) {
	    ManualTaskBean tb = new ManualTaskBean(t.getId());
        tb.setName(t.getName());
        setDocumentationAndExtensions(tb, t);
        addDataAssociations(tb, t);
        if(t.hasDefault()) {
            defaultFlows.put(t.getDefault().getId(), tb);
        }
        
        pBean.addTask(tb);
        elements.put(tb.getId(), tb);
    }
	
	private static void adaptCallActivity(WithFlowElementsBean pBean, CallActivity a, Map<String, FlowElementBean> elements, DefinitionsBean defs) {
	    CallActivityBean ca  = new CallActivityBean(a.getId());
	    ca.setName(a.getName());
	    setDocumentationAndExtensions(ca, a);
	    addDataAssociations(ca, a);
	    ca.setIoSpecification(adaptIOSpecifications(a.getIoSpecification(), defs));
	    
	    callActivityProcesses.put(ca, a.getCalledElement());
	    
	    pBean.addCallActivity(ca);
	    elements.put(ca.getId(), ca);
	}
	
    private static void adaptSimpleTask(WithFlowElementsBean pBean, Task t, Map<String, FlowElementBean> elements, DefinitionsBean defs) {
		TaskBean tb = new TaskBean(t.getId());
		tb.setName(t.getName());
		setDocumentationAndExtensions(tb, t);
		addDataAssociations(tb, t);
		tb.setIoSpecification(adaptIOSpecifications(t.getIoSpecification(), defs));
		if(t.hasDefault()) {
            defaultFlows.put(t.getDefault().getId(), tb);
        }
		
		pBean.addTask(tb);
		elements.put(tb.getId(), tb);
	}

	private static void adaptServiceTask(WithFlowElementsBean pBean, ServiceTask st, Map<String, FlowElementBean> elements, DefinitionsBean defs) {
		ServiceTaskBean stb = new ServiceTaskBean(st.getId());
		stb.setName(st.getName());
		setDocumentationAndExtensions(stb, st);
		if(st.getOperationRef()!=null){
			stb.setOperation(defs.getOperationById(st.getOperationRef().getLocalPart()));
		}
		addDataAssociations(stb, st);
		stb.setIoSpecification(adaptIOSpecifications(st.getIoSpecification(), defs));
		if(st.hasDefault()) {
            defaultFlows.put(st.getDefault().getId(), stb);
        }
		
		pBean.addTask(stb);
		elements.put(stb.getId(), stb);
	}

	private static void adaptSendtask(WithFlowElementsBean pBean, SendTask st, Map<String, FlowElementBean> elements, DefinitionsBean defs) {
		SendTaskBean stb = new SendTaskBean(st.getId());
		stb.setName(st.getName());
		setDocumentationAndExtensions(stb, st);
		if(st.getOperationRef()!=null){
			stb.setOperation(defs.getOperationById(st.getOperationRef().getLocalPart()));
		}
		if(st.getMessageRef()!=null){
			stb.setMessage(defs.getMessageById(st.getMessageRef().getLocalPart()));
		}
		addDataAssociations(stb,st);
		stb.setIoSpecification(adaptIOSpecifications(st.getIoSpecification(), defs));
		if(st.hasDefault()) {
            defaultFlows.put(st.getDefault().getId(), stb);
        }
		
		pBean.addTask(stb);
		elements.put(stb.getId(), stb);
	}

	private static void adaptReceiveTask(WithFlowElementsBean pBean, ReceiveTask rt, Map<String, FlowElementBean> elements, DefinitionsBean defs) {
		ReceiveTaskBean rtb = new ReceiveTaskBean(rt.getId()); 
		rtb.setName(rt.getName());
		rtb.setInstanciate(rt.isInstantiate());
		setDocumentationAndExtensions(rtb, rt);
		if(rt.getOperationRef()!=null){
			rtb.setOperation(defs.getOperationById(rt.getOperationRef().getLocalPart()));
		}
		if(rt.getMessageRef()!=null){
			rtb.setMessage(defs.getMessageById(rt.getMessageRef().getLocalPart()));
		}
		addDataAssociations(rtb,rt);
		rtb.setIoSpecification(adaptIOSpecifications(rt.getIoSpecification(), defs));
		if(rt.hasDefault()) {
		    defaultFlows.put(rt.getDefault().getId(), rtb);
		}
		
		pBean.addTask(rtb);
		elements.put(rtb.getId(), rtb);
	}

	
	private static void addDataInputAssociations(WithDataInputAssociationBean wdib, WithDataInputAssociation wdi){
	    if(wdi.getDataInputAssociation()!=null){
            for(DataInputAssociation dia : wdi.getDataInputAssociation()){
                DataInputAssociationBean diab = new DataInputAssociationBean(dia.getId());
                setDocumentationAndExtensions(diab, dia);
                if(dia.hasSourceRef()){
                    ensureIdIsSet(diab);
                    dataAssociationSources.put(diab, dia.getSourceRef());
                }
                if(dia.hasTargetRef()){
                    ensureIdIsSet(diab);
                    dataAssociationTargets.put(diab, dia.getTargetRef());
                }
                if(dia.getAssignment()!=null){
                    for(Assignment a : dia.getAssignment()){
                        diab.addAssignement(adaptAssignement(a));
                    }
                }
                wdib.addDataInputAssociation(diab);
            }
        }
	}
	
    private static void addDataOutputAssociations(WithDataOutputAssociationBean wdob, WithDataOutputAssociation wdo){
        if(wdo.getDataOutputAssociation()!=null){
            for(DataOutputAssociation doa : wdo.getDataOutputAssociation()){
                DataOutputAssociationBean doab = new DataOutputAssociationBean(doa.getId());
                setDocumentationAndExtensions(doab, doa);
                if(doa.hasSourceRef()){
                    ensureIdIsSet(doab);
                    dataAssociationSources.put(doab, doa.getSourceRef());
                }
                if(doa.hasTargetRef()){
                    ensureIdIsSet(doab);
                    dataAssociationTargets.put(doab, doa.getTargetRef());
                }
                if(doa.getAssignment()!=null){
                    for(Assignment a : doa.getAssignment()){
                        doab.addAssignement(adaptAssignement(a));
                    }
                }
                wdob.addDataOutputAssociation(doab);
            }
        }
    }
	
	private static IOSpecificationBean adaptIOSpecifications(InputOutputSpecification iospec, DefinitionsBean defs){
	    if(iospec!=null){
	        IOSpecificationBean res = new IOSpecificationBean(iospec.getId());
	        setDocumentationAndExtensions(res, iospec);
	       
	        adaptDataInputs(iospec, res, defs);
	        adaptDataOutputs(iospec, res, defs);
	        
	        return res;
	    }
	    return null;
	}
	
	private static void adaptDataInputs(WithDataInput wdi, WithDataInputBean wdib, DefinitionsBean defs){
	    if(wdi.hasDataInput()){
            for(DataInput di : wdi.getDataInput()){
                DataInputBean dib = new DataInputBean(di.getId());
                setDocumentationAndExtensions(dib, di);
                dib.setName(di.getName());
                dib.setCollection(di.isIsCollection());
                adaptItemAwareElement(dib, di, defs);
                
                wdib.addDataInput(dib);
                
                itemAwareElements.put(di, dib);
            }
        }
	}
	
	private static void adaptDataOutputs(WithDataOutput wdi, WithDataOutputBean wdib, DefinitionsBean defs){
	    if(wdi.hasDataOutput()){
            for(DataOutput do_ : wdi.getDataOutput()){
                DataOutputBean dob = new DataOutputBean(do_.getId());
                setDocumentationAndExtensions(dob, do_);
                dob.setName(do_.getName());
                dob.setCollection(do_.isIsCollection());
                adaptItemAwareElement(dob, do_, defs);
                
                wdib.addDataOutput(dob);
                
                itemAwareElements.put(do_, dob);
            }
        }
	}
	
	private static void adaptItemAwareElement(IItemAwareElementBean iaeb, ItemAwareElement iae, DefinitionsBean defs){
	    if(iae.hasItemSubjectRef()){
            iaeb.setItemSubject(defs.getItemDefinitionById(iae.getItemSubjectRef().getLocalPart()));
        }
        if(iae.hasDataState()){
            iaeb.setDataState(iae.getDataState().getName());
        }
	}
	
	private static void addDataAssociations(ActivityBean tb, Activity t) {
	    addDataInputAssociations(tb, t);
	    addDataOutputAssociations(tb, t);
	}

	private static AssignmentBean adaptAssignement(Assignment a) {
		AssignmentBean ab = null;
		if(a!=null){
			ab = new AssignmentBean(a.getId());
			setDocumentationAndExtensions(ab, a);
			ab.setFrom(adaptExpression(a.getFrom()));
			ab.setTo(adaptExpression(a.getTo()));
		}
		return ab;
	}

	private static ExpressionBean adaptExpression(Expression exp) {
		ExpressionBean expb = new ExpressionBean(exp.getId());
		expb.setContent(exp.getTextContent());
		//setDocumentation(expb, exp);
		return expb;
	}

	private static OperationBean adaptOperation(DefinitionsBean defs, Operation op) {
		OperationBean opBean = new OperationBean(op.getId());
		setDocumentationAndExtensions(opBean, op);
		opBean.setName(op.getName());
		opBean.setMessageIn(defs.getMessageById(op.getInMessageRef().getLocalPart()));
		if(op.getOutMessageRef()!=null){
			opBean.setMessageOut(defs.getMessageById(op.getOutMessageRef().getLocalPart()));
		}
		if(op.getErrorRef()!=null){
			for(QName qn : op.getErrorRef()){
			    for(IErrorBean err : defs.getErrors()) {
			        if(err.getId().equals(qn.getLocalPart())) {
			            opBean.addError(err);
			            break;
			        }
			    }
			}
		}
		if(op.hasImplementationRef()){
		    opBean.setWsdlImplementationRef(getStringFromQname(op.getImplementationRef(), defs));
		    //opBean.setWsdlImplementationRef(op.getImplementationRef().toString());
		}
		return opBean;
	}

	private static void adaptMessages(DefinitionsBean defs, Message[] messages){
		if(messages!=null){
			for(Message msg : messages){
				MessageBean msgBean = new MessageBean(msg.getId());
				msgBean.setName(msg.getName());
				setDocumentationAndExtensions(msgBean, msg);
				if(msg.getItemRef()!=null){
					msgBean.setItemDefinition(defs.getItemDefinitionById(msg.getItemRef().getLocalPart()));
				}
				
				defs.addMessage(msgBean);
			}
		}
	}
	
	private static void adaptItemDefinitions(DefinitionsBean defs, ItemDefinition[] itemDefinitions) {
		if(itemDefinitions!=null){
			for(ItemDefinition itemDefinition : itemDefinitions){
				ItemDefinitionBean result = new ItemDefinitionBean(itemDefinition.getId());
				setDocumentationAndExtensions(result, itemDefinition);
				result.setCollection(itemDefinition.isIsCollection());
				result.setItemKind(adaptItemKind(itemDefinition.getItemKind()));
				
				if(itemDefinition.hasStructureRef()){
				    result.setStructureRef(getStringFromQname(itemDefinition.getStructureRef(), defs));
					//result.setStructureRef(itemDefinition.getStructureRef().toString());
				}
				
				defs.addItemDefinition(result);
			}
		}
	}

	private static ItemKind adaptItemKind(TItemKind itemKind) {
		ItemKind result = null;
		if(itemKind!=null){
			result = retrieveItemKind(itemKind);
		}
		return result;
	}

	private static List<ImportBean> adaptImports(DefinitionsBean defBean, Import[] imports, Definitions defs) throws BPMNException {
		List<ImportBean> result = null;
		if(imports!=null){
			result = new ArrayList<ImportBean>();
			Map<String,String> prefixes = defs.findImportPrefixes(); 
			for(Import impt : imports){
			    String prefix = prefixes.get(impt.getNamespace());
			    ImportBean importBean;
			    if(impt.isXSDImport()){
			        importBean = ImportAdapter.adaptSchemaImport(prefix, impt.getLocation(), impt.getSchema());
			    }
			    else if(impt.isWSDL11Import()) {
			        importBean = ImportAdapter.adaptWSDLImport(prefix, impt.getLocation(), impt.getWSDL11Definitions());
			    }
			    else if(impt.isBPMN20Import()) {
			        importBean = ImportAdapter.adaptBPMNImport(prefix, impt.getLocation(), impt.getBPMNDefinitions());
			    }
			    else {
			        NamespaceDeclaration nsd = new NamespaceDeclaration(null, impt.getNamespace());
			        importBean = new ImportBean(nsd);
	                importBean.setLocation(impt.getLocation().toString());
	                importBean.setImportType(impt.getImportType().toString());
			    }
				
			    defBean.addImportNSDeclaration(importBean.getNSDeclaration());
				defBean.addImport(importBean);
				result.add(importBean);
			}
		}
		return result;
	}
	
	
	private static void adaptArtifacts(WithArtifact wa, WithArtifactsBean wab,Map<String, IBaseElementBean> elements){
	    if(wa.getArtifact()==null){return;}
	    for(Artifact artifact : wa.getArtifact()){
	        if(artifact instanceof TextAnnotation){
	            TextAnnotation ta = (TextAnnotation) artifact;
	            TextAnnotationBean tab = new TextAnnotationBean(ta.getId());
	            setDocumentationAndExtensions(tab, ta);
	            tab.setTextFormat(ta.getTextFormat());
	            if(ta.getText()!=null){
	                tab.setText(ta.getText().getXmlObjectTextContent());
	            }
	            wab.addArtifact(tab);
	        }
	        else if(artifact instanceof Association){
	            Association a = (Association) artifact;
	            AssociationBean ab = new AssociationBean(a.getId());
	            setDocumentationAndExtensions(ab, a);
	            ab.setDirection(retrieveAssociationDirection(a.getAssociationDirection()));
	            if(a.getSourceRef()!=null){
	                ab.setSource(elements.get(a.getSourceRef().getLocalPart()));
	            }
	            if(a.getTargetRef()!=null){
                    ab.setTarget(elements.get(a.getTargetRef().getLocalPart()));
                }
	            wab.addArtifact(ab);
	        }
	        else{
	            throw new IllegalArgumentException(artifact.getClass()+" is not supported yet.");
	        }
	    }
	}
	
	
	private static void setDocumentationAndExtensions(BaseElementBean bean, BaseElement element){
	    //assign a random id if the element did not have any
	    if(element.getId()==null) {
	        bean.setId(UUID.randomUUID().toString());
	    }
	    
	    if(nsDeclaration!=null) {
	        bean.setNSDeclaration(nsDeclaration);
	    }
	    
		if(element.getDocumentations()!=null){
		    StringBuffer sb = new StringBuffer();
			for(Documentation doc : element.getDocumentations()){
				sb.append(doc.getXmlObjectTextContent());
			}
			bean.setDocumentation(sb.toString());
		}
		
		if(element.hasExtensionElements()){
			ExtensionElements ee = element.getExtensionElements();
			for(XmlObject ext : ee.getAnyXmlObjects()){
				ObjectExtension obj = bindingManager.serverToClient(ext);
				bean.addObjectExtension(obj);
			}
		}
		adaptAttributeExtensions(bean, element);
	}
	
	private static void adaptAttributeExtensions(WithAttributeExtension bean, WithOtherAttributes element){
		for(QName qn : element.getOtherAttributes().keySet()){
			String value = element.getOtherAttribute(qn);
			bean.addAttributeExtension(new AttributeExtension(qn.getNamespaceURI(), qn.getLocalPart(), value));
		}
	}
	
	private static <T extends BaseElementBean> T findObjectInDefinitions(DefinitionsBean defsBean,
	        QName ref, Class<T> clazz) {
	    return clazz.cast(DefinitionsHelper.getInstance().getElementById(defsBean, ref.getLocalPart(), ref.getNamespaceURI()));
	}
	
	private static EventDefinitionBean buildEventDefinition(EventDefinition ed, DefinitionsBean defs){
		EventDefinitionBean result = null;
		if(ed instanceof MessageEventDefinition){
			result = new MessageEventDefinitionBean(ed.getId());
			QName messageID = ((MessageEventDefinition)ed).getMessageRef();
			if(messageID!=null){
				((MessageEventDefinitionBean)result).setMessage(findObjectInDefinitions(defs, messageID, MessageBean.class));
			}
			QName operationId = ((MessageEventDefinition)ed).getOperationRef();
			if(operationId!=null) {
			    ((MessageEventDefinitionBean)result).setOperation(findObjectInDefinitions(defs, operationId, OperationBean.class));
			}
		}
		else if(ed instanceof ConditionalEventDefinition){
			result = new ConditionalEventDefinitionBean(ed.getId());
			Expression exp = ((ConditionalEventDefinition)ed).getCondition();
			if(exp!=null){
				((ConditionalEventDefinitionBean)result).setCondition(exp.getTextContent());
			}
		}
		else if(ed instanceof TerminateEventDefinition){
			result = new TerminateEventDefinitionBean(ed.getId());
		}
		else if(ed instanceof TimerEventDefinition){
			result = new TimerEventDefinitionBean(ed.getId());
			Expression expC = ((TimerEventDefinition)ed).getTimeCycle();
			if(expC!=null){
				((TimerEventDefinitionBean)result).setTimeCycle(expC.getTextContent());
			}
			Expression expT = ((TimerEventDefinition)ed).getTimeDate();
			if(expT!=null){
				((TimerEventDefinitionBean)result).setTimeDate(expT.getTextContent());
			}
			Expression expD = ((TimerEventDefinition)ed).getTimeDuration();
			if(expD!=null){
				((TimerEventDefinitionBean)result).setTimeDuration(expD.getTextContent());
			}
		}
		
		if(result!=null){
			setDocumentationAndExtensions(result, ed);
			defs.addEventDefinition(result);
			return result;
		}
		else{
			throw new IllegalArgumentException(ed.getClass()+" is not supported yet.");
		}
	}
	
	private static String getStringFromQname(QName ref, DefinitionsBean defsBean) {
	    if(ref!=null){
            if(ref.getPrefix()!=null && !ref.getPrefix().isEmpty()) {
                if(defsBean.getNsDeclarationByNamespace(ref.getNamespaceURI())!=null) {
                    defsBean.getNsDeclarationByNamespace(ref.getNamespaceURI()).setPrefix(ref.getPrefix());
                }
                return ref.getPrefix()+":"+ref.getLocalPart();
            }
            else if(ref.getNamespaceURI()!=null 
                    && defsBean.getNsDeclarationByNamespace(ref.getNamespaceURI())!=null){
                return defsBean.getNsDeclarationByNamespace(ref.getNamespaceURI()).getPrefix()+":"+ref.getLocalPart();
            }
            else {
                return ref.getLocalPart();
            }
        }
	    return null;
	}
	
	private static ProcessTypes retrieveProcessType(TProcessType type) {
		ProcessTypes result = null;
		if(type!=null){
			switch (type) {
			case Private:
				result = ProcessTypes.PRIVATE;
				break;
			case None:
				result = ProcessTypes.NONE;
				break;
			case Public:
				result = ProcessTypes.PUBLIC;
				break;
			}
		}
		return result;
	}
	
	private static ItemKind retrieveItemKind(TItemKind itemKind) {
		ItemKind result = null;
		switch (itemKind) {
		case Information:
			result = ItemKind.INFORMATION;
			break;
		case Physical:
			result = ItemKind.PHYSICAL;
			break;
		}
		return result;
	}

	private static EventGatewayType retrieveEventGatewayType(TEventBasedGatewayType eventGatewayType) {
	    EventGatewayType result = null;
	    switch(eventGatewayType){
	        case Exclusive:
	            result = EventGatewayType.Exclusive;
	            break;
	        case Parallel:
	            result = EventGatewayType.Parallel;
	            break;
	    }
        return result;
    }
	
	private static ChoreographyLoopType retrieveChoreographyLoopType(TChoreographyLoopType type){
	    ChoreographyLoopType result = null;
        if(type!=null){
            switch (type) {
                case MultiInstanceParallel:
                    result = ChoreographyLoopType.MultiInstanceParallel;
                    break;
                case MultiInstanceSequential:
                    result = ChoreographyLoopType.MultiInstanceSequential;
                    break;
                case None:
                    result = ChoreographyLoopType.None;
                    break;
                case Standard:
                    result = ChoreographyLoopType.Standard;
                    break;
            }
        }
        return result;
    }
	
	private static AssociationDirection retrieveAssociationDirection(TAssociationDirection dir) {
        AssociationDirection result = null;
        if(dir!=null){
            switch(dir){
                case Both:
                    result = AssociationDirection.Both;
                    break;
                case None:
                    result = AssociationDirection.None;
                    break;
                case One:
                    result = AssociationDirection.One;
                    break;
            }
        }
        return result;
    }
	
	private static void ensureIdIsSet(BaseElementBean e) {
	    if(e.getId()==null){
	        e.setId(IdGenerator.createUniqueId());
	    }
    }
	
	private static Map<String,IBaseElementBean> getMap(List<? extends IBaseElementBean> l){
	    Map<String,IBaseElementBean> res = new HashMap<String,IBaseElementBean>();
	    for(IBaseElementBean e : l){
	        res.put(e.getId(), e);
	    }
	    return res;
	}
}
