/**
 * petalsbpm-service - BPMN Editor service - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.petalsbpm.server.service.bpmn2.serverToClient;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.xml.namespace.QName;

import com.ebmwebsourcing.easybpmn.bpmn20.api.BPMNException;
import com.ebmwebsourcing.easybpmn.bpmn20.api.ImportType;
import com.ebmwebsourcing.easyschema10.api.element.Schema;
import com.ebmwebsourcing.easywsdl11.api.element.Binding;
import com.ebmwebsourcing.easywsdl11.api.element.Definitions;
import com.ebmwebsourcing.easywsdl11.api.element.Fault;
import com.ebmwebsourcing.easywsdl11.api.element.Message;
import com.ebmwebsourcing.easywsdl11.api.element.Operation;
import com.ebmwebsourcing.easywsdl11.api.element.PortType;
import com.ebmwebsourcing.geasyschema.domain.api.ISchema;
import com.ebmwebsourcing.geasyschema.service.serverToClient.SchemaParser;
import com.ebmwebsourcing.geasywsdl.domain.api.IDefinitions;
import com.ebmwebsourcing.geasywsdl.service.serverToClient.DefinitionsParser;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.Constants.ItemKind;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IErrorBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IItemDefinitionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IMessageBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.service.IInterfaceBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.common.ErrorBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.common.ItemDefinitionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.common.MessageBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.infrastructure.DefinitionsBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.infrastructure.ImportBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.infrastructure.NamespaceDeclaration;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.infrastructure.xsd.BPMN20ImportBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.infrastructure.xsd.WSDLImportBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.infrastructure.xsd.XSDImportBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.service.InterfaceBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.service.OperationBean;

public class ImportAdapter {
    
    public static BPMN20ImportBean adaptBPMNImport(String nsprefix, String location, com.ebmwebsourcing.easybpmn.bpmn20.api.element.Definitions defs) throws BPMNException{
        NamespaceDeclaration nsd = new NamespaceDeclaration(nsprefix, defs.getTargetNamespace());
        DefinitionsBean defsBean = ServerToClient.adapt(defs,nsd);
        
        BPMN20ImportBean res = new BPMN20ImportBean(nsd, defsBean);
        res.setLocation(location);
        
        return res;
    }
	
    public static XSDImportBean adaptSchemaImport(String nsprefix, String location, Schema schema){
        ISchema ischema = new SchemaParser().parseSchema(schema);
        NamespaceDeclaration nsd = new NamespaceDeclaration(nsprefix, schema.getTargetNamespace());
        
        XSDImportBean res = new XSDImportBean(nsd,ischema);
        res.setLocation(location);
        
        return res;
    }

    public static ImportBean adaptWSDLImport(String nsprefix, String location, Definitions wsdl11Definitions) {
        messages = new HashMap<String,MessageBean>();
        errors = new HashMap<String, ErrorBean>();

        IDefinitions defs = new DefinitionsParser().parseDefinitions(wsdl11Definitions);
        NamespaceDeclaration nsd = new NamespaceDeclaration(nsprefix, wsdl11Definitions.getTargetNamespace());
        WSDLImportBean res = new WSDLImportBean(nsd,defs);
        
        res.setLocation(location);
        res.setImportType(ImportType.WSDL11.getValue());
        
        Set<String> implementedItfNames = new HashSet<String>(); 
        for(Binding binding : wsdl11Definitions.getBindings()) {
            implementedItfNames.add(binding.getType().getLocalPart());
        }
        
        Set<IInterfaceBean> interfaces = new HashSet<IInterfaceBean>();
        Set<IItemDefinitionBean> itemDefinitions = new HashSet<IItemDefinitionBean>();
        for(PortType pt : wsdl11Definitions.getPortTypes()) {
            if(!implementedItfNames.contains(pt.getName())) {
                continue;
            }
            
            InterfaceBean itf = new InterfaceBean(pt.getName()+"Interface");
            itf.setName(pt.getName());
            itf.setWsdlImplementationRef(new QName(wsdl11Definitions.getTargetNamespace(),pt.getName()).toString());
            
            interfaces.add(itf);
            
            for(Operation op : pt.getOperations()) {
                OperationBean opb = new OperationBean(op.getName()+"Operation");
                opb.setName(op.getName());
                opb.setWsdlImplementationRef(new QName(wsdl11Definitions.getTargetNamespace(),op.getName()).toString());
                itf.addOperation(opb);
                
                if(op.hasInput()) {
                    IMessageBean inMsg = createMessageFromRef(op.getInput().getMessage(),wsdl11Definitions, itemDefinitions);
                    opb.setMessageIn(inMsg);
                }
                if(op.hasOutput()) {
                    IMessageBean outMsg = createMessageFromRef(op.getOutput().getMessage(),wsdl11Definitions, itemDefinitions);
                    opb.setMessageOut(outMsg);
                }
                for(Fault fault : op.getFaults()) {
                    IErrorBean err = createErrorFromRef(fault.getMessage(), wsdl11Definitions, itemDefinitions);
                    opb.addError(err);
                }
            }
        }
        
        res.setInterfaces(interfaces);
        res.setItemDefinitions(itemDefinitions);
        return res;
    }

    private static Map<String, MessageBean> messages;
    private static Map<String, ErrorBean> errors;
    
    private static IMessageBean createMessageFromRef(QName messageRef, Definitions wsdl11Definitions, Set<IItemDefinitionBean> itemDefinitions) {
        if(messages.containsKey(messageRef.getLocalPart())) {
            return messages.get(messageRef.getLocalPart());
        }
        
        Message m = null;
        for(Message msg : wsdl11Definitions.getMessages()) {
            if(msg.getName().equals(messageRef.getLocalPart())) {
                m = msg;
                break;
            }
        }

        MessageBean mb = new MessageBean(m.getName()+"Message");
        mb.setName(m.getName());
        
        ItemDefinitionBean idb = new ItemDefinitionBean(mb.getId()+"ItemDefinition");
        idb.setCollection(false);
        idb.setStructureRef(m.getParts()[0].getElement().toString());
        idb.setItemKind(ItemKind.INFORMATION);
        mb.setItemDefinition(idb);
        itemDefinitions.add(idb);
        
        messages.put(messageRef.getLocalPart(), mb);
        
        return mb;
    }
    
    private static IErrorBean createErrorFromRef(QName messageRef, Definitions wsdl11Definitions, Set<IItemDefinitionBean> itemDefinitions) {
        if(errors.containsKey(messageRef.getLocalPart())) {
            return errors.get(messageRef.getLocalPart());
        }
        
        Message m = null;
        for(Message msg : wsdl11Definitions.getMessages()) {
            if(msg.getName().equals(messageRef.getLocalPart())) {
                m = msg;
                break;
            }
        }
        
        ErrorBean eb = new ErrorBean(m.getName()+"Error");
        eb.setName(m.getName());
        
        ItemDefinitionBean idb = new ItemDefinitionBean(eb.getId()+"ItemDefinition");
        idb.setCollection(false);
        idb.setStructureRef(m.getParts()[0].getElement().toString());
        idb.setItemKind(ItemKind.INFORMATION);
        eb.setStructureRef(idb);
        itemDefinitions.add(idb);
        
        errors.put(messageRef.getLocalPart(), eb);
        
        return eb;
    }
    
}
