importPackage(java.lang);
importPackage(java.io);
importPackage(java.util);
importPackage(java.util.zip);

// OS switch
var isWindows = System.getProperty("os.name").startsWith("Windows");
var isMac = System.getProperty("os.name").startsWith("Mac");

// OS independent absolute path to Java executable
var javaExe = System.getProperty("java.home") + File.separator + "bin"
		+ File.separator + (isWindows ? "java.exe" : "java");

// the dir that contains 
var rootDir = "/home/bft/soceda_livrable";

// The dir that contains all installed components
var instDir = new File("inst");

// The dir that contains all components installers
var originDir = new File("origin");

// To open a cmd in new terminal on Windows
var termWindows = [ "cmd.exe", "/c", "start" ];

// The Unix command prefix
var xTerm = "/usr/bin/xterm";
var gnomeTerm = "/usr/bin/gnome-terminal";
var kdeTerm = "/usr/bin/konsole";

var termMac = [ "/usr/bin/open", "-a", "Terminal" ];

// if tmux is available, launch the different programs in different windows of a
// tmux session
// Using macport under mac os x, you can find it under /opt/usr/local/bin
var tmuxTerm = "/usr/bin/tmux";

var sshCmd = "ssh -X";

var curlCmd = "/usr/bin/curl";

/**
 * Check for correct dirs.
 */
function checkCurrentLocation() {
	// Perform several checks and run the installation if required
	// The dir that contains all installed components
	if (!instDir.exists()) {
		throw "Invalid location, unable to find " + instDir
				+ " please run this script in the correct dir";
	}

	// The dir that contains all components installers
	if (!originDir.exists()) {
		throw "Invalid location, unable to find " + originDir
				+ " please run this script in the correct dir";
	}
}

function createCmdForRemoteExec(cmd, cmdarray, wdir, host) {
	//format of the command: cmd ssh host cmdarray
	//println("calling createCmdForRemoteExec(" + cmd + " ## " + cmdarray + " ## " + wdir + " ## " + host);
	
	wdir = rootDir.concat("/", wdir);
	//println(wdir);
	
	// sleep is there only for debugging: if a fault occurs, then the window does not terminate
	var sshcmd = sshCmd.concat(" ", host, " 'cd ", wdir, "; ", cmdarray.join(" "), "; sleep 86400'");
	//println(sshcmd);
	
	cmd = cmd.concat(sshcmd);
	//println(cmd);
	
	return cmd;
}

function execCmd(cmdarray, wdir, host) {
	execCmdAsync(cmdarray, wdir, host);
	// println("--> exitvalue: " + process.waitFor());
}

/**
 * Runs a native command in an external process and waits until the end of the
 * process.
 * 
 * @param cmdarray
 *            the command in a form of a javascript array
 * @param wdir
 *            the instance of java.io.File that defines the working dir
 */
function execCmdAsync(cmdarray, wdir, host) {
	var jarray;

	if (host == "localhost") {
		jarray = toJavaArray(cmdarray);
	} else {
		var cmd = [];
		cmd = createCmdForRemoteExec(cmd, cmdarray, wdir, host);
		jarray = toJavaArray(cmd);
	}
	
	println("Running cmd: " + Arrays.toString(jarray));

	var pb = new ProcessBuilder(jarray);
	pb.redirectErrorStream(true);
	pb.directory(wdir);

	var process = pb.start();
	var reader = new BufferedReader(new InputStreamReader(process
			.getInputStream()));
	var line = null;
	while ((line = reader.readLine()) != null) {
		println("...: " + line);
	}

	return process;
}

function toJavaArray(cmdarray) {
	var jarray = java.lang.reflect.Array.newInstance(java.lang.String, cmdarray.length);
	for ( var i = 0; i < cmdarray.length; i++) {
		jarray[i] = cmdarray[i];
	}
	return jarray;
}

function waitForever() {
	var cmd = [ "sleep", "86400" ];
	execCmd(cmd, instDir, "localhost");
}

function createTmuxSessionIfNeeded() {
	if (!isWindows && new File(tmuxTerm).exists()) {
		var cmd;
		var jarray;
		var pb;

		// kill the old session, if it exists
		cmd = [ tmuxTerm, "kill-session", "-tsoceda" ];
		execCmd(cmd, instDir, "localhost");

		// create the new one
		cmd = [ tmuxTerm, "new-session", "-d", "-ssoceda", "" ];
		execCmd(cmd, instDir, "localhost");
	} else {
		println("Unable to use tmux. Will create several terminals instead...");
	}
}

function execCmdNewTermAsync(name, cmdarray, wdir, host) {
	var cmd;

	if (isWindows) {
		cmd = termWindows;
		cmd = cmd.concat('\"' + name + '\"', "/WAIT");
		cmd = cmd.concat(cmdarray);
	} else if (isMac) {
		if (new File(tmuxTerm).exists()) {
			cmd = [ tmuxTerm, "new-window", "-tsoceda" ];
			cmd = cmd.concat("-n", name.substring(0,10));
		} else {
			cmd = [ termMac, "-e" ];
		}
		
		if (host == "localhost") {
			cmd = cmd.concat(cmdarray.join(" "));
		} else {
			cmd = createCmdForRemoteExec(cmd, cmdarray, wdir, host);
		}
	} else {
		if (new File(tmuxTerm).exists()) {
			cmd = [ tmuxTerm, "new-window", "-tsoceda" ];
			cmd = cmd.concat("-n", name.substring(0,10));
		} else if (new File(xTerm).exists()) {
			cmd = [ xTerm, "-e" ];
		} else if (new File(gnomeTerm).exists()) {
			cmd = [ gnomeTerm, "-e" ];
		} else if (new File(kdeTerm).exists()) {
			cmd = [ kdeTerm, "-e" ];
		} else {
			throw "No avalaible terminal, please be sure to have one of the following: "
					+ tmuxTerm + ", " + xTerm + ", " + gnomeTerm + ", " + kdeTerm;
		}
		
		if (host == "localhost") {
			cmd = cmd.concat(cmdarray.join(" "));
		} else {
			cmd = createCmdForRemoteExec(cmd, cmdarray, wdir, host);
		}
	}

	var jarray = toJavaArray(cmd);
	println("Running cmd: " + Arrays.toString(jarray));

	var pb = new ProcessBuilder(jarray);
	pb.directory(wdir);

	return pb.start();
}

function copy(sourceFile, destinationFile) {
	if (sourceFile.isDirectory()) {
		copyDirectory(sourceFile, destinationFile);
	} else {
		copyFile(sourceFile, destinationFile);
	}
}

function copyDirectory(sourceFile, destinationFile) {
	if (!sourceFile.isDirectory()) {
		throw "Source (" + sourceFile.getPath() + ") must be a directory.";
	}
	if (!sourceFile.exists()) {
		throw "Source directory (" + sourceFile.getPath() + ") doesn't exist.";
	}
	if (destinationFile.exists()) {
		throw "Destination (" + destinationFile.getPath() + ") exists.";
	}
	destinationFile.mkdirs();
	var files = sourceFile.listFiles();
	for ( var i = 0; i < files.length; i++) {
		var file = files[i];
		if (file.isDirectory()) {
			copyDirectory(file, new File(destinationFile, file.getName()));
		} else {
			copyFile(file, new File(destinationFile, file.getName()));
		}
	}
}

function copyFile(sourceFile, destinationFile) {
	var sourceChannel = new FileInputStream(sourceFile).getChannel();
	var targetChannel = new FileOutputStream(destinationFile).getChannel();
	sourceChannel.transferTo(0, sourceChannel.size(), targetChannel);
	sourceChannel.close();
	targetChannel.close();
}

function downloadFromHttp(urlString, destFile) {
	var ressource = new URL(urlString);
	var rbc = java.nio.channels.Channels.newChannel(ressource.openStream());
	var fos = new FileOutputStream(destFile);
	fos.getChannel().transferFrom(rbc, 0, 1 << 28);
}

function extractFolder(zipFile, destDirFile) { // throws ZipException,
	// IOException
	// println("---> UNCOMPRESSING : " + zipFile);

	var BUFFER = 2048;
	var zip = new ZipFile(zipFile);

	// Extract the name
	var filenameWithExt = zipFile.getName();
	var filenameWithoutExtension = filenameWithExt.substring(0, filenameWithExt
			.length() - 4);

	var zipFileEntries = zip.entries();

	// Process each entry
	while (zipFileEntries.hasMoreElements()) {
		// grab a zip file entry
		var entry = zipFileEntries.nextElement();
		var currentEntry = entry.getName();

		// print("... extracting ::: " + currentEntry);

		// If needed remove the internal filename if same as the archive
		if (currentEntry.startsWith(filenameWithoutExtension)) {
			currentEntry = currentEntry.substring(filenameWithoutExtension
					.length() + 1, currentEntry.length());
		}

		// The destination create the parent directory structure if needed
		var destFile = new File(destDirFile, currentEntry);
		var destinationParent = destFile.getParentFile();
		destinationParent.mkdirs();

		// println(" ::: to ::: " + destFile);

		if (!entry.isDirectory()) {
			var is = new BufferedInputStream(zip.getInputStream(entry));
			var currentByte;
			// establish buffer for writing file
			var data = java.lang.reflect.Array.newInstance(java.lang.Byte.TYPE,
					BUFFER);

			// write the current file to disk
			var fos = new FileOutputStream(destFile);
			var dest = new BufferedOutputStream(fos, BUFFER);

			// read and write until last byte is encountered
			while ((currentByte = is.read(data, 0, BUFFER)) != -1) {
				dest.write(data, 0, currentByte);
			}

			dest.flush();
			dest.close();
			is.close();
		}

		// if (currentEntry.endsWith(".zip")) {
		// // found a zip file, try to open
		// extractFolder(destFile.getAbsolutePath());
		// }
	}
}

function isPortAvailable(port) { // throws IOException
	var sock = null;
	try {
		sock = new java.net.ServerSocket(port);
	} catch (ee) {
		return false;
	} finally {
		if (sock != null) {
			try {
				sock.close();
			} catch (e) {
			}
		}
	}
	return true;
}

function tomcatAccessible(port) {
	var jarray = toJavaArray([ curlCmd, "http://" + cepedMachine + ":" + port ]);
	//println("Running cmd: " + Arrays.toString(jarray));

	var pb = new ProcessBuilder(jarray);
	pb.redirectErrorStream(true);
	pb.directory(instDir);

	var process = pb.start();
	
	var ret = process.waitFor();
	//println("--> exitvalue: " + ret); 
	return (ret == 0);
}

