importPackage(java.lang);
importPackage(java.io);
importPackage(java.util);
importPackage(java.util.zip);

// OS switch
var isWindows = System.getProperty("os.name").startsWith("Windows");
var isMac = System.getProperty("os.name").startsWith("Mac");

// OS independent absolute path to Java executable
var javaExe = System.getProperty("java.home") + File.separator + "bin"
		+ File.separator + (isWindows ? "java.exe" : "java");

// Contains all installed components
var instDir = new File("inst");
// Contains all components installers
var originDir = new File("origin");
// Contains configuration files that specifies hosts and ports
var configDir = new File("config");
// Contains deployment and actions scripts 
var scriptsDir = new File("scripts");

// To open a cmd in new terminal on Windows
var termWindows = [ "cmd.exe", "/c", "start" ];

// The Unix command prefix
var xTerm = "/usr/bin/xterm";
var gnomeTerm = "/usr/bin/gnome-terminal";
var kdeTerm = "/usr/bin/konsole";

var termMac = [ "/usr/bin/open", "-a", "Terminal" ];

/**
 * Check for correct dirs.
 */
function checkCurrentLocation() {
	// Perform several checks and run the installation if required
	// The dir that contains all installed components
	if (!instDir.exists()) {
		throw "Invalid location, unable to find " + instDir
				+ " please run this script in the correct dir";
	}

	// The dir that contains all components installers
	if (!originDir.exists()) {
		throw "Invalid location, unable to find " + originDir
				+ " please run this script in the correct dir";
	}
}

function execCmd(cmdarray, wdir) {
	execCmdAsync(cmdarray, wdir);
	//println("--> exitvalue: " + process.waitFor());
}

/**
 * Runs a native command in an external process and waits until the end of the
 * process.
 * 
 * @param cmdarray
 *            the command in a form of a javascript array
 * @param wdir
 *            the instance of java.io.File that defines the working dir
 */
function execCmdAsync(cmdarray, wdir) {

	var jarray = toJavaArray(cmdarray);
	println("Running cmd: " + Arrays.toString(jarray));

	var pb = new ProcessBuilder(jarray);
	pb.redirectErrorStream(true);
	pb.directory(wdir);

	var process = pb.start();
	var reader = new BufferedReader(new InputStreamReader(process
			.getInputStream()));
	var line = null;
	while ((line = reader.readLine()) != null) {
		println("...: " + line);
	}

	return process;
}

function toJavaArray(cmdarray) {
	var jarray = java.lang.reflect.Array.newInstance(java.lang.String, cmdarray.length);
	for ( var i = 0; i < cmdarray.length; i++) {
		jarray[i] = cmdarray[i];
	}
	return jarray;
}

function execCmdNewTermAsync(name, cmdarray, wdir) {
	var cmd;
	if (isWindows) {
		cmd = termWindows;
		cmd = cmd.concat('\"' + name + '\"', "/WAIT");
		cmd = cmd.concat(cmdarray);
	} else if (isMac) {
		cmd = termMac;
	} else {
		if (new File(xTerm).exists()) {
			cmd = [ xTerm ];
		} else if (new File(gnomeTerm).exists()) {
			cmd = [ gnomeTerm ];
		} else if (new File(kdeTerm).exists()) {
			cmd = [ kdeTerm ];
		} else {
			throw "No avalaible terminal, please be sure to have one of the following: "
					+ xTerm + ", " + gnomeTerm + ", " + kdeTerm;
		}
		cmd = cmd.concat("-e", cmdarray.join(" "));
	}

	var jarray = toJavaArray(cmd);
	//println("Running cmd: " + Arrays.toString(jarray));

	var pb = new ProcessBuilder(jarray);
	pb.directory(wdir);

	return pb.start();
}

function copy(sourceFile, destinationFile) {
	if (sourceFile.isDirectory()) {
		copyDirectory(sourceFile, destinationFile);
	} else {
		copyFile(sourceFile, destinationFile);
	}
}

function copyDirectory(sourceFile, destinationFile) {
	if (!sourceFile.isDirectory()) {
		throw "Source (" + sourceFile.getPath() + ") must be a directory.";
	}
	if (!sourceFile.exists()) {
		throw "Source directory (" + sourceFile.getPath() + ") doesn't exist.";
	}
	if (destinationFile.exists()) {
		throw "Destination (" + destinationFile.getPath() + ") exists.";
	}
	destinationFile.mkdirs();
	var files = sourceFile.listFiles();
	for ( var i = 0; i < files.length; i++) {
		var file = files[i];
		if (file.isDirectory()) {
			copyDirectory(file, new File(destinationFile, file.getName()));
		} else {
			copyFile(file, new File(destinationFile, file.getName()));
		}
	}
}

function copyFile(sourceFile, destinationFile) {
	var sourceChannel = new FileInputStream(sourceFile).getChannel();
	var targetChannel = new FileOutputStream(destinationFile).getChannel();
	sourceChannel.transferTo(0, sourceChannel.size(), targetChannel);
	sourceChannel.close();
	targetChannel.close();
}

function downloadFromHttp(urlString, destFile) {	
    var ressource = new URL(urlString);
    var rbc = java.nio.channels.Channels.newChannel(ressource.openStream());
    var fos = new FileOutputStream(destFile);
    fos.getChannel().transferFrom(rbc, 0, 1 << 28);
}

function extractFolder(zipFile, destDirFile){ // throws ZipException, IOException    
    var BUFFER = 2048;
    var zip = new ZipFile(zipFile);
    
    // Extract the name
    var filenameWithExt = zipFile.getName();   
    var filenameWithoutExtension = filenameWithExt.substring(0,filenameWithExt.length() - 4);
    
    var zipFileEntries = zip.entries();
    
    // Process each entry
    while (zipFileEntries.hasMoreElements())
    {    		
        // grab a zip file entry
        var entry = zipFileEntries.nextElement();
        var currentEntry = entry.getName();
        
        // println("... extracting ::: " + currentEntry);               
        
        // If needed remove the internal filename if same as the archive
        if (currentEntry.startsWith(filenameWithoutExtension)) {
        	currentEntry = currentEntry.substring(filenameWithoutExtension.length()+1, currentEntry.length());
        } else {
        	// Remove also the first sub dir if the filename starts with subdir name        	
        	var elts = currentEntry.split('/');
        	if (elts.length > 0) {        		
        		var firstSubDir = elts[0];
        		if (filenameWithoutExtension.startsWith(firstSubDir)) {
        			currentEntry = currentEntry.substring(firstSubDir.length()+1, currentEntry.length());
        		}        		
        	}
        		
        }
        
        // The destination create the parent directory structure if needed
        var destFile = new File(destDirFile, currentEntry);
        var destinationParent = destFile.getParentFile();
        destinationParent.mkdirs();
        
        //println(" ::: to ::: " + destFile);        
                
        if (!entry.isDirectory()) {
            var is = new BufferedInputStream(zip.getInputStream(entry));
            var currentByte;
            // establish buffer for writing file
            var data = java.lang.reflect.Array.newInstance(java.lang.Byte.TYPE, BUFFER);

            // write the current file to disk
            var fos = new FileOutputStream(destFile);
            var dest = new BufferedOutputStream(fos, BUFFER);

            // read and write until last byte is encountered
            while ((currentByte = is.read(data, 0, BUFFER)) != -1) {
                dest.write(data, 0, currentByte);
            }
            
            dest.flush();
            dest.close();
            is.close();
        }

        //if (currentEntry.endsWith(".zip")) {
        //    // found a zip file, try to open
        //    extractFolder(destFile.getAbsolutePath());
        //}
    }  
}

function isPortAvailable(port) { // throws IOException
	var sock = null;
	try {
		sock = new java.net.ServerSocket(port);
	} catch (ee) {
		return false;
	} finally {
		if (sock!=null) {
			try {
			sock.close();
			} catch (e){} 
		}
	}
	return true;
}

function deleteLockedFile(binDir) {
	// Check for locked file in bin dir
	var lockedFile = new File(binDir, "locked");
	if  (lockedFile.exists()) {
		lockedFile['delete']();
	}
}

/**
 * Prompts the user for input and reads from standard input (stdin).
 * Note: Always check if the return is null!
 * 
 * @param description Describes the user input.
 * @return A String of the input, or null when failed.
 */
function readStdin(description) {
    println(description + ": ");
    var input = null;

    var /*InputStreamReader*/ stream;
    var /*BufferedReader*/ reader;
    try {
        // Open a stream to stdin
        stream = new InputStreamReader(System["in"]);

        // Create a buffered reader to stdin
        reader = new BufferedReader(stream);

        // Try to read the string
        input = reader.readLine();           
    } catch (e) {
        println("Unable to get input due to " + e);
    }
    return input;
}


function findDownloadComponentURL(componentKey) {
	var fr = new java.io.FileReader(new File("./downloadWSs.properties"));
	var br = new BufferedReader(fr);
	var line;
	var componentUrl;
	while ((line=br.readLine())!=null){
		if(line.startsWith(componentKey + " ")) {
			componentUrl = line.substring(line.indexOf("=")+1, line.length()).trim();
			if(componentUrl.contains("1.0-SNAPSHOT")) {
				
				if(componentUrl.startsWith("http://research-distributions.petalslink.org/projects/soceda/")) {
					var shortNameWithVersion = componentUrl.substring(componentUrl.lastIndexOf("/"), componentUrl.length());
					var shortNameWithoutVersion = shortNameWithVersion.substring(0, shortNameWithVersion.lastIndexOf("-1.0-SNAPSHOT"));
					componentUrl = componentUrl.substring(0, componentUrl.lastIndexOf("/"));
					componentUrl = componentUrl.replace("http://research-distributions.petalslink.org/projects/soceda/", "");
					componentUrl = componentUrl.replace("/", "/target/");
					componentUrl = componentUrl.replace(".", "/");
					var path = findPropertyInInstaller("svn.path");
					var newPath = path.replace("projects\\soceda\\wp5-usecases\\uc1-crisis-management\\crisis-management-installer", "");
					componentUrl = "file:///" + newPath + componentUrl + shortNameWithoutVersion + "/target" + shortNameWithVersion; 
				} else if(componentUrl.startsWith("http://research-distributions.petalslink.org/experimental/")) {
					var shortNameWithVersion = componentUrl.substring(componentUrl.lastIndexOf("/"), componentUrl.length());
					var shortNameWithoutVersion = shortNameWithVersion.substring(0, shortNameWithVersion.lastIndexOf("-1.0-SNAPSHOT"));
					componentUrl = componentUrl.substring(0, componentUrl.lastIndexOf("/"));
					componentUrl = componentUrl.replace("http://research-distributions.petalslink.org/experimental/", "");
					componentUrl = componentUrl.replace("/", "/target/");
					componentUrl = componentUrl.replace(".", "/");
					var path = findPropertyInInstaller("svn.path");
					var newPath = path.replace("projects\\soceda\\wp5-usecases\\uc1-crisis-management\\crisis-management-installer", "");
					newPath = newPath + "/dev/experimental/";
					componentUrl = "file:///" + newPath + componentUrl + shortNameWithoutVersion + "/target" + shortNameWithVersion; 
				} 
			}
			println("component url found: " + componentUrl);
			
		}
	}

	br.close();
	fr.close();
	
	return componentUrl;
}

function findInstallerNameOfComponent(componentKey) {
	var fr = new java.io.FileReader(new File("./downloadWSs.properties"));
	var br = new BufferedReader(fr);
	var line;
	var componentUrl;
	while ((line=br.readLine())!=null){
		if(line.startsWith(componentKey + " ")) {
			println("component installer name found: " + line.substring(line.lastIndexOf("/")+1, line.length()).trim());
			componentUrl = line.substring(line.lastIndexOf("/")+1, line.length()).trim();
		}
	}

	br.close();
	fr.close();
	
	return componentUrl;
}

function findPropertyInInstaller(property) {
	var fr = new java.io.FileReader(new File("./src/installer-version.properties"));
	var br = new BufferedReader(fr);
	var line;
	var property;
	while ((line=br.readLine())!=null){
		if(line.startsWith(property)) {
			property = line.substring(line.indexOf("=")+1, line.length()).trim();
		}
	}

	br.close();
	fr.close();
	
	return property;
}

