/****************************************************************************
 *
 * Copyright (c) 2008-2012, EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA 
 *
 *****************************************************************************/
 
package com.ebmwebsourcing.wsstar.addressing.definition.api;

import java.io.File;
import java.net.URI;
import java.util.Map;

import org.w3c.dom.Document;
import org.xml.sax.InputSource;

/**
 * This interface describes a collection of methods that enable conversion of a
 * SchemaImpl document (in XML, following the SchemaImpl parent described in the
 * SchemaImpl specification) into a SchemaImpl model.
 * 
 * @author Nicolas Salatge - eBM WebSourcing
 */
public interface WSAddressingReader {

    /**
     * Constants for the Message Exchange Patterns.
     * 
     */
    public enum FeatureConstants {
        VERBOSE("org.ow2.easywsdl.schema.verbose"), IMPORT_DOCUMENTS(
                "org.ow2.easywsdl.schema.importDocuments");

        private final String value;

        /**
         * Creates a new instance of {@link FeatureConstants}
         * 
         * @param value
         */
        private FeatureConstants(final String value) {
            this.value = value;
        }

        /**
         * 
         * @return
         */
        public String value() {
            return this.value;
        }

        /**
         * Please use this equals method instead of using :<code>
         * value().equals(value)
         * </code> which is
         * almost 10 times slower...
         * 
         * @param value
         * @return
         */
        public boolean equals(final String val) {
            return this.toString().equals(val);
        }

        /*
         * (non-Javadoc)
         * 
         * @see java.lang.Enum#toString()
         */
        @Override
        public String toString() {
            return this.value;
        }
    }

    /**
     * Sets the specified feature to the specified value.
     * <p>
     * The minimum features that must be supported are:
     * <p>
     * <table border=1>
     * <tr>
     * <th>Name</th>
     * <th>Description</th>
     * <th>Default Value</th>
     * </tr>
     * <tr>
     * <td><center>org.ow2.easywsdl.schema.verbose</center></td>
     * <td>If set to true, status messages will be displayed.</td>
     * <td><center>type: boolean - default value: false</center></td>
     * </tr>
     * <tr>
     * <td><center>org.ow2.easywsdl.schema.importDocuments</center></td>
     * <td>If set to true, imported WSDL documents will be retrieved and
     * processed.</td>
     * <td><center>type: boolean - default value: true</center></td>
     * </tr>
     * <tr>
     * <td>
     * <center>org.ow2.easywsdl.schema.pathDirectoryOfImportLocations
     * </center></td>
     * <td>If the location is set, imported WSDL documents will be retrieved at
     * this location (Set the importDocuments Features at true).</td>
     * <td><center>type: String</center></td>
     * </tr>
     * </table>
     * <p>
     * All feature names must be fully-qualified, Java package style. All names
     * starting with com.ebmwebsourcing. are reserved for features defined by
     * the specification. It is recommended that implementation- specific
     * features be fully-qualified to match the package name of that
     * implementation. For example: com.abc.featureName
     * 
     * @param name
     *            the name of the feature to be set.
     * @param value
     *            the value to set the feature to.
     * @throws IllegalArgumentException
     *             if the feature name is not recognized.
     * @see #getFeature(String)
     */
    public void setFeature(FeatureConstants name, Object value) throws IllegalArgumentException;

    /**
     * Gets the value of the specified feature.
     * 
     * @param name
     *            the name of the feature to get the value of.
     * @return the value of feature
     * @throws IllegalArgumentException
     *             if the feature name is not recognized.
     * @see #setFeature(String, boolean)
     */
    public Object getFeature(FeatureConstants name);

    /**
     * Gets all features.
     * 
     * @return the features
     * @see #setFeature(String, boolean)
     */
    public Map<FeatureConstants, Object> getFeatures();

    /**
     * Read the SchemaImpl document accessible via the specified URI into a
     * SchemaImpl definition.
     * 
     * @param schemaURI
     *            a URI (can be a filename or URL) pointing to a SchemaImpl XML
     *            definition.
     * @return the definition.
     */
    public EndpointReferenceType readEndpointReferenceType(URI uri) throws WSAddressingException;

    /**
     * Read the specified SchemaImpl document into a SchemaImpl definition.
     * 
     * @param documentBaseURI
     *            the document base URI of the SchemaImpl definition described
     *            by the document. Will be set as the documentBaseURI of the
     *            returned Definition. Can be null, in which case it will be
     *            ignored.
     * @param wsdlDocument
     *            the SchemaImpl document, an XML document obeying the
     *            SchemaImpl parent.
     * @return the definition described in the document.
     */
    public EndpointReferenceType readEndpointReferenceType(Document doc) throws WSAddressingException;

    /**
     * Read a SchemaImpl document into a SchemaImpl definition.
     * 
     * @param schemaURI
     *            the uri of the parent
     * @param inputSource
     *            an InputSource pointing to the SchemaImpl document, an XML
     *            document obeying the SchemaImpl parent.
     * @return the definition described in the document pointed to by the
     *         InputSource.
     */
    public EndpointReferenceType readEndpointReferenceType(URI uri, InputSource inputSource) throws WSAddressingException;
    
    /**
     * Read a EndpointReferanceType object from a file (used be persistence feature)
     * 
     * @param file the {@link File} object which the {@link EndpointReferenceType} object must be read from 
     * @return a {@link EndpointReferenceType} Java object instance
     * @throws WSAddressingException
     */
    public EndpointReferenceType readEndpointReferenceType(File file) throws WSAddressingException;

}
