package edu.uga.cs.lsdis.sawsdl.impl.extensions.schema;

import edu.uga.cs.lsdis.sawsdl.Definition;
import edu.uga.cs.lsdis.sawsdl.ModelReference;
import edu.uga.cs.lsdis.sawsdl.WSDLSException;
import edu.uga.cs.lsdis.sawsdl.extensions.schema.Schema;
import edu.uga.cs.lsdis.sawsdl.impl.Constants;
import edu.uga.cs.lsdis.sawsdl.impl.util.ModelRefUtil;
import edu.uga.cs.lsdis.sawsdl.impl.util.SchemaUtils;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;

import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 *
 */
public class SchemaImpl extends com.ibm.wsdl.extensions.schema.SchemaImpl implements Schema {

    /**
     * Get the modelReference on the element located by the given path.
     * @param startElement The starting element of the path
     * @param path Xpath expression representing the element
     * @return The modelReference
     */
    public ModelReference getModelReference(Element startElement, String path, Definition def)
            throws WSDLSException, URISyntaxException {
        Set<ModelReference> mrefs = getModelReferences(startElement, path, def);
        return mrefs.toArray(new ModelReference[mrefs.size()])[0];
    }

    /**
     * Set the modelReference on the element located by the given path.
     * @param startElement The element for the Xpath to start
     * @param path Xpath expression representing the element
     * @param modelReference The desired modelReference
     */
    public void addModelReference(Element startElement, String path, ModelReference modelReference)
            throws WSDLSException{
        Element el = getXSDElement(startElement, path);
        if (el!= null){
            if (modelReference != null){
                //tostring has been overidden to return the proper string for this URI
                String strModelReference = modelReference.toString();
                Attr attr = el.getAttributeNodeNS(
                        Constants.NS_URI_SAWSDL, Constants.ATTR_MODELREF);
                if(attr == null) {
                    attr = el.getOwnerDocument().createAttributeNS(Constants.NS_URI_SAWSDL, Constants.ATTR_MODELREF);
                    attr.setPrefix(Constants.PREFIX_SAWSDL);
                    el.setAttributeNodeNS(attr);
                }
                String value = attr.getValue();
                if(value != null) {
                    value += " " + strModelReference;
                } else {
                    value = strModelReference;
                }
                attr.setValue(value);
                modelReference.setParent(el);
            }
            else{
                el.removeAttributeNS(Constants.NS_URI_SAWSDL, Constants.ATTR_MODELREF);
            }
        }else{
            throw new WSDLSException("","Xpath did not result in an element");
        }
    }





    /**
     * Return an XML Schema Element by locating it from the XSD Element startElement and using the path.
     *
     * @param startElement Root element to start
     * @param path Xpath expression representing the element
     * @return an XML Schema Element
     * @throws WSDLSException  in any case of exceptional behavior
     */
    protected Element getXSDElement(Element startElement, String path) throws WSDLSException{
        return SchemaUtils.findXSDElement(startElement, path, this);
    }


    /**
     * Get the list of model references
     * @param startElement The element for the Xpath to start
     * @param path Xpath expression representing the element
     * @param def WSDL definition object
     * @return  a list of model references
     * @throws WSDLSException
     */
    public Set<ModelReference> getModelReferences(Element startElement, String path, Definition def)
            throws WSDLSException, URISyntaxException {
        Element el = getXSDElement(startElement, path);
        if (el == null)
            return null;
        String attrModelReferenceString = el.
                getAttributeNS(Constants.NS_URI_SAWSDL, Constants.ATTR_MODELREF).trim();

        //right here we have to do an extra check to do the mapping propogation
        Set<ModelReference> propogateModelRefs = null;
        if ((Constants.Q_SCEHMA_ELEMENT.getLocalPart().equals(el.getLocalName())
                || Constants.Q_SCEHMA_ATTRIBUTE.getLocalPart().equals(el.getLocalName()))
                && Constants.Q_SCEHMA_ELEMENT.getNamespaceURI().equals(el.getNamespaceURI())){
            propogateModelRefs =  getPropogatedReferences(el, def);
        }

        if (attrModelReferenceString.equals(""))
            return propogateModelRefs;

        Set<ModelReference> mrefs = ModelRefUtil.
                parseModelReference(attrModelReferenceString, def,el);

        if(mrefs.size() == 0)
            return propogateModelRefs;
        if (propogateModelRefs!= null) mrefs.addAll(propogateModelRefs);
        return mrefs;
    }

    /**
     * Get the propogated types
     * @param el
     * @param def
     * @return
     * @throws WSDLSException
     * @throws URISyntaxException
     */
    private Set<ModelReference> getPropogatedReferences(Element el, Definition def) throws WSDLSException, URISyntaxException {
        javax.wsdl.extensions.schema.Schema schema = getRelevantSchema(el,def);
        if (schema==null){
            // the schema  is not found or it is a schema we want to ignore
            return null;

        }else{
            String typeAttributeString = el.getAttribute(Constants.ATTR_TYPE);
            String typeName = null;
            if (typeAttributeString.indexOf(':')>0){
                typeName = typeAttributeString.split(":")[1];
            }else{
                //the type is in the default namespace.assign the type name directly
                typeName = typeAttributeString;
            }

            //first look for simple types
            Set<ModelReference> modelRefs =
                    getModelReferences(schema.getElement(),"//xsd:simpleType[@name=\"" + typeName+"\"]",def);
            if (modelRefs==null){
                //try with complex types
                modelRefs =
                        getModelReferences(schema.getElement(),"//xsd:complexType[@name=\"" + typeName+"\"]",def);
            }

            return modelRefs;

        }
    }

    /**
     * Get the propogated types
     * @param el
     * @param def
     * @return
     * @throws WSDLSException
     * @throws URISyntaxException
     */
    private Set<String> getPropogatedMapping(Element el, Definition def,String type) throws WSDLSException, URISyntaxException {
        javax.wsdl.extensions.schema.Schema schema = getRelevantSchema(el,def);
        if (schema==null){
            // the schema  is not found or it is a schema we want to ignore
            return null;

        }else{
            String typeAttributeString = el.getAttribute(Constants.ATTR_TYPE).trim();
            String typeName = null;
            if (typeAttributeString.indexOf(':')>0){
                typeName = typeAttributeString.split(":")[1];
            }else{
                //the type is in the default namespace.assign the type name directly
                typeName = typeAttributeString;
            }

            //first look for simple types
            Set<String> modelRefs =
                    getAttValueAsSet(schema.getElement(),"//xsd:simpleType[@name=\"" + typeName+"\"]",type,def);
            if (modelRefs==null){
                //try with complex types
                modelRefs =
                        getAttValueAsSet(schema.getElement(),"//xsd:complexType[@name=\"" + typeName+"\"]",type,def);
            }

            return modelRefs;

        }
    }
    /**
     *
     * @param el
     * @param def
     * @return
     * @throws WSDLSException
     * @throws URISyntaxException
     */
    private javax.wsdl.extensions.schema.Schema getRelevantSchema(Element el, Definition def) throws WSDLSException, URISyntaxException {
        //propagate the type model references
        String typeAttributeString = el.getAttribute(Constants.ATTR_TYPE).trim();
        if ("".equals(typeAttributeString)){
            // the type is not specified - hence the type would be an anonymous simple/complexType
            // do we need to handle the case where the annotations are added into the anonymous
            // type ??
            return null;

        }else{
            //if the type reference is prefixed then we may need to look for this reference
            //in other schema objects than this one. pick the right namespace uri for the schema
            //we need to use
            String typeSchemaNamespace = null;
            if (typeAttributeString.indexOf(':')>0){
                String nsprefix = typeAttributeString.split(":")[0];
                typeSchemaNamespace = (String) def.getNamespaces().get(nsprefix);
            }else{
                //the type is in the default namespace.get the default namespace for this
                //schema
                // the lookup prefix of null should return the default namespace for this schema
                typeSchemaNamespace = el.getOwnerDocument().lookupNamespaceURI(null);
            }
            if (typeSchemaNamespace==null || "".equals(typeSchemaNamespace)){
                throw new WSDLSException("","type namespace prefix did not match any known schema" +
                        " namespace");
                //should we just log this and let this pass ???
            }

            //ignore the types with the schema namespace
            if (Constants.NS_URI_2001_SCHEMA.equals(typeSchemaNamespace)) return null;

            //find the schema that we need to use
            List schemaList = def.getTypes().getExtensibilityElements();
            javax.wsdl.extensions.schema.Schema schema = null;
            for (Object aSchemaList : schemaList) {
                schema =
                        (javax.wsdl.extensions.schema.Schema)aSchemaList;
                String tns = schema.getElement().getAttribute(Constants.ATTR_TARGET_NAMESPACE);
                if (tns.equals(typeSchemaNamespace)){
                    break;
                }

            }
            if (schema==null){
                throw new WSDLSException("","could not find any schema object for the" +
                        " namespace" + typeSchemaNamespace);
            }


            return schema;

        }
    }

    /**
     *
     * @param startElement The element for the Xpath to start
     * @param path Xpath expression representing the element
     * @param refs a list of model references
     * @throws WSDLSException
     */
    public void setModelReferences(Element startElement, String path, Set<ModelReference> refs) throws WSDLSException {
        for(ModelReference ref : refs) {
            addModelReference(startElement, path, ref);
        }
    }

    /**
     * add a single lifting schema mapping on the element located by the given path.
     * if there is already a mapping this mapping will be added to the list
     *
     * @param startElement   The starting element of the path
     * @param path           (a valid xpath expression)
     * @param liftingMapping The desired lifting mapping
     * @param def
     */
    public void addLiftingMapping(Element startElement, String path, String liftingMapping, Definition def)
            throws WSDLSException, URISyntaxException {
        Set<String> liftingMappings = getLiftingMappings(startElement, path, def);
        if (liftingMappings !=null && !liftingMappings.isEmpty()){
            liftingMappings.add(liftingMapping);
        }else{
            Set<String> tempSet = new HashSet<String>();
            tempSet.add(liftingMapping);
            setLiftingMappings(startElement,path,tempSet);
        }

    }

    /**
     * Get the lifting schema mappings pertaining to the schema construct given by the
     * xpath (path)
     *
     * @param startElement Root of the XPath
     * @param path         XPath expression that returns a schema construct
     * @param def
     * @return  a list of  strings
     * @throws edu.uga.cs.lsdis.sawsdl.WSDLSException
     *
     */
    public Set<String> getLiftingMappings(Element startElement, String path, Definition def) throws WSDLSException, URISyntaxException {
        return getAttValueAsSet(startElement,path,Constants.ATTR_LIFTING_MAPPING,def);
    }



    /**
     * Set the lifting schema mappings  to the element (type) given by the Xpath (path)
     *
     * @param startElement The element for the Xpath to start
     * @param path     Xpath expression representing the element
     * @param mappings  list of strings
     * @throws edu.uga.cs.lsdis.sawsdl.WSDLSException
     *
     */
    public void setLiftingMappings(Element startElement, String path, Set<String> mappings) throws WSDLSException {
        setAttValueAsSet(startElement,path,Constants.ATTR_LIFTING_MAPPING,mappings);
    }

    /**
     * add a single lifting schema mapping on the element located by the given path.
     * if there is already a mapping this mapping will be added to the list
     *
     * @param startElement   The starting element of the path
     * @param path           (a valid xpath expression)
     * @param loweringMapping The desired lifting mapping
     * @param def
     */
    public void addLoweringMapping(Element startElement, String path, String loweringMapping, Definition def) throws WSDLSException, URISyntaxException {
        Set<String> loweringMappings = getLoweringMappings(startElement, path, def);
        if (loweringMappings!=null && !loweringMappings.isEmpty()){
            loweringMappings.add(loweringMapping);
        }else{
            Set<String> tempSet = new HashSet<String>();
            tempSet.add(loweringMapping);
            setLoweringMappings(startElement,path,tempSet);
        }

    }

    /**
     * Get the lowering schema mappings pertaining to the schema construct given by the
     * xpath (path)
     *
     * @param startElement Root of the XPath
     * @param path         XPath expression that returns a schema construct
     * @param def
     * @return list of strings
     * @throws edu.uga.cs.lsdis.sawsdl.WSDLSException
     *
     */
    public Set<String> getLoweringMappings(Element startElement, String path, Definition def) throws WSDLSException, URISyntaxException {
        return getAttValueAsSet(startElement,path,Constants.ATTR_LOWERING_MAPPING,def);
    }

    /**
     * Set the lowering schema mappings  to the element (type) given by the Xpath (path)
     * This operation will *remove* any existing schema mappings
     *
     * @param startElement The element for the Xpath to start
     * @param path         (a valid xpath expression)
     * @param mappings list of mappings as strings
     * @throws edu.uga.cs.lsdis.sawsdl.WSDLSException
     *
     */
    public void setLoweringMappings(Element startElement, String path, Set<String> mappings) throws WSDLSException {
        setAttValueAsSet(startElement,path,Constants.ATTR_LOWERING_MAPPING,mappings);
    }




    /**
     * Convenience method that is shared between get methods only for lifting and lowering mappings though
     * @param startElement  The element for the Xpath to start
     * @param path Xpath expression representing the element
     * @param attNameToGet name of the attribute to extract
     * @return  a list of strings
     * @throws WSDLSException in any case of exceptional behavior
     */
    protected Set<String> getAttValueAsSet(Element startElement,
                                           String path,
                                           String attNameToGet,
                                           Definition def) throws WSDLSException, URISyntaxException {
        Element el = getXSDElement(startElement, path);
        if (el == null){
            return null;  // nothing to return! specified element cannot be found!
        }
        String mappingString = el.getAttributeNS(Constants.NS_URI_SAWSDL, attNameToGet).trim();
        if (!el.hasAttributeNS(Constants.NS_URI_SAWSDL, attNameToGet)) {
            //look for a propogated mapping
            Set<String> propogateModelRefs = null;
            if (Constants.Q_SCEHMA_ELEMENT.getLocalPart().equals(el.getLocalName())
                    && Constants.Q_SCEHMA_ELEMENT.getNamespaceURI().equals(el.getNamespaceURI())){
                propogateModelRefs =  getPropogatedMapping(el, def,attNameToGet);
            }


            return propogateModelRefs;
        }

        if ("".equals(mappingString)) {
            return null;
        }

        String[] mappings = ModelRefUtil.splitString(mappingString);
        if(mappings.length == 0){
            return null;
        }
        return new HashSet<String>(Arrays.asList(mappings));
    }

    /**
     * Convenience method that is shared between set methods
     * @param startElement The element for the Xpath to start
     * @param path Xpath expression representing the element
     * @param attNameToSet  name of the attribute to set the value
     * @param values A list of string values to set as the values
     * @return a boolean indicating the success/failure of the operation
     * @throws WSDLSException  in any case of exceptional behavior
     */
    protected boolean setAttValueAsSet(Element startElement,
                                       String path,
                                       String attNameToSet,
                                       Set<String> values) throws WSDLSException {

        Element el = getXSDElement(startElement, path);
        if (el == null){
            return false;
        }

        String valueString = "";
        for (String s : values) {
            valueString += " " + s;
        }

        el.setAttributeNS(Constants.NS_URI_SAWSDL, attNameToSet,valueString);
        return true;
    }
}
