/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.dnsservice.dnsjava;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import javax.annotation.PostConstruct;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.configuration.HierarchicalConfiguration;
import org.apache.james.dnsservice.api.DNSService;
import org.apache.james.dnsservice.api.DNSServiceMBean;
import org.apache.james.dnsservice.api.TemporaryResolutionException;
import org.apache.james.lifecycle.api.Configurable;
import org.apache.james.lifecycle.api.LogEnabled;
import org.slf4j.Logger;
import org.xbill.DNS.ARecord;
import org.xbill.DNS.Address;
import org.xbill.DNS.Cache;
import org.xbill.DNS.ExtendedResolver;
import org.xbill.DNS.Lookup;
import org.xbill.DNS.MXRecord;
import org.xbill.DNS.Name;
import org.xbill.DNS.PTRRecord;
import org.xbill.DNS.Record;
import org.xbill.DNS.Resolver;
import org.xbill.DNS.ResolverConfig;
import org.xbill.DNS.ReverseMap;
import org.xbill.DNS.TXTRecord;
import org.xbill.DNS.TextParseException;

public class DNSJavaService
implements DNSService,
DNSServiceMBean,
LogEnabled,
Configurable {
    protected Resolver resolver;
    protected Cache cache;
    private int maxCacheSize = 50000;
    private int dnsCredibility;
    private List<String> dnsServers = new ArrayList<String>();
    private Name[] searchPaths = null;
    private Comparator<MXRecord> mxComparator = new MXRecordComparator();
    private boolean setAsDNSJavaDefault;
    private String localHostName;
    private String localCanonicalHostName;
    private String localAddress;
    private Logger logger;

    public void setLog(Logger logger) {
        this.logger = logger;
    }

    public void configure(HierarchicalConfiguration configuration) throws ConfigurationException {
        boolean authoritative;
        boolean autodiscover = configuration.getBoolean("autodiscover", true);
        ArrayList<Name> sPaths = new ArrayList<Name>();
        if (autodiscover) {
            Name[] systemSearchPath;
            this.logger.info("Autodiscovery is enabled - trying to discover your system's DNS Servers");
            String[] serversArray = ResolverConfig.getCurrentConfig().servers();
            if (serversArray != null) {
                for (int i = 0; i < serversArray.length; ++i) {
                    this.dnsServers.add(serversArray[i]);
                    this.logger.info("Adding autodiscovered server " + serversArray[i]);
                }
            }
            if ((systemSearchPath = ResolverConfig.getCurrentConfig().searchPath()) != null && systemSearchPath.length > 0) {
                sPaths.addAll(Arrays.asList(systemSearchPath));
            }
            if (this.logger.isInfoEnabled()) {
                for (Name searchPath : sPaths) {
                    this.logger.info("Adding autodiscovered search path " + searchPath.toString());
                }
            }
        }
        this.setAsDNSJavaDefault = configuration.getBoolean("setAsDNSJavaDefault", true);
        List serversConfigurations = configuration.getList("servers.server");
        for (int i = 0; i < serversConfigurations.size(); ++i) {
            this.dnsServers.add((String)serversConfigurations.get(i));
        }
        List searchPathsConfiguration = configuration.getList("searchpaths.searchpath");
        for (int i = 0; i < searchPathsConfiguration.size(); ++i) {
            try {
                sPaths.add(Name.fromString((String)((String)searchPathsConfiguration.get(i))));
                continue;
            }
            catch (TextParseException e) {
                throw new ConfigurationException("Unable to parse searchpath host: " + (String)searchPathsConfiguration.get(i), (Throwable)e);
            }
        }
        this.searchPaths = sPaths.toArray(new Name[0]);
        if (this.dnsServers.isEmpty()) {
            this.logger.info("No DNS servers have been specified or found by autodiscovery - adding 127.0.0.1");
            this.dnsServers.add("127.0.0.1");
        }
        this.dnsCredibility = (authoritative = configuration.getBoolean("authoritative", false)) ? 4 : 3;
        this.maxCacheSize = configuration.getInt("maxcachesize", this.maxCacheSize);
    }

    @PostConstruct
    public void init() throws Exception {
        this.logger.debug("DNSService init...");
        if (this.dnsServers.isEmpty()) {
            try {
                this.dnsServers.add(InetAddress.getLocalHost().getHostName());
            }
            catch (UnknownHostException ue) {
                this.dnsServers.add("127.0.0.1");
            }
        }
        String[] serversArray = this.dnsServers.toArray(new String[0]);
        if (this.logger.isInfoEnabled()) {
            for (int c = 0; c < serversArray.length; ++c) {
                this.logger.info("DNS Server is: " + serversArray[c]);
            }
        }
        try {
            this.resolver = new ExtendedResolver(serversArray);
        }
        catch (UnknownHostException uhe) {
            this.logger.error("DNS service could not be initialized.  The DNS servers specified are not recognized hosts.", (Throwable)uhe);
            throw uhe;
        }
        this.cache = new Cache(1);
        this.cache.setMaxEntries(this.maxCacheSize);
        if (this.setAsDNSJavaDefault) {
            Lookup.setDefaultResolver((Resolver)this.resolver);
            Lookup.setDefaultCache((Cache)this.cache, (int)1);
            Lookup.setDefaultSearchPath((Name[])this.searchPaths);
            this.logger.info("Registered cache, resolver and search paths as DNSJava defaults");
        }
        InetAddress addr = this.getLocalHost();
        this.localCanonicalHostName = addr.getCanonicalHostName();
        this.localHostName = addr.getHostName();
        this.localAddress = addr.getHostAddress();
        this.logger.debug("DNSService ...init end");
    }

    public String[] getDNSServers() {
        return this.dnsServers.toArray(new String[0]);
    }

    public Name[] getSearchPaths() {
        return this.searchPaths;
    }

    private List<String> findMXRecordsRaw(String hostname) throws TemporaryResolutionException {
        Record[] answers = this.lookup(hostname, 15, "MX");
        ArrayList<String> servers = new ArrayList<String>();
        if (answers == null) {
            return servers;
        }
        MXRecord[] mxAnswers = new MXRecord[answers.length];
        for (int i = 0; i < answers.length; ++i) {
            mxAnswers[i] = (MXRecord)answers[i];
        }
        Arrays.sort(mxAnswers, this.mxComparator);
        int currentPrio = -1;
        ArrayList<String> samePrio = new ArrayList<String>();
        for (int i = 0; i < mxAnswers.length; ++i) {
            boolean same = false;
            boolean lastItem = i + 1 == mxAnswers.length;
            MXRecord mx = mxAnswers[i];
            if (i == 0) {
                currentPrio = mx.getPriority();
            } else {
                same = currentPrio == mx.getPriority();
            }
            String mxRecord = mx.getTarget().toString();
            if (same) {
                samePrio.add(mxRecord);
            } else {
                Collections.shuffle(samePrio);
                servers.addAll(samePrio);
                samePrio.clear();
                samePrio.add(mxRecord);
            }
            if (lastItem) {
                Collections.shuffle(samePrio);
                servers.addAll(samePrio);
            }
            this.logger.debug(new StringBuffer("Found MX record ").append(mxRecord).toString());
        }
        return servers;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Collection<String> findMXRecords(String hostname) throws TemporaryResolutionException {
        List<Object> servers = new ArrayList();
        try {
            servers = this.findMXRecordsRaw(hostname);
            Collection<String> collection = Collections.unmodifiableCollection(servers);
            return collection;
        }
        finally {
            if (servers.size() == 0) {
                StringBuffer logBuffer = new StringBuffer(128).append("Couldn't resolve MX records for domain ").append(hostname).append(".");
                this.logger.info(logBuffer.toString());
                try {
                    this.getByName(hostname);
                    servers.add(hostname);
                }
                catch (UnknownHostException uhe) {
                    logBuffer = new StringBuffer(128).append("Couldn't resolve IP address for host ").append(hostname).append(".");
                    this.logger.error(logBuffer.toString());
                }
            }
        }
    }

    protected Record[] lookup(String namestr, int type, String typeDesc) throws TemporaryResolutionException {
        try {
            Lookup l = new Lookup(namestr, type);
            l.setCache(this.cache);
            l.setResolver(this.resolver);
            l.setCredibility(this.dnsCredibility);
            l.setSearchPath(this.searchPaths);
            Record[] r = l.run();
            try {
                if (l.getResult() == 2) {
                    throw new TemporaryResolutionException("DNSService is temporary not reachable");
                }
                return r;
            }
            catch (IllegalStateException ise) {
                this.logger.debug("Error determining result ", (Throwable)ise);
                throw new TemporaryResolutionException("DNSService is temporary not reachable");
            }
        }
        catch (TextParseException tpe) {
            this.logger.error("Couldn't parse name " + namestr, (Throwable)tpe);
            return null;
        }
    }

    protected Record[] lookupNoException(String namestr, int type, String typeDesc) {
        try {
            return this.lookup(namestr, type, typeDesc);
        }
        catch (TemporaryResolutionException e) {
            return null;
        }
    }

    private static String allowIPLiteral(String host) {
        String possible_ip_literal;
        if (host.charAt(host.length() - 1) == '.' && Address.isDottedQuad((String)(possible_ip_literal = host.substring(0, host.length() - 1)))) {
            host = possible_ip_literal;
        }
        return host;
    }

    public InetAddress getByName(String host) throws UnknownHostException {
        String name = DNSJavaService.allowIPLiteral(host);
        try {
            if (name.equalsIgnoreCase(this.localHostName) || name.equalsIgnoreCase(this.localCanonicalHostName) || name.equals(this.localAddress)) {
                return this.getLocalHost();
            }
            return Address.getByAddress((String)name);
        }
        catch (UnknownHostException e) {
            Record[] records = this.lookupNoException(name, 1, "A");
            if (records != null && records.length >= 1) {
                ARecord a = (ARecord)records[0];
                return InetAddress.getByAddress(name, a.getAddress().getAddress());
            }
            throw e;
        }
    }

    public InetAddress[] getAllByName(String host) throws UnknownHostException {
        String name = DNSJavaService.allowIPLiteral(host);
        try {
            if (name.equalsIgnoreCase(this.localHostName) || name.equalsIgnoreCase(this.localCanonicalHostName) || name.equals(this.localAddress)) {
                return new InetAddress[]{this.getLocalHost()};
            }
            InetAddress addr = Address.getByAddress((String)name);
            return new InetAddress[]{addr};
        }
        catch (UnknownHostException e) {
            Record[] records = this.lookupNoException(name, 1, "A");
            if (records != null && records.length >= 1) {
                InetAddress[] addrs = new InetAddress[records.length];
                for (int i = 0; i < records.length; ++i) {
                    ARecord a = (ARecord)records[i];
                    addrs[i] = InetAddress.getByAddress(name, a.getAddress().getAddress());
                }
                return addrs;
            }
            throw e;
        }
    }

    public Collection<String> findTXTRecords(String hostname) {
        ArrayList<String> txtR = new ArrayList<String>();
        Record[] records = this.lookupNoException(hostname, 16, "TXT");
        if (records != null) {
            for (int i = 0; i < records.length; ++i) {
                TXTRecord txt = (TXTRecord)records[i];
                txtR.add(txt.rdataToString());
            }
        }
        return txtR;
    }

    public String getHostName(InetAddress addr) {
        String result = null;
        Name name = ReverseMap.fromAddress((InetAddress)addr);
        Record[] records = this.lookupNoException(name.toString(), 12, "PTR");
        if (records == null) {
            result = addr.getHostAddress();
        } else {
            PTRRecord ptr = (PTRRecord)records[0];
            result = ptr.getTarget().toString();
        }
        return result;
    }

    public InetAddress getLocalHost() throws UnknownHostException {
        return InetAddress.getLocalHost();
    }

    public int getMaximumCacheSize() {
        return this.maxCacheSize;
    }

    public int getCurrentCacheSize() {
        return this.cache.getSize();
    }

    public void clearCache() {
        this.cache.clearCache();
    }

    private static class MXRecordComparator
    implements Comparator<MXRecord> {
        private MXRecordComparator() {
        }

        @Override
        public int compare(MXRecord a, MXRecord b) {
            int pa = a.getPriority();
            int pb = b.getPriority();
            return pa - pb;
        }
    }
}

