
package org.ow2.petals.flowwatch.flowmanager;

import java.util.Date;
import java.util.List;

import org.hibernate.Session;
import org.hibernate.Transaction;
import org.ow2.petals.flowwatch.flowmanager.bo.Flow;
import org.ow2.petals.flowwatch.flowmanager.bo.FlowStep;
import org.ow2.petals.flowwatch.flowmanager.bo.FlowStepRef;
import org.ow2.petals.flowwatch.flowmanager.dao.FlowDAO;
import org.ow2.petals.flowwatch.flowmanager.dao.FlowStepDAO;
import org.ow2.petals.flowwatch.flowmanager.dao.FlowStepRefDAO;

public class FlowManager extends ConfigAndFactoryManager {

    private static FlowManager instance;

    private static FlowDAO flowDAO = new FlowDAO();

    private static FlowStepDAO flowStepDAO = new FlowStepDAO();

    private static FlowStepRefDAO flowStepRefDAO = FlowStepRefDAO.getInstance();

    static {
        instance = new FlowManager();
    }

    public FlowManager() {
        super();
    }

    public static FlowManager getInstance() {
        return instance;
    }

    public Flow load(final String id) {
        final Session session = this.configAndFactory.getFactory().getCurrentSession();
        final Transaction tx = session.beginTransaction();
        try {
            final Flow flow = flowDAO.loadByPetalsId(id);
            if (flow != null) {
                this.updateFlowStatusAndDates(flow);
            }
            tx.commit();
            return flow;
        } catch (final RuntimeException e) {
            this.configAndFactory.getFactory().getCurrentSession().getTransaction().rollback();
            throw e;
        }

    }

    public void remove(final String id) {
        final Session session = this.configAndFactory.getFactory().getCurrentSession();
        final Transaction tx = session.beginTransaction();
        try {
            flowDAO.remove(flowDAO.loadByPetalsId(id));
            tx.commit();
        } catch (final RuntimeException e) {
            this.configAndFactory.getFactory().getCurrentSession().getTransaction().rollback();
            throw e;
        }
    }

    public void removeStep(final String flowId, final String meUUID, final String interfaceName,
            final String serviceName, final String endpointName) {
        final Session session = this.configAndFactory.getFactory().getCurrentSession();
        final Transaction tx = session.beginTransaction();
        try {
            final FlowStep flowStep = flowStepDAO.loadByMeUUID(meUUID);
            flowStepDAO.remove(flowStep);
            tx.commit();
        } catch (final RuntimeException e) {
            this.configAndFactory.getFactory().getCurrentSession().getTransaction().rollback();
            throw e;
        }
    }

    public String saveOrUpdate(final Flow flow) {
        final Session session = this.configAndFactory.getFactory().getCurrentSession();
        final Transaction tx = session.beginTransaction();
        try {
            flowDAO.save(flow);
            final String result = flow.getIdpetals();
            tx.commit();
            return result;
        } catch (final RuntimeException e) {
            this.configAndFactory.getFactory().getCurrentSession().getTransaction().rollback();
            throw e;
        }
    }

    public List<FlowStep> loadSteps(final String id) {
        final Session session = this.configAndFactory.getFactory().getCurrentSession();
        final Transaction tx = session.beginTransaction();
        try {
            final List<FlowStep> result = flowStepDAO.loadByFlowId(id);
            tx.commit();
            return result;
        } catch (final RuntimeException e) {
            this.configAndFactory.getFactory().getCurrentSession().getTransaction().rollback();
            throw e;
        }
    }

    public List<FlowStep> loadRunningSteps(final String flowId) {
        final Session session = this.configAndFactory.getFactory().getCurrentSession();
        final Transaction tx = session.beginTransaction();
        try {
            final List<FlowStep> result = flowStepDAO.loadRunningByFlowId(flowId);
            tx.commit();
            return result;
        } catch (final RuntimeException e) {
            this.configAndFactory.getFactory().getCurrentSession().getTransaction().rollback();
            throw e;
        }
    }

    public FlowStep loadFlowStep(final String flowId, final String meUUID,
            final String interfaceName, final String serviceName, final String endpointName) {
        final Session session = this.configAndFactory.getFactory().getCurrentSession();
        final Transaction tx = session.beginTransaction();
        try {
            final FlowStep result = flowStepDAO.loadByMeUUID(meUUID);
            tx.commit();
            return result;
        } catch (final RuntimeException e) {
            this.configAndFactory.getFactory().getCurrentSession().getTransaction().rollback();
            throw e;
        }
    }

    public List<FlowStep> loadFlowSteps(final String flowId, final String interfaceName,
            final String serviceName) {
        final Session session = this.configAndFactory.getFactory().getCurrentSession();
        final Transaction tx = session.beginTransaction();
        try {
            final List<FlowStep> result = flowStepDAO.loadByFlowIdAndMeProps(flowId, interfaceName,
                    serviceName);
            tx.commit();
            return result;
        } catch (final RuntimeException e) {
            this.configAndFactory.getFactory().getCurrentSession().getTransaction().rollback();
            throw e;
        }
    }

    public FlowStep loadFlowStep(final long flowStepId) {
        final Session session = this.configAndFactory.getFactory().getCurrentSession();
        final Transaction tx = session.beginTransaction();
        try {
            final FlowStep result = flowStepDAO.find(flowStepId);
            tx.commit();
            return result;
        } catch (final RuntimeException e) {
            this.configAndFactory.getFactory().getCurrentSession().getTransaction().rollback();
            throw e;
        }
    }

    public List<Flow> loadAllStarted() {
        final Session session = this.configAndFactory.getFactory().getCurrentSession();
        final Transaction tx = session.beginTransaction();
        try {
            final List<Flow> result = flowDAO.loadAllStarted();
            if (result != null) {
                for (final Flow flow : result) {
                    this.updateFlowStatusAndDates(flow);
                }
            }
            tx.commit();
            return result;
        } catch (final RuntimeException e) {
            this.configAndFactory.getFactory().getCurrentSession().getTransaction().rollback();
            throw e;
        }
    }

    public void addStep(final Flow flow, final FlowStep flowStep) {
        final Session session = this.configAndFactory.getFactory().getCurrentSession();
        final Transaction tx = session.beginTransaction();
        try {
            flowStep.setFlow(flow);
            flowStepDAO.save(flowStep);
            tx.commit();
        } catch (final RuntimeException e) {
            this.configAndFactory.getFactory().getCurrentSession().getTransaction().rollback();
            throw e;
        }
    }

    public void updateStep(final FlowStep flowStep) {
        final Session session = this.configAndFactory.getFactory().getCurrentSession();
        final Transaction tx = session.beginTransaction();
        try {
            flowStepDAO.save(flowStep);
            tx.commit();
        } catch (final RuntimeException e) {
            this.configAndFactory.getFactory().getCurrentSession().getTransaction().rollback();
            throw e;
        }
    }

    public List<Flow> loadByType(final int type) {
        final Session session = this.configAndFactory.getFactory().getCurrentSession();
        final Transaction tx = session.beginTransaction();
        try {
            final List<Flow> result = flowDAO.loadByType(type);
            if (result != null) {
                for (final Flow flow : result) {
                    this.updateFlowStatusAndDates(flow);
                }
            }
            tx.commit();
            return result;
        } catch (final RuntimeException e) {
            this.configAndFactory.getFactory().getCurrentSession().getTransaction().rollback();
            throw e;
        }
    }

    public List<Flow> loadAllStartedByType(final int type) {
        final Session session = this.configAndFactory.getFactory().getCurrentSession();
        final Transaction tx = session.beginTransaction();
        try {
            final List<Flow> result = flowDAO.loadAllStartedByType(type);
            if (result != null) {
                for (final Flow flow : result) {
                    this.updateFlowStatusAndDates(flow);
                }
            }
            tx.commit();
            return result;
        } catch (final RuntimeException e) {
            this.configAndFactory.getFactory().getCurrentSession().getTransaction().rollback();
            throw e;
        }
    }

    public List<Flow> loadAllStartedByTypeWithDateFilter(final int type, final Date startDate,
            final Date endDate) {
        final Session session = this.configAndFactory.getFactory().getCurrentSession();
        final Transaction tx = session.beginTransaction();
        try {
            final List<Flow> result = flowDAO.loadAllStartedByTypeWithDateFilter(type, startDate,
                    endDate);
            if (result != null) {
                for (final Flow flow : result) {
                    this.updateFlowStatusAndDates(flow);
                }
            }
            tx.commit();
            return result;
        } catch (final RuntimeException e) {
            this.configAndFactory.getFactory().getCurrentSession().getTransaction().rollback();
            throw e;
        }
    }

    /**
     * Flow status is calculated at runtime because it could change when the
     * corresponding flow ref is defined
     * 
     * @param idpetals
     * @return
     */
    private void updateFlowStatusAndDates(final Flow flow) {
        // Try to find a flow step ref defined as the end of the flow
        final FlowStepRef startStepRef = flowStepRefDAO.loadEndStepByFlowType(flow.getType());
        if (startStepRef != null) {
            // The flow status is equal to this step status
            // Load the dynamic flow step corresponding to this step service
            // name and interface name
            final List<FlowStep> flowSteps = flowStepDAO.loadByFlowIdAndMeProps(flow.getIdpetals(),
                    startStepRef.getInterfaceName(), startStepRef.getServiceName());
            if (flowSteps != null && flowSteps.size() == 1) {
                flow.setStatus(flowSteps.get(0).getStatus());
                flow.setEndDate(flowSteps.get(0).getEndDate());
            }
        }
        // Try to find a flow step ref defined as the start of the flow
        final FlowStepRef endStepRef = flowStepRefDAO.loadStartStepByFlowType(flow.getType());
        if (endStepRef != null) {
            // The flow status is equal to this step status
            // Load the dynamic flow step corresponding to this step service
            // name and interface name
            final List<FlowStep> flowSteps = flowStepDAO.loadByFlowIdAndMeProps(flow.getIdpetals(),
                    endStepRef.getInterfaceName(), endStepRef.getServiceName());
            if (flowSteps != null && flowSteps.size() == 1) {
                flow.setStartDate(flowSteps.get(0).getStartDate());
            }
        } else {
            flow.setStatus(-1);
        }
    }

    public List<Flow> getAllFlows(Date after, Date before, RequestOptions requestOptions) {

        return flowDAO.getAllFlows(after, before, requestOptions);
    }

    public int countAllFlows(Date after, Date before, RequestOptions requestOptions) {

        return flowDAO.countAllFlows(after, before, requestOptions);
    }

    public List<FlowWithParams> getAllFlowsByType(Date after, Date before,
            List<GlobalFlowParam> globalParams, final short flowType, RequestOptions requestOptions) {

        return flowDAO.getAllFlowsByType(after, before, globalParams, flowType, requestOptions);
    }

    public int countAllFlowsByType(Date after, Date before, List<GlobalFlowParam> globalParams,
            final short flowType, RequestOptions requestOptions) {

        return flowDAO.countAllFlowsByType(after, before, globalParams, flowType, requestOptions);
    }
}
