/**
 * 
 */

package org.ow2.petals.flowwatch.notification;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;

import org.hibernate.Session;
import org.hibernate.Transaction;
import org.ow2.petals.flowwatch.flowmanager.ConfigAndFactoryManager;
import org.ow2.petals.flowwatch.flowmanager.bo.Flow;
import org.ow2.petals.flowwatch.flowmanager.bo.FlowStep;
import org.ow2.petals.flowwatch.flowmanager.dao.FlowDAO;
import org.ow2.petals.flowwatch.flowmanager.dao.FlowStepDAO;
import org.ow2.petals.flowwatch.utils.HibernateUtils;
import org.ow2.petals.flowwatch.utils.NotificationProcessingException;

/**
 * This class is in charge of processing a {@link Notification} instance to
 * create/update appropriated objects ( {@link Flow} and {@link FlowStep} ) in
 * database
 * 
 * @author ofabre
 */
public class NotificationProcessor extends ConfigAndFactoryManager {

    private FlowDAO flowDAO = null;

    private FlowStepDAO flowStepDAO = null;

    private Map<String, Flow> flowMap = null;

    private Map<String, FlowStep> flowStepMap = null;

    public NotificationProcessor() {
        super();
        this.flowDAO = new FlowDAO();
        this.flowStepDAO = new FlowStepDAO();
        this.flowMap = new ConcurrentHashMap<String, Flow>();
        this.flowStepMap = new ConcurrentHashMap<String, FlowStep>();
    }

    public void process(final List<Notification> notificationList)
            throws NotificationProcessingException {
        if (notificationList != null) {
            Transaction tx = null;
            try {
                final Session session = HibernateUtils.getConfigAndFactory().getFactory()
                        .getCurrentSession();
                tx = session.beginTransaction();
                final Iterator<Notification> iterator = notificationList.iterator();
                while (iterator.hasNext()) {
                    final Notification notification = iterator.next();
                    if (notification != null && notification.isValid()) {
                        final Flow flow = this.createOrRetrieveFlow(notification);
                        this.createOrRetrieveFlowStep(notification, flow);
                    }
                }
                this.save();
                tx.commit();
            } catch (final Throwable t) {
                tx.rollback();
                throw new NotificationProcessingException(t);
            }
        } else {
            throw new NotificationProcessingException("Can't process a null notification list");
        }
    }

    /**
     * Retrieve if exists and create if not exists a flow step
     * 
     * @param notification
     *            The notification
     * @param flow
     *            The flow
     * @return The flow step
     * @throws NotificationProcessingException
     */
    private void createOrRetrieveFlowStep(final Notification notification, final Flow flow)
            throws NotificationProcessingException {
        final FlowStep flowStep = this.retrieveFlowStep(notification);
        if (flowStep == null) {
            this.createFlowStep(notification, flow);
        }
        this.updateFlowStep(notification);
    }

    /**
     * Retrieve if exists and create if not exists a flow
     * 
     * @param notification
     *            The notification
     * @return the flow
     * @throws Throwable
     */
    private Flow createOrRetrieveFlow(final Notification notification) throws Throwable {
        Flow flow = null;
        flow = this.retrieveFlow(notification);
        if (flow == null) {
            flow = this.createFlow(notification);
        }
        return flow;
    }

    /**
     * Save the flows and flowSteps
     * 
     * @param notification
     *            The notification
     * @return the flow
     * @throws Throwable
     */
    private void save() {
        final Collection<Flow> flowCollection = this.flowMap.values();
        final Flow[] flowArray = flowCollection.toArray(new Flow[flowCollection.size()]);
        this.flowDAO.save(flowArray);
        final Collection<FlowStep> flowStepCollection = this.flowStepMap.values();
        final FlowStep[] flowStepArray = flowStepCollection.toArray(new FlowStep[flowStepCollection
                .size()]);
        this.flowStepDAO.save(flowStepArray);
        this.flowMap.clear();
        this.flowStepMap.clear();
    }

    private FlowStep retrieveFlowStep(final Notification notification) {
        FlowStep flowStep = null;
        flowStep = this.flowStepMap.get(notification.getMeUUID());
        if (flowStep == null) {
            flowStep = this.flowStepDAO.loadByMeUUID(notification.getMeUUID());
            if (flowStep != null) {
                this.flowStepMap.put(flowStep.getMeUUID(), flowStep);
            }
        }
        return flowStep;
    }

    private Flow retrieveFlow(final Notification notification) {
        Flow flow = null;

        flow = this.flowMap.get(notification.getProcessId());
        if (flow == null) {
            flow = this.flowDAO.loadByPetalsId(notification.getProcessId());
            if (flow != null) {
                this.flowMap.put(flow.getIdpetals(), flow);
            }
        }
        return flow;
    }

    private Flow createFlow(final Notification notification) {
        final Flow flow = new Flow();
        // We admit that the process Id is generated before
        // notification emit and is unique
        flow.setIdpetals(notification.getProcessId());
        flow.setType(notification.getProcessType());
        this.flowMap.put(flow.getIdpetals(), flow);
        return flow;
    }

    private FlowStep createFlowStep(final Notification notification, final Flow flow)
            throws NotificationProcessingException {
        final FlowStep flowStep = new FlowStep();
        flowStep.setEndpointName(notification.getEndpointName());
        flowStep.setInterfaceName(notification.getInterfaceName());
        flowStep.setServiceName(notification.getServiceName());
        flowStep.setMeUUID(notification.getMeUUID());
        flowStep.setStatus(Flow.IN_PROGRESS_STATUS);
        flowStep.setFlow(flow);
        this.flowStepMap.put(flowStep.getMeUUID(), flowStep);
        return flowStep;
    }

    /**
     * @param notification
     * @param flowStep
     * @throws NotificationProcessingException
     */
    private FlowStep updateFlowStep(final Notification notification)
            throws NotificationProcessingException {
        final FlowStep flowStep = this.flowStepMap.get(notification.getMeUUID());
        if (flowStep != null) {
            switch (notification.getStatus()) {
                case Flow.IN_PROGRESS_STATUS:
                    flowStep.setStartDate(notification.getNotifDate());
                    final List<String> paramList = new CopyOnWriteArrayList<String>();
                    if (notification.getParams() != null && !notification.getParams().isEmpty()) {
                        paramList.addAll(notification.getParams());
                    }
                    if (flowStep.getParameters() != null && !flowStep.getParameters().isEmpty()) {
                        paramList.addAll(flowStep.getParameters());
                    }
                    flowStep.setParameters(paramList);
                    break;
                default:
                    if (notification.getStatus() < -1) {
                        throw new NotificationProcessingException(
                                "Try to process an unsupported notif status: "
                                        + notification.getStatus());
                    }
                    flowStep.setStatus(notification.getStatus());
                    flowStep.setEndDate(notification.getNotifDate());
                    break;
            }
            this.flowStepMap.put(flowStep.getMeUUID(), flowStep);
        }
        return flowStep;
    }
}
