/**
 * PETALS - PETALS Services Platform. Copyright (c) 2009 EBM Websourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 */
package org.ow2.petals.ant;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;
import org.ow2.petals.ant.util.NioUtil;

/**
 * Abstract class for all tasks configuring an archive.
 * 
 * @author rnaudin
 * 
 */
public class AbstractConfigureArchiveAntTask extends Task {

    protected final static String JBIDESCRIPTOR_ZIPENTRY_NAME = "META-INF/jbi.xml";

    /**
     * Ant task attribute 'file' locating the archive to configure.
     */
    protected String file;

    /**
     * Ant task attribute 'outputFile' defined the filename of the archive
     * configured.
     */
    protected String outputFile;

    /**
     * Overwrite the output archive
     */
    protected boolean overwriteOutputFile = true;

    @Override
    public void execute() throws BuildException {
        super.execute();

        final URL outputFileURL = validateFileParameter(this.outputFile, "outputFile");
        final File outputFileFile;
        try {
            outputFileFile = new File(outputFileURL.toURI());
            if (!this.overwriteOutputFile && outputFileFile.exists()) {
                throw new BuildException("The output component archive '" + this.outputFile
                        + "' already exists.");
            }
        } catch (final URISyntaxException e) {
            // This exception should not occur.
            throw new BuildException(e);
        }
    }

    /**
     * Download the given URL into a File.
     * 
     * @param componentURL
     * @return
     * @throws IOException
     */
    protected static final File downloadURL(URL componentURL) throws IOException {
        InputStream inputStream = null;
        FileOutputStream fileOutputStream = null;
        File outputFile = null;
        try {
            String path = componentURL.getPath();
            String fileName = path.substring(path.lastIndexOf('/') + 1);
            outputFile = File.createTempFile(fileName, null);
            inputStream = componentURL.openStream();
            NioUtil.copyStreamToFile(inputStream, outputFile);
        } finally {
            if (inputStream != null) {
                inputStream.close();
            }
            if (fileOutputStream != null) {
                fileOutputStream.close();
            }
        }

        return outputFile;
    }

    /**
     * Validate the file parameter. The file parameter should not be null and
     * must be a jar or a zip archive.
     * 
     * @throws BuildException
     */
    protected static final URL validateFileParameter(String fileParameter, String paramaterName)
            throws BuildException {
        if (fileParameter == null) {
            throw new BuildException("Missing attribute '" + paramaterName + "'");
        }

        if (fileParameter.isEmpty()) {
            throw new BuildException("Empty attribute '" + paramaterName + "'");
        }

        if (!fileParameter.endsWith(".zip") && !fileParameter.endsWith(".jar")) {
            throw new BuildException("File '" + fileParameter
                    + "' is not a valid archive (zip or jar required)");
        }

        URL fileURL;
        try {
            fileURL = new URL(fileParameter);
        } catch (MalformedURLException e) {
            try {
                fileURL = new File(fileParameter).toURI().toURL();
            } catch (MalformedURLException e1) {
                throw new BuildException("Wrong format for attribute '" + paramaterName
                        + "', a valid file location is required");
            }
        }
        return fileURL;
    }

    /**
     * File getter.
     * 
     * @return
     */
    public String getFile() {
        return file;
    }

    /**
     * File setter.
     * 
     * @param file
     */
    public void setFile(String file) {
        this.file = file;
    }

    /**
     * OutputFile getter.
     * 
     * @return
     */
    public String getOutputFile() {
        return outputFile;
    }

    /**
     * OutputFile setter.
     * 
     * @param outputFile
     */
    public void setOutputFile(String outputFile) {
        this.outputFile = outputFile;
    }

    /**
     * @return the overwriteOutputFile
     */
    public boolean isOverwriteOutputFile() {
        return this.overwriteOutputFile;
    }

    /**
     * @param overwriteOutputFile
     *            the overwriteOutputFile to set
     */
    public void setOverwriteOutputFile(boolean overwriteOutputFile) {
        this.overwriteOutputFile = overwriteOutputFile;
    }

}