/**
 * PETALS - PETALS Services Platform. Copyright (c) 2009 EBM Websourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 */

package org.ow2.petals.ant.task;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;

import org.apache.tools.ant.BuildException;
import org.ow2.petals.ant.AbstractConfigureArchiveAntTask;
import org.ow2.petals.ant.util.ZipUtil;
import org.ow2.petals.ant.util.ZipUtil.ZipEntryCallback;
import org.ow2.petals.jbi.descriptor.JBIDescriptorException;
import org.ow2.petals.jbi.descriptor.original.JBIDescriptorBuilder;
import org.ow2.petals.jbi.descriptor.original.generated.Component;
import org.ow2.petals.jbi.descriptor.original.generated.Jbi;

/**
 * Ant task : petals-configure-component. This task pre-configure a JBI
 * component (service engine of binding component).
 * 
 * @author rnaudin
 * 
 */
public class ConfigureComponentTask extends AbstractConfigureArchiveAntTask {

    /**
     * SharedLibrary class.
     * 
     * @author rnaudin
     * 
     */
    public static final class SharedLibrary {

        private String file;

        public SharedLibrary() {
            super();
        }

        public SharedLibrary(String file) {
            super();
            this.file = file;
        }

        public String getFile() {
            return file;
        }

        public void setFile(String file) {
            this.file = file;
        }
    }

    /**
     * Inner structure.
     * 
     * @author rnaudin
     * 
     */
    private static final class _SharedLibrary {
        public String identification;

        public String version;
    }

    /**
     * The aimed identification of the component.
     */
    private String identification;

    private final List<SharedLibrary> sharedLibraries = new ArrayList<SharedLibrary>();

    /**
     * Location of the file that contains Shared Libraries properties
     */
    private String slProperties;

    /**
     * A Shared Library to include in the component configuration.
     * 
     * @return
     */
    public SharedLibrary createSharedLibrary() {
        SharedLibrary sharedLibrary = new SharedLibrary();
        this.sharedLibraries.add(sharedLibrary);
        return sharedLibrary;
    }

    @Override
    public void execute() throws BuildException {
        super.execute();

        final List<_SharedLibrary> _sharedLibraries = new ArrayList<_SharedLibrary>();

        try {
            this.loadSharedLibrariesProperties();

            for (SharedLibrary sharedLibrary : this.sharedLibraries) {
                final URL slUrl = validateFileParameter(sharedLibrary.getFile(),
                        "sharedlibrary.file");
                final File slFile = downloadURL(slUrl);
                try {
                    final ZipFile slZip = new ZipFile(slFile);
                    try {
                        final ZipEntry jbiZipEntry = slZip.getEntry(JBIDESCRIPTOR_ZIPENTRY_NAME);
                        final InputStream jbiInputStream = slZip.getInputStream(jbiZipEntry);
                        try {
                            final Jbi slJbi = JBIDescriptorBuilder
                                    .buildJavaJBIDescriptor(jbiInputStream);
                            _SharedLibrary _sharedLibrary = new _SharedLibrary();
                            _sharedLibrary.identification = slJbi.getSharedLibrary()
                                    .getIdentification().getName();
                            _sharedLibrary.version = slJbi.getSharedLibrary().getVersion();
                            _sharedLibraries.add(_sharedLibrary);
                        } finally {
                            jbiInputStream.close();
                        }
                    } finally {
                        slZip.close();
                    }
                } finally {
                    slFile.delete();
                }
            }

            final URL url = validateFileParameter(this.file, "file");
            final File componentFile = downloadURL(url);
            try {
                final ZipFile zipComponentFile = new ZipFile(componentFile);
                try {
                    final ZipOutputStream zipOutputFile = new ZipOutputStream(new FileOutputStream(
                            this.outputFile));

                    ZipUtil.copyAndUpdateZipFile(zipComponentFile, zipOutputFile,
                            new ZipEntryCallback() {
                                public InputStream onZipEntry(final ZipEntry zipEntry,
                                        final InputStream zipEntryInputStream) throws IOException,
                                        JBIDescriptorException {
                                    if (JBIDESCRIPTOR_ZIPENTRY_NAME.equals(zipEntry.getName())) {
                                        InputStream resultStream = null;
                                        try {
                                            Jbi componentJbi = JBIDescriptorBuilder
                                                    .buildJavaJBIDescriptor(zipEntryInputStream);
                                            List<Component.SharedLibrary> componentSLs = componentJbi
                                                    .getComponent().getSharedLibraryList();
                                            componentSLs.clear();
                                            for (_SharedLibrary _sharedLibrary : _sharedLibraries) {
                                                Component.SharedLibrary sharedLibrary = new Component.SharedLibrary();
                                                sharedLibrary
                                                        .setContent(_sharedLibrary.identification);
                                                sharedLibrary.setVersion(_sharedLibrary.version);
                                                componentSLs.add(sharedLibrary);
                                            }

                                            if (ConfigureComponentTask.this.identification != null) {
                                                componentJbi
                                                        .getComponent()
                                                        .getIdentification()
                                                        .setName(
                                                                ConfigureComponentTask.this.identification);
                                            }

                                            String jbiString = JBIDescriptorBuilder
                                                    .buildXmlStringJBIdescriptor(componentJbi);

                                            resultStream = new ByteArrayInputStream(jbiString
                                                    .getBytes());
                                        } finally {
                                            zipEntryInputStream.close();
                                        }
                                        return resultStream;
                                    } else {
                                        return zipEntryInputStream;
                                    }
                                }
                            });
                    zipOutputFile.flush();
                    zipOutputFile.close();
                } finally {
                    zipComponentFile.close();
                }
            } finally {
                componentFile.delete();
            }
        } catch (IOException e) {
            throw new BuildException(e);
        } catch (JBIDescriptorException e) {
            throw new BuildException(e);
        }

        log("Component configuration set");
    }

    /**
     * Identification getter.
     * 
     * @return
     */
    public String getIdentification() {
        return identification;
    }

    /**
     * SlProperties getter.
     * 
     * @return
     */
    public String getSlProperties() {
        return slProperties;
    }

    /**
     * Identification setter.
     * 
     * @param identification
     */
    public void setIdentification(String identification) {
        this.identification = identification;
    }

    /**
     * SlProperties setter.
     * 
     * @param slProperties
     */
    public void setSlProperties(String slProperties) {
        this.slProperties = slProperties;
    }

    /**
     * Load the Shared Libraries properties.
     * 
     * @throws IOException
     */
    private void loadSharedLibrariesProperties() throws IOException {
        Properties props = new Properties();
        if (this.slProperties != null) {
            props.load(new FileInputStream(this.slProperties));
        }
        for (Object value : props.values()) {
            SharedLibrary sl = new SharedLibrary((String) value);
            this.sharedLibraries.add(sl);
        }
    }

}
