/**
 * PETALS: PETALS Services Platform Copyright (C) 2005 EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 * 
 * Initial developer(s): EBM WebSourcing
 */
package org.ow2.petals.ant.task;

import java.io.FileInputStream;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Set;

import javax.management.MBeanAttributeInfo;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.types.LogLevel;
import org.ow2.petals.ant.AbstractJBIAntTask;
import org.ow2.petals.jmx.RuntimeConfigurationComponentClient;

/**
 * Ant task : petals--configure-runtime-component. This task allow to update JBI
 * component attributes at runtime (service engine or binding component) into
 * the JBI environment.
 * 
 * @author aruffie
 * 
 */
public class ConfigureRuntimeComponentTask extends AbstractJBIAntTask {

    /**
     * Parameter class for nested <param> elements
     * 
     * @author chamerling
     * 
     */
    public static class Param {
        /**
         * Name attribute
         */
        private String name;

        /**
         * Value attribute
         */
        private String value;

        /**
         * Creates a new instance of {@link Param}
         * 
         */
        public Param() {
            this("", "");
        }

        /**
         * Creates a new instance of {@link Param}
         * 
         * @param name
         * @param value
         */
        public Param(String name, String value) {
            this.name = name;
            this.value = value;
        }

        /**
         * @return the name
         */
        public String getName() {
            return name;
        }

        /**
         * @return the value
         */
        public String getValue() {
            return value;
        }

        /**
         * @param name
         *            the name to set
         */
        public void setName(String name) {
            this.name = name;
        }

        /**
         * @param value
         *            the value to set
         */
        public void setValue(String value) {
            this.value = value;
        }
    }

    private final static Hashtable<String, Class<?>> primitiveClasses = new Hashtable<String, Class<?>>(
            12);

    /**
     * Name of the JBI component
     */
    private String name;

    /**
     * The nested elements <param>
     */
    private List<Param> nestedParams = new ArrayList<Param>();

    /**
     * Location of the params file that contains MBean properties
     */
    private String params;

    /**
     * Creates a new instance of {@link ConfigureRuntimeComponentTask}
     * 
     */
    public ConfigureRuntimeComponentTask() {
        super();
    }

    static {
        primitiveClasses.put(Boolean.class.getName(), Boolean.class);
        primitiveClasses.put(Boolean.TYPE.getName(), Boolean.class);
        primitiveClasses.put(Short.class.getName(), Short.class);
        primitiveClasses.put(Short.TYPE.getName(), Short.class);
        primitiveClasses.put(Integer.class.getName(), Integer.class);
        primitiveClasses.put(Integer.TYPE.getName(), Integer.class);
        primitiveClasses.put(Long.class.getName(), Long.class);
        primitiveClasses.put(Long.TYPE.getName(), Long.class);
        primitiveClasses.put(Float.class.getName(), Float.class);
        primitiveClasses.put(Float.TYPE.getName(), Float.class);
        primitiveClasses.put(Double.class.getName(), Double.class);
        primitiveClasses.put(Double.TYPE.getName(), Double.class);
    }

    /**
     * 
     * @return
     */
    public Param createParam() {
        Param param = new Param();
        nestedParams.add(param);
        return param;
    }

    /**
     * Execute the ant task
     * 
     * @throws IOException
     */
    public void doTask() throws Exception {
        if (this.name == null) {
            throw new BuildException("Missing attribute 'name'");
        }

        Properties paramsProperties = loadProperties();
        final RuntimeConfigurationComponentClient rccc = this.getJMXClient()
                .getRuntimeConfigurationClient(this.name);
        this.log(this.nestedParams.size() + paramsProperties.size()
                + " runtime parameters will be set to the component '" + this.name + "'");
        if (rccc != null) {
            final Map<MBeanAttributeInfo, Object> runtimeAttributes = rccc
                    .getConfigurationMBeanAttributes();
            setRuntimeAttributes(runtimeAttributes, nestedParams, paramsProperties);
            rccc.setAttributes(runtimeAttributes);
            log("Runtime configuration of the component '" + this.name + "' set");
        } else {
            this.log("No runtime configuration MBean proposed by the runtime MBean!", LogLevel.WARN
                    .getLevel());
        }
    }

    /**
     * @return the nestedParams
     */
    public List<Param> getNestedParams() {
        return nestedParams;
    }

    /**
     * Set the name parameter
     * 
     * @param name
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * @param nestedParams
     *            the nestedParams to set
     */
    public void setNestedParams(List<Param> nestedParams) {
        this.nestedParams = nestedParams;
    }

    /**
     * @nestedParams params the params to set
     */
    public void setParams(String params) {
        this.params = params;
    }

    /**
     * Id the nested params contains this param?
     * 
     * @param nestedParams
     * @param name
     * @return
     */
    private Param containParam(final List<Param> nestedParams, final String name) {
        for (final Param p : nestedParams) {
            if (p.getName().equals(name)) {
                return p;
            }
        }
        return null;
    }

    /**
     * Is the properties params contains this param?
     * 
     * @param props
     * @param name
     * @return
     */
    private Param containParam(Properties props, final String name) {
        for (final Object key : props.keySet()) {
            if (name.equals((String) key)) {
                return new Param((String) key, (String) props.get(key));
            }
        }
        return null;
    }

    /**
     * Get the properties from the params file.
     * 
     * @return
     * @throws IOException
     */
    private Properties loadProperties() throws IOException {
        Properties props = new Properties();
        if (params != null)
            props.load(new FileInputStream(params));

        return props;
    }

    /**
     * Allow to set all provided attributes
     * 
     * @param Map
     *            <MBeanAttributeInfo, Object> attrs
     * @param List
     *            <Param> nestedParams
     */
    private void setRuntimeAttributes(final Map<MBeanAttributeInfo, Object> attrs,
            final List<Param> nestedParam, final Properties propertiesParam) {
        final Set<Entry<MBeanAttributeInfo, Object>> entriesSet = attrs.entrySet();
        for (final Entry<MBeanAttributeInfo, Object> entry : entriesSet) {
            Param param = containParam(nestedParams, entry.getKey().getName());
            if (param == null) {
                param = containParam(propertiesParam, entry.getKey().getName());
                if (param != null) {
                    this.log("Setting attribute from properties " + param.getName() + " = "
                            + param.getValue());
                }
            } else {
                this.log("Setting attribute " + param.getName() + " = " + param.getValue());

            }
            if (param != null) {
                // try to marshall
                Object castedValue = null;
                if (primitiveClasses.containsKey(entry.getKey().getType())) {
                    try {
                        Class<?> primitiveClass = primitiveClasses.get(entry.getKey().getType());
                        Method valueOfMethod = primitiveClass.getMethod("valueOf",
                                new Class[] { String.class });
                        castedValue = valueOfMethod.invoke(null, param.getValue());
                    } catch (SecurityException e) {

                    } catch (NoSuchMethodException e) {
                        throw new RuntimeException(e);
                    } catch (IllegalArgumentException e) {
                        throw new RuntimeException(e);
                    } catch (IllegalAccessException e) {
                        throw new RuntimeException(e);
                    } catch (InvocationTargetException e) {
                        throw new RuntimeException(e);
                    }
                } else if (entry.getKey().getType().equals(String.class.getName())) {
                    castedValue = param.getValue();
                }
                if (castedValue == null) {
                    throw new RuntimeException("Failed to identify primitive type '"
                            + entry.getKey().getType() + "' of parameter '"
                            + entry.getKey().getName() + "'");
                }
                entry.setValue(castedValue);
            }
        }

    }

}
