/**
 * PETALS: PETALS Services Platform Copyright (C) 2005 EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 * 
 * Initial developer(s): EBM WebSourcing
 */
package org.ow2.petals.ant.task;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import javax.jbi.management.LifeCycleMBean;
import javax.management.ObjectName;

import org.apache.tools.ant.BuildException;
import org.ow2.petals.ant.AbstractJBIAntTask;

/**
 * Task used to list the binding components
 * 
 * @author ddesjardins chamerling - eBMWebsourcing
 */
public class ListBindingComponentsTask extends AbstractJBIAntTask {

    /**
     * XML namespace
     */
    private static final String XMLNS = "http://java.sun.com/xml/ns/jbi/component-info-list";

    /**
     * If supplied, only the named binding component is reported. If the binding
     * doesn’t exist, an empty component-list report is given.
     */
    private String bindingComponentName;

    /**
     * List the Bindings that have Service Units deployed to them as part of the
     * Service Assembly deployed in the JBI Environment. Without this parameter,
     * no Service Assembly dependency is verified.
     */
    private String serviceAssemblyName;

    /**
     * List the Bindings that are dependent on the specified shared library.
     * Without this parameter, no shared library dependency will be verified.
     */
    private String sharedLibraryName;

    /**
     * Binding component state Valid state are : shutdown, stated, stopped
     */
    private String state;

    /**
     * If supplied, set the given Ant property name value to the XML version of
     * this listing task.
     */
    private String xmlOutput;

    /**
     * Creates a new instance of {@link ListBindingComponentsTask}
     * 
     */
    public ListBindingComponentsTask() {
        super();
    }

    /**
     * Execute the ant task
     * 
     * @throws IOException
     */
    public void doTask() throws Exception {
        if (state != null
                && !(LifeCycleMBean.SHUTDOWN.equalsIgnoreCase(state)
                        || LifeCycleMBean.STARTED.equalsIgnoreCase(state) || LifeCycleMBean.STOPPED
                        .equalsIgnoreCase(state))) {
            throw new BuildException("Invalid state '" + state + "' (Valid ones are : '"
                    + LifeCycleMBean.SHUTDOWN + "', '" + LifeCycleMBean.STARTED + "', or '"
                    + LifeCycleMBean.STOPPED + "')");
        }

        ObjectName[] bindings = this.getJMXClient().getAdminServiceClient().getBindingComponents();

        Map<ObjectName, String> components = new HashMap<ObjectName, String>();
        for (ObjectName objectName : bindings) {
            String currentState = this.getJMXClient().getComponentClient(objectName).getState();
            String name = objectName.getKeyProperty("name");

            // skip this BC if name has been provided and is not equal to the
            // current one
            if (bindingComponentName != null) {
                if (bindingComponentName.equals(name)) {
                    components.put(objectName, currentState);
                    break;
                } else {
                    continue;
                }
            }

            // skip this BC if state has been provided and is not equal to the
            // current one
            if (state != null && !state.equalsIgnoreCase(currentState)) {
                continue;
            }

            // skip this BC if service assembly has been provided and if the BC
            // does not use it
            if (serviceAssemblyName != null) {
                boolean found = false;
                String[] serviceAssemblyNames = this.getJMXClient().getDeploymentServiceClient()
                        .getDeployedServiceAssembliesForComponent(name);
                for (String saName : serviceAssemblyNames) {
                    if (serviceAssemblyName.equals(saName)) {
                        found = true;
                        break;
                    }
                }
                if (!found) {
                    continue;
                }
            }

            // skip this BC if shared library has been provided and if the BC
            // does not use it
            if (sharedLibraryName != null) {
                boolean found = false;
                String[] componentNames = this.getJMXClient().getInstallationServiceClient()
                        .getInstalledComponentsForSharedlibrary(sharedLibraryName);
                for (String componentName : componentNames) {
                    if (name.equals(componentName)) {
                        found = true;
                        break;
                    }
                }
                if (!found) {
                    continue;
                }
            }

            // add BC to list
            components.put(objectName, currentState);
        }

        // display
        StringBuffer output = new StringBuffer();
        StringBuffer outputXML = new StringBuffer();

        output.append("#######################################################\n");
        output.append("#                   Binding Components                #\n");
        output.append("#######################################################\n");

        outputXML.append("<?xml version='1.0' encoding='UTF-8'?>\n");
        outputXML.append("<component-info-list xmlns='");
        outputXML.append(XMLNS);
        outputXML.append("' version='1.0'>\n");

        for (ObjectName objectName : components.keySet()) {
            String name = objectName.getKeyProperty("name");
            // TODO : retrieve the description via a dedicated MBean method?
            String description = "";
            // (String) connection.getAttribute(objectName,"Description");
            String objState = components.get(objectName);

            // Add the component to the output
            output.append("Name        : " + name + "\n");
            output.append("State       : " + objState + "\n");
            output.append("Description : " + description + "\n\n");

            outputXML.append("\t<component-info type='binding-component' name='" + name
                    + "' state='" + objState + "'>\n");
            outputXML.append("\t\t<description>");
            outputXML.append(description);
            outputXML.append("</description>\n");
            outputXML.append("\t</component-info>\n");
        }
        outputXML.append("</component-info-list>");

        // display or save as property
        if (xmlOutput == null) {
            try {
                log(output.toString());
            } catch (NullPointerException e) {
            }
        } else {
            try {
                getProject().setNewProperty(xmlOutput, outputXML.toString());
            } catch (Exception e) {
            }
        }
    }

    /**
     * @param bindingComponentName
     *            the bindingComponentName to set
     */
    public void setBindingComponentName(String bindingComponentName) {
        this.bindingComponentName = bindingComponentName;
    }

    /**
     * @param serviceAssemblyName
     *            the serviceAssemblyName to set
     */
    public void setServiceAssemblyName(String serviceAssemblyName) {
        this.serviceAssemblyName = serviceAssemblyName;
    }

    /**
     * @param sharedLibraryName
     *            the sharedLibraryName to set
     */
    public void setSharedLibraryName(String sharedLibraryName) {
        this.sharedLibraryName = sharedLibraryName;
    }

    /**
     * @param state
     *            the state to set
     */
    public void setState(String state) {
        this.state = state;
    }

    /**
     * @param xmlOutput
     *            the xmlOutput to set
     */
    public void setXmlOutput(String xmlOutput) {
        this.xmlOutput = xmlOutput;
    }

}
