/**
 * PETALS: PETALS Services Platform Copyright (C) 2005 EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 * 
 * Initial developer(s): EBM WebSourcing
 */

package org.ow2.petals.ant.task;

import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.List;

import javax.jbi.management.DeploymentServiceMBean;

import org.apache.tools.ant.BuildException;
import org.ow2.petals.ant.AbstractJBIAntTask;
import org.ow2.petals.jbi.descriptor.original.JBIDescriptorBuilder;
import org.ow2.petals.jbi.descriptor.original.generated.Jbi;
import org.ow2.petals.jbi.descriptor.original.generated.ServiceUnit;

/**
 * Ant task : jbi-list-service-assemblies. This task prints information about
 * service assemblies deployed in the JBI environment.
 * 
 * @author ddesjardins chamerling - eBMWebsourcing
 */
public class ListServiceAssembliesTask extends AbstractJBIAntTask {

    /**
     * XML namespace
     */
    private static final String XMLNS = "http://java.sun.com/xml/ns/jbi/service-assembly-info-list";

    /**
     * List the service assemblies that has service units deployed to this
     * component. Without this parameter, all the service assemblies that are
     * deployed in the JBI Environment will be displayed.
     */
    private String componentName;

    /**
     * List the Bindings that have Service Units deployed to them as part of the
     * Service Assembly deployed in the JBI Environment. Without this parameter,
     * no Service Assembly dependency is verified.
     */
    private String serviceAssemblyName;

    /**
     * Binding component state Valid state are : shutdown, started, stopped (not
     * case sensitive).
     */
    private String state;

    /**
     * If supplied, set the given Ant property name value to the XML version of
     * this listing task.
     */
    private String xmlOutput;

    /**
     * Creates a new instance of {@link ListServiceAssembliesTask}
     * 
     */
    public ListServiceAssembliesTask() {
        super();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.ow2.petals.ant.AbstractJBIAntTask#doTask()
     */
    public void doTask() throws Exception {
        if (state != null
                && !(DeploymentServiceMBean.SHUTDOWN.equalsIgnoreCase(state)
                        || DeploymentServiceMBean.STARTED.equalsIgnoreCase(state) || DeploymentServiceMBean.STOPPED
                        .equalsIgnoreCase(state))) {
            throw new BuildException("Valid states are : '" + DeploymentServiceMBean.SHUTDOWN
                    + "', '" + DeploymentServiceMBean.STARTED + "', or '"
                    + DeploymentServiceMBean.STOPPED + "'");
        }

        String[] serviceAssemblyNames;
        if (this.componentName != null) {
            serviceAssemblyNames = this.getJMXClient().getDeploymentServiceClient()
                    .getDeployedServiceAssembliesForComponent(this.componentName);

        } else {
            List<String> saList = new ArrayList<String>();
            String[] saNames = this.getJMXClient().getDeploymentServiceClient()
                    .getDeployedServiceAssemblies();

            for (String saName : saNames) {

                // skip this SA if name has been provided and is not equal to
                // the current one
                if (serviceAssemblyName != null) {
                    if (saName.equals(serviceAssemblyName)) {
                        saList.add(saName);
                        break;
                    } else {
                        continue;
                    }
                }

                // skip this SA if state has been provided and is not equal to
                // the current one
                if (state != null) {
                    String currentState = this.getJMXClient().getDeploymentServiceClient()
                            .getState(saName);
                    if (state.equalsIgnoreCase(currentState)) {
                        continue;
                    }
                }
                saList.add(saName);
            }
            serviceAssemblyNames = saList.toArray(new String[0]);
        }

        // Output
        StringBuffer output = new StringBuffer();
        StringBuffer outputXML = new StringBuffer();
        output.append("#######################################################\n");
        output.append("#                 Service Assemblies                  #\n");
        output.append("#######################################################\n");
        outputXML.append("<?xml version=\"1.0\" encoding=\"utf-8\"?>\n");
        outputXML.append("<service-assembly-info-list xmlns=\"").append(
                ListServiceAssembliesTask.XMLNS).append("\" version=\"1.0\">\n");

        for (String saName : serviceAssemblyNames) {
            // Get the description field from the Service Assembly descriptor
            String descriptor = this.getJMXClient().getDeploymentServiceClient()
                    .getServiceAssemblyDescriptor(saName);
            ByteArrayInputStream bais = new ByteArrayInputStream(descriptor.getBytes());
            Jbi saJbiDescriptor = JBIDescriptorBuilder.buildJavaJBIDescriptor(bais);
            bais.close();
            String saDescription = saJbiDescriptor.getServiceAssembly().getIdentification()
                    .getDescription();

            // Get the state of the Service Assembly
            String saState = this.getJMXClient().getDeploymentServiceClient().getState(saName);

            // Add the component to the output
            output.append("Name : " + saName + "\n");
            output.append("State : " + saState + "\n");
            output.append("Description : " + saDescription + "\n\n");
            outputXML.append("\t<service-assembly-info name=\"").append(saName).append(
                    "\" state=\"").append(saState).append("\">\n");
            outputXML.append("\t\t<description>").append(saDescription).append("</description>\n");

            // get service units names for the current SA
            String[] sus = this.getJMXClient().getDeploymentServiceClient()
                    .getServiceUnitForServiceAssembly(saName);
            for (String suName : sus) {
                // Get SU Description from the SA JBI descriptor
                String suDescription = "";
                String suComponentName = "";
                List<ServiceUnit> suDescriptors = saJbiDescriptor.getServiceAssembly()
                        .getServiceUnit();
                for (ServiceUnit suDescriptor : suDescriptors) {
                    if (suName.equals(suDescriptor.getIdentification().getName())) {
                        suDescription = suDescriptor.getIdentification().getDescription();
                        suComponentName = suDescriptor.getTarget().getComponentName();
                        break;
                    }
                }

                // TODO the state of the SU is not necessary the SA one! Get the
                // real state from a dedicated MBean method
                output.append("--------------------Service Unit-----------------------\n");
                output.append("Name : " + suName + "\n");
                output.append("State : " + saState + "\n");
                output.append("Description : " + suDescription + "\n\n");
                outputXML.append("\t\t<service-unit-info name=\"").append(suName).append(
                        "\" state=\"").append(saState).append("\" deployed-on=\"").append(
                        suComponentName).append("\">\n");
                outputXML.append("\t\t\t<description>").append(suDescription).append(
                        "</description>\n");
                outputXML.append("\t\t</service-unit-info>\n");
            }

            outputXML.append("\t</service-assembly-info>\n");
        }
        outputXML.append("</service-assembly-info-list>");

        if (xmlOutput == null) {
            log(output.toString());
        } else {
            getProject().setNewProperty(xmlOutput, outputXML.toString());
        }
    }

    /**
     * @param componentName
     *            the componentName to set
     */
    public void setComponentName(String componentName) {
        this.componentName = componentName;
    }

    /**
     * @param serviceAssemblyName
     *            the serviceAssemblyName to set
     */
    public void setServiceAssemblyName(String serviceAssemblyName) {
        this.serviceAssemblyName = serviceAssemblyName;
    }

    /**
     * @param state
     *            the state to set
     */
    public void setState(String state) {
        this.state = state;
    }

    /**
     * @param xmlOutput
     *            the xmlOutput to set
     */
    public void setXmlOutput(String xmlOutput) {
        this.xmlOutput = xmlOutput;
    }
}
