/**
 * PETALS: PETALS Services Platform Copyright (C) 2005 EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 * 
 * Initial developer(s): EBM WebSourcing
 */
package org.ow2.petals.ant.task;

import java.util.ArrayList;
import java.util.List;

import org.ow2.petals.ant.AbstractJBIAntTask;

/**
 * Ant task : jbi-list-shared-libraries. This task prints information about all
 * shared libraries installed in JBI environment.
 * 
 * @author ddesjardins chamerling - eBMWebsourcing
 */
public class ListSharedLibrariesTask extends AbstractJBIAntTask {

    private static final String XMLNS = "http://java.sun.com/xml/ns/jbi/component-info-list";

    /**
     * List the shared libraries that the component depends on. Without this
     * parameter, all the shared libraries in the JBI Environment are displayed.
     */
    private String componentName;

    /**
     * If supplied, only the named shared library is reported. If the shared
     * library doesn’t exist, an empty component-list report is given.
     */
    private String sharedLibraryName;

    /**
     * If supplied, set the given Ant property name value to the XML version of
     * this listing task.
     */
    private String xmlOutput;

    /**
     * Creates a new instance of {@link ListSharedLibrariesTask}
     * 
     */
    public ListSharedLibrariesTask() {
        super();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.ow2.petals.ant.AbstractJBIAntTask#doTask()
     */
    public void doTask() throws Exception {

        String[] sharedLibraryNames = this.getJMXClient().getInstallationServiceClient()
                .getInstalledSharedLibraries();

        List<String> sharedLibraryList = new ArrayList<String>();

        for (String slName : sharedLibraryNames) {

            // skip this SL if component name has been provided and is not equal
            // to the component dependencies to the current one
            if (componentName != null) {
                String[] componentNames = this.getJMXClient().getInstallationServiceClient()
                        .getInstalledComponentsForSharedlibrary(slName);

                boolean found = false;
                for (String compName : componentNames) {
                    if (componentName.equals(compName)) {
                        found = true;
                        break;
                    }
                }
                if (!found) {
                    continue;
                }
            }

            // skip this SL if name has been provided and is not equal to
            // the current one
            if (sharedLibraryName != null) {
                if (!sharedLibraryName.equals(slName)) {
                    continue;
                }
            }

            sharedLibraryList.add(slName);
        }

        StringBuffer output = new StringBuffer();
        StringBuffer outputXML = new StringBuffer();
        output.append("#######################################################\n");
        output.append("#                    Shared Libraries                 #\n");
        output.append("#######################################################\n");
        outputXML.append("<?xml version='1.0' encoding='UTF-8'?>\n");
        outputXML.append("<component-info-list ");
        outputXML.append("xmlns='" + XMLNS + "' version='1.0'>\n");

        for (String name : sharedLibraryList) {
            // TODO get the state and the description
            String state = "TODO";
            String description = "TODO";

            output.append("Name        : " + name + "\n");
            output.append("State       : " + state + "\n");
            output.append("Description : " + description + "\n\n");

            outputXML.append("\t<component-info type='shared-library' name='");
            outputXML.append(name);
            outputXML.append("' state='");
            outputXML.append(state);
            outputXML.append("'>\n\t\t<description>");
            outputXML.append(description);
            outputXML.append("</description>\n");
            outputXML.append("\t</component-info>");
        }
        outputXML.append("</component-info-list>");

        if (xmlOutput == null) {
            log(output.toString());

        } else {
            getProject().setNewProperty(xmlOutput, outputXML.toString());
        }
    }

    /**
     * @param componentName
     *            the componentName to set
     */
    public void setComponentName(String componentName) {
        this.componentName = componentName;
    }

    /**
     * @param sharedLibraryName
     *            the sharedLibraryName to set
     */
    public void setSharedLibraryName(String sharedLibraryName) {
        this.sharedLibraryName = sharedLibraryName;
    }

    /**
     * @param xmlOutput
     *            the xmlOutput to set
     */
    public void setXmlOutput(String xmlOutput) {
        this.xmlOutput = xmlOutput;
    }

}
