/**
 * PETALS - PETALS Services Platform. Copyright (c) 2005 EBM Websourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 */
package org.ow2.petals.ant.util;

import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Enumeration;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;

import org.ow2.petals.jbi.descriptor.JBIDescriptorException;

public class ZipUtil {

    public interface ZipEntryCallback {
        public InputStream onZipEntry(final ZipEntry zipEntry, final InputStream zipEntryInputStream)
                throws IOException, JBIDescriptorException;
    }

    /**
     * Add a file to a zip stream, without checking if already zipped.
     * 
     * @param zipOutputStream
     *            OutputStream of the zip file
     * @parma inputStream InputStream of the file to add
     * @param zipEntryName
     *            Zip entry name of the file to add
     */
    public final static void addFile(final ZipOutputStream zipOutputStream,
            final InputStream inputStream, final String zipEntryName) throws IOException {

        ZipEntry zipEntry = new ZipEntry(zipEntryName);

        ZipUtil.putEntry(zipOutputStream, inputStream, zipEntry);
    }

    /**
     * Add a file to a zip stream specifiing zip entry time, without checking if
     * already zipped.
     * 
     * @param zipOutputStream
     *            OutputStream of the zip file
     * @parma inputStream InputStream of the file to add
     * @param zipEntryName
     *            Zip entry name of the file to add
     * @param zipEntryTime
     *            Zip entry time of the file to add
     */
    public final static void addFile(final ZipOutputStream zipOutputStream,
            final InputStream inputStream, final String zipEntryName, final long zipEntryTime)
            throws IOException {

        ZipEntry zipEntry = new ZipEntry(zipEntryName);
        zipEntry.setTime(zipEntryTime);

        ZipUtil.putEntry(zipOutputStream, inputStream, zipEntry);
    }

    public final static void copyAndUpdateZipFile(final ZipFile zipInputFile,
            final ZipOutputStream zipOutputStream, final ZipEntryCallback callback)
            throws IOException, JBIDescriptorException {

        Enumeration<? extends ZipEntry> zipEntries = zipInputFile.entries();
        while (zipEntries.hasMoreElements()) {
            final ZipEntry zipEntry = zipEntries.nextElement();

            final ZipEntry newZipEntry = new ZipEntry(zipEntry.getName());
            newZipEntry.setComment(zipEntry.getComment());
            newZipEntry.setTime(zipEntry.getTime());
            ZipUtil.putEntry(zipOutputStream, callback.onZipEntry(newZipEntry, zipInputFile
                    .getInputStream(zipEntry)), newZipEntry);

        }
        ;
    }

    /**
     * Add a zip entry file to a zip stream, without checking if already zipped.
     * 
     * @param zipOutputStream
     *            OutputStream of the zip file
     * @parma inputStream InputStream of the file to add
     * @param zipEntry
     *            Zip entry of the file to add
     */
    private final static void putEntry(final ZipOutputStream zipOutputStream,
            final InputStream inputStream, final ZipEntry zipEntry) throws IOException {
        final DataInputStream dis = new DataInputStream(inputStream);
        try {
            byte[] content = new byte[dis.available()];
            dis.readFully(content);

            zipOutputStream.putNextEntry(zipEntry);
            zipOutputStream.write(content);
            zipOutputStream.closeEntry();
        } finally {
            dis.close();
        }
    }
}
