/****************************************************************************
 *
 * Copyright (c) 2005-2012, EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA 
 *
 *****************************************************************************/
 
package org.ow2.petals.ant.task;

import java.io.FileInputStream;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.management.MBeanAttributeInfo;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.types.LogLevel;
import org.ow2.petals.ant.AbstractInstallerAntTask;
import org.ow2.petals.ant.task.primitive.PrimitiveConverter;
import org.ow2.petals.ant.task.primitive.PrimitiveConverterBoolean;
import org.ow2.petals.ant.task.primitive.PrimitiveConverterDouble;
import org.ow2.petals.ant.task.primitive.PrimitiveConverterFloat;
import org.ow2.petals.ant.task.primitive.PrimitiveConverterInteger;
import org.ow2.petals.ant.task.primitive.PrimitiveConverterLong;
import org.ow2.petals.ant.task.primitive.PrimitiveConverterShort;
import org.ow2.petals.jmx.ConfigurationInstallerComponentClient;
import org.ow2.petals.jmx.InstallerComponentClient;
import org.ow2.petals.jmx.exception.AttributeErrorException;

/**
 * Ant task : jbi-install-component. This task install a JBI component (service
 * engine of binding component) into the JBI environment.
 * 
 * @author ddesjardins - EBM WebSourcing
 * @author Christophe Hamerling - EBM WebSourcing
 */
public class InstallComponentTask extends AbstractInstallerAntTask {

    /**
     * Parameter class for nested <param> elements
     * 
     * @author chamerling
     * 
     */
    public static class Param {
        /**
         * Name attribute
         */
        private String name;

        /**
         * Value attribute
         */
        private String value;

        /**
         * Creates a new instance of {@link Param}
         * 
         */
        public Param() {
            this("", "");
        }

        /**
         * Creates a new instance of {@link Param}
         * 
         * @param name
         * @param value
         */
        public Param(String name, String value) {
            this.name = name;
            this.value = value;
        }

        /**
         * @return the name
         */
        public String getName() {
            return name;
        }

        /**
         * @return the value
         */
        public String getValue() {
            return value;
        }

        /**
         * @param name
         *            the name to set
         */
        public void setName(String name) {
            this.name = name;
        }

        /**
         * @param value
         *            the value to set
         */
        public void setValue(String value) {
            this.value = value;
        }
    }

    private final static Map<String, PrimitiveConverter> primitiveClasses = new Hashtable<String, PrimitiveConverter>();

    /**
     * the map of the Installer Configuration MBean attributes.
     */
    private Set<MBeanAttributeInfo> configurationAttributes;

    /**
     * The nested elements <param>
     */
    private final List<Param> nestedParams = new ArrayList<Param>();

    /**
     * Location of the params file that contains MBean properties
     */
    private String params;

    /**
     * Creates a new instance of {@link InstallComponentTask}
     * 
     */
    public InstallComponentTask() {
        super();
    }

    static {
        primitiveClasses.put(Boolean.class.getName(), new PrimitiveConverterBoolean());
        primitiveClasses.put(Boolean.TYPE.getName(), new PrimitiveConverterBoolean());
        primitiveClasses.put(Short.class.getName(), new PrimitiveConverterShort());
        primitiveClasses.put(Short.TYPE.getName(), new PrimitiveConverterShort());
        primitiveClasses.put(Integer.class.getName(), new PrimitiveConverterInteger());
        primitiveClasses.put(Integer.TYPE.getName(), new PrimitiveConverterInteger());
        primitiveClasses.put(Long.class.getName(), new PrimitiveConverterLong());
        primitiveClasses.put(Long.TYPE.getName(), new PrimitiveConverterLong());
        primitiveClasses.put(Float.class.getName(), new PrimitiveConverterFloat());
        primitiveClasses.put(Float.TYPE.getName(), new PrimitiveConverterFloat());
        primitiveClasses.put(Double.class.getName(), new PrimitiveConverterDouble());
        primitiveClasses.put(Double.TYPE.getName(), new PrimitiveConverterDouble());
    }

    /**
     * 
     * @return
     */
    public Param createParam() {
        Param param = new Param();
        nestedParams.add(param);
        return param;
    }

    /**
     * Execute the ant task
     * 
     * @throws IOException
     */
    public void doTask() throws Exception {
        URL fileURL = this.validateFileParameter();

        InstallerComponentClient installer = this.getJMXClient().getInstallationServiceClient()
                .loadNewInstaller(fileURL);

        Properties propertiesParams = loadProperties();

        if (nestedParams.size() > 0 || propertiesParams.size() > 0) {
            this.log(this.nestedParams.size() + propertiesParams.size()
                    + " installation parameters will be set to the installer");

            // Get the installer configuration
            ConfigurationInstallerComponentClient installerConfigurationMBean = installer
                    .getConfigurationInstallerClient();
            if (installerConfigurationMBean != null) {
                this.configurationAttributes = installerConfigurationMBean.getAttributes().keySet();
                setConnectionAttributeFromNestedParam(installerConfigurationMBean);
                setConnectionAttributeFromPropertiesFile(installerConfigurationMBean,
                        propertiesParams);
            } else {
                this.log("No installer configuration MBean proposed by the installer MBean!",
                        LogLevel.WARN.getLevel());
            }
        }

        // Install the component
        installer.install();
        log("Component '" + installer.getMBeanName().getKeyProperty("name") + "' installed");
    }

    /**
     * @return the params
     */
    public String getParams() {
        return params;
    }

    /**
     * 
     * @param params
     */
    public void setParams(String params) {
        this.params = params;
    }

    /**
     * Get the properties from the params file.
     * 
     * @return
     * @throws IOException
     */
    private Properties loadProperties() throws IOException {
        Properties props = new Properties();
        if (params != null)
            props.load(new FileInputStream(params));

        return props;
    }

    private void setAttribute(ConfigurationInstallerComponentClient installerConfigurationMBean,
            String key, String value) throws AttributeErrorException {
        Object castedValue = null;
        boolean found = false;
        for (MBeanAttributeInfo attributeInfo : this.configurationAttributes) {
            if (attributeInfo.getName().equals(key)) {
                if (primitiveClasses.containsKey(attributeInfo.getType())) {
                    final PrimitiveConverter primitiveConverter = primitiveClasses
                            .get(attributeInfo.getType());
                    try {
                        castedValue = primitiveConverter.convert(value);
                    } catch (final BuildException e) {
                        throw new BuildException("Invalid value for attribute '" + key + "': "
                                + e.getMessage());
                    }
                } else if (attributeInfo.getType().equals(String.class.getName())) {
                    castedValue = value;
                }
                found = true;
                break;
            }
        }

        if (castedValue == null) {
            if (found) {
                throw new RuntimeException("Failed to identify primitive type of parameter '" + key
                        + "'");
            } else {
                throw new RuntimeException("Failed to identify parameter '" + key + "'");
            }
        }
        installerConfigurationMBean.setAttribute(key, castedValue);
    }

    /**
     * Set the connection attribute from the nested params
     * 
     * @nestedParams connection
     * @nestedParams configurationMBean
     * @throws Exception
     */
    private void setConnectionAttributeFromNestedParam(
            ConfigurationInstallerComponentClient installerConfigurationMBean) throws Exception {
        // Set the properties of the <Param> element
        for (Param p : nestedParams) {
            this.log("Setting attribute " + p.getName() + " = " + p.getValue());
            setAttribute(installerConfigurationMBean, p.getName(), p.getValue());
        }
    }

    /**
     * Set the connection attributes from the properties file
     * 
     * @nestedParams connection
     * @nestedParams configurationMBean
     * @throws Exception
     */
    private void setConnectionAttributeFromPropertiesFile(
            ConfigurationInstallerComponentClient installerConfigurationMBean,
            Properties propertiesParams) throws Exception {
        for (Map.Entry<Object, Object> entry : propertiesParams.entrySet()) {
            this.log("Setting attribute from properties " + (String) entry.getKey() + " = "
                    + (String) entry.getValue());
            setAttribute(installerConfigurationMBean, (String) entry.getKey(),
                    (String) entry.getValue());
        }
    }
}
