/**
 * PETALS - PETALS Services Platform. Copyright (c) 2007 EBM Websourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * -------------------------------------------------------------------------
 * $Id$
 * -------------------------------------------------------------------------
 */

package org.ow2.petals.bc.ftp.listener;

import java.io.IOException;

import javax.jbi.messaging.MessagingException;
import javax.xml.namespace.QName;

import org.apache.commons.pool.impl.GenericObjectPool;
import org.ow2.petals.bc.ftp.FTPComponent;
import org.ow2.petals.bc.ftp.FTPConstants;
import org.ow2.petals.bc.ftp.FTPUtil;
import org.ow2.petals.bc.ftp.MissingElementException;
import org.ow2.petals.bc.ftp.connection.FTPConnectionInfo;
import org.ow2.petals.bc.ftp.connection.FTPConnectionInfoBuilder;
import org.ow2.petals.bc.ftp.connection.WrappedFTPClient;
import org.ow2.petals.bc.ftp.service.FTPService;
import org.ow2.petals.component.framework.api.Wsdl;
import org.ow2.petals.component.framework.api.message.Exchange;
import org.ow2.petals.component.framework.listener.AbstractJBIListener;
import org.ow2.petals.component.framework.util.WSDLUtilImpl;
import org.w3c.dom.Document;
import org.w3c.dom.Node;

import com.ebmwebsourcing.easycommons.lang.StringHelper;
import com.ebmwebsourcing.easycommons.logger.Level;
import com.ebmwebsourcing.easycommons.xml.XMLHelper;

/**
 * The FTP JBI listener. Writes a file to FTP server on incoming JBI message.
 * 
 * @author alouis
 * @author "Mathieu CARROLLE - mathieu.carrolle@petalslink.com"
 */
public class JBIListener extends AbstractJBIListener {

    private transient FTPService service = null;

    @Override
    public void init() {
        service = new FTPService(getLogger());
    }

    /**
     * Validate the operation. If the operation namespace is null, we set the
     * default namespace
     * 
     * @param operation
     */
    private static final void validateExchange(final Exchange exchange) throws MessagingException {
        final QName operation = exchange.getOperation();
        // Null operation
        if (operation == null) {
            throw new MessagingException(
                    "The JBI operation must be set by the consumer of the service.");
        }
        // if no namespace is provided we use the default namespace.
        if (StringHelper.isNullOrEmpty(operation.getNamespaceURI())) {
            exchange.setOperation(new QName(FTPConstants.FTP_SERVICE_NS, operation.getLocalPart()));
        }
    }

    /**
     * process the exchange depending on the operation: <li><b>dir</b> return
     * the filenames list or the remote directory</li> <li><b>get</b> return the
     * remote file with the name specified in the exchange (as attachment or
     * source, depending on the SU property)</li> <li><b>mget</b> return all the
     * remote files</li> <li><b>put</b> put the message</li> <li><b>mput</b> put
     * attachments</li>
     */
    public boolean onJBIMessage(Exchange exchange) {
        GenericObjectPool pool = null;
        WrappedFTPClient client = null;
        if (exchange.isActiveStatus()) {
            try {
                validateExchange(exchange);
                final QName operation = exchange.getOperation();
                final Document inMessage = exchange.getInMessageContentAsDocument();
                final Wsdl nativeWSDL = this.component.getNativeWsdl();

                // test if the component exposes a generic service
                if (nativeWSDL != null
                        && WSDLUtilImpl.isDescriptionContaining(nativeWSDL.getDescription(),
                                exchange.getEndpoint())) {
                    // search a "connection" elements in the children of the
                    // document
                    final Node connection = XMLHelper.findChild(inMessage.getFirstChild(), null,
                            FTPConstants.FTP_CONNECTION, false);
                    final FTPConnectionInfo connectionInfo = FTPConnectionInfoBuilder
                            .buildFTPConnectionInfo(connection);
                    client = new WrappedFTPClient(connectionInfo, this.getLogger());
                    client.connectAndLog();
                    client.configureConnection();
                } else {
                    // Not generic service call
                    pool = ((FTPComponent) this.component).getMapOfPool().get(this.getProvides());
                    client = (WrappedFTPClient) pool.borrowObject();
                }

                if (FTPConstants.DIR_OPERATION.equalsIgnoreCase(operation.getLocalPart())) {
                    service.processDir(exchange, client);
                } else if (FTPConstants.GET_OPERATION.equalsIgnoreCase(operation.getLocalPart())) {
                    service.processGet(exchange, inMessage, client);
                } else if (FTPConstants.GET_ATTACHMENT_OPERATION.equalsIgnoreCase(operation
                        .getLocalPart())) {
                    service.processGetAsAttachment(exchange, inMessage, client);
                } else if (FTPConstants.MGET_OPERATION.equalsIgnoreCase(operation.getLocalPart())) {
                    service.processMGet(exchange, inMessage, client);
                } else if (FTPConstants.PUT_OPERATION.equalsIgnoreCase(operation.getLocalPart())) {
                    service.processPut(exchange, inMessage, client);
                } else if (FTPConstants.MPUT_OPERATION.equalsIgnoreCase(operation.getLocalPart())) {
                    service.processMPut(exchange, inMessage, client);
                } else if (FTPConstants.DEL_OPERATION.equalsIgnoreCase(operation.getLocalPart())) {
                    service.processDel(exchange, inMessage, client);
                } else {
                    final String message = "The operation '" + operation + "' is not recognized";
                    throw new MessagingException(message);
                }
                
                // MONIT ext end
                
            } catch (MissingElementException e) {
                this.getLogger().log(Level.WARNING,
                        e.getClass().getSimpleName() + " : " + e.getMessage());
                if (exchange.isInOnlyPattern()) {
                    exchange.setError(e);
                } else {
                    try {
                        FTPUtil.setMissingElementFaultOnExchange(exchange, e.getElement());
                    } catch (MessagingException e1) {
                        exchange.setError(e1);
                    }
                }
            } catch (Exception e) {
                this.getLogger().log(Level.WARNING,
                        e.getClass().getSimpleName() + " : " + e.getMessage());
                exchange.setError(e);
            } finally {
                try {
                    if (pool == null) {
                        // Generic Service
                        if (client != null && client.isConnected()) {
                            this.getLogger().finest(
                                    "Disconnect FTPClient for the generic service connection");
                            client.logout();
                            client.disconnect();
                        }
                    } else {
                        if (client != null) {
                            this.getLogger().finest("Return FTPClient to the pool");
                            pool.returnObject(client);
                        }
                    }
                } catch (IOException e) {
                    this.getLogger().finest(
                            "Cannot disconnect the FTPClient for the generic service connection : ["
                                    + e.getMessage() + "]");
                } catch (Exception e2) {
                    this.getLogger().finest(
                            "Cannot release object of the pool : [" + e2.getMessage() + "]");
                }
            }
        }

        return true;
    }
}
