/**
 * PETALS - PETALS Services Platform. Copyright (c) 2007 EBM Websourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * -------------------------------------------------------------------------
 * $Id$
 * -------------------------------------------------------------------------
 */

package org.ow2.petals.bc.ftp;

import org.apache.commons.pool.ObjectPool;
import org.apache.commons.pool.impl.GenericObjectPool;
import org.apache.commons.pool.impl.GenericObjectPool.Config;
import org.ow2.petals.bc.ftp.connection.FTPConnectionInfo;
import org.ow2.petals.bc.ftp.connection.FTPConnectionInfoBuilder;
import org.ow2.petals.bc.ftp.connection.WrappedFTPClientFactory;
import org.ow2.petals.component.framework.AbstractComponent;
import org.ow2.petals.component.framework.api.configuration.ConfigurationExtensions;
import org.ow2.petals.component.framework.api.exception.PEtALSCDKException;
import org.ow2.petals.component.framework.jbidescriptor.generated.Jbi;
import org.ow2.petals.component.framework.jbidescriptor.generated.Provides;
import org.ow2.petals.component.framework.su.AbstractServiceUnitManager;
import org.ow2.petals.component.framework.su.ServiceUnitDataHandler;

/**
 * 
 * @author "Mathieu CARROLLE - mathieu.carrolle@petalslink.com"
 * 
 */
public class FtpSuManager extends AbstractServiceUnitManager {

    public FtpSuManager(AbstractComponent component) {
        super(component);
    }

    @Override
    protected void doDeploy(String serviceUnitName, String suRootPath, Jbi jbiDescriptor)
            throws PEtALSCDKException {
        final ServiceUnitDataHandler handler = this.getServiceUnitDataHandlers().get(
                serviceUnitName);
        final Provides provides = handler.getDescriptor().getServices().getProvides().get(0);
        final ConfigurationExtensions extensions = handler.getConfigurationExtensions(provides);
        FTPConnectionInfo connectionConfiguration;
        try {
            connectionConfiguration = FTPConnectionInfoBuilder.buildFTPConnectionInfo(extensions,
                    provides.getRetrypolicy());
        } catch (MissingElementException e) {
            throw new PEtALSCDKException(e);
        }
        final WrappedFTPClientFactory factory = new WrappedFTPClientFactory(
                connectionConfiguration, this.logger);
        final GenericObjectPool pool = new GenericObjectPool(factory);
        pool.setConfig(this.configurePool(connectionConfiguration, pool));
        ((FTPComponent) this.component).getMapOfPool().put(provides, pool);
    }

    @Override
    protected void doUndeploy(String serviceUnitName) throws PEtALSCDKException {
        final ServiceUnitDataHandler handler = this.getServiceUnitDataHandlers().get(
                serviceUnitName);
        final Provides provides = handler.getDescriptor().getServices().getProvides().get(0);
        final ObjectPool pool = ((FTPComponent) this.component).getMapOfPool().get(provides);
        try {
            this.logger.finest("Closing pool's resource");
            pool.clear();
            pool.close();
        } catch (Exception e) {
            this.logger.finest("Exception when closing pool's resource: " + e.getMessage());
        }
    }

    /**
     * Configure the pool
     * 
     * @param connectionConfiguration
     * @param pool
     * @return
     */
    private Config configurePool(final FTPConnectionInfo connectionConfiguration,
            final GenericObjectPool pool) {
        Config conf = new Config();
        int maxConnection = connectionConfiguration.getMaxConnection();
        if (maxConnection <= 0) {
            conf.whenExhaustedAction = GenericObjectPool.WHEN_EXHAUSTED_GROW;
        } else {
            this.logger.finest("Number of maximum connection for the pool: " + maxConnection);
            // On borrow action, we block until an object is free
            conf.whenExhaustedAction = GenericObjectPool.WHEN_EXHAUSTED_BLOCK;
            // Wait is unlimited;
            conf.maxWait = -1;
            conf.maxActive = maxConnection;
            conf.maxIdle = maxConnection;
        }
        // Minimun time before an object become eligible to eviction
        conf.minEvictableIdleTimeMillis = connectionConfiguration.getMaxIdleTime();
        conf.timeBetweenEvictionRunsMillis = connectionConfiguration.getMaxIdleTime() / 2;
        conf.testOnBorrow = true;
        return conf;
    }
}
