/**
 * PETALS - PETALS Services Platform. Copyright (c) 2007 EBM Websourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * -------------------------------------------------------------------------
 * $Id$
 * -------------------------------------------------------------------------
 */

package org.ow2.petals.bc.ftp.service;

import java.io.IOException;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.activation.DataHandler;
import javax.jbi.messaging.MessagingException;

import org.ow2.petals.bc.ftp.FTPConstants;
import org.ow2.petals.bc.ftp.FTPUtil;
import org.ow2.petals.bc.ftp.MissingElementException;
import org.ow2.petals.bc.ftp.connection.WrappedFTPClient;
import org.ow2.petals.component.framework.api.Message;
import org.ow2.petals.component.framework.api.Message.MEPConstants;
import org.ow2.petals.component.framework.api.exception.PEtALSCDKException;
import org.ow2.petals.component.framework.api.message.Exchange;
import org.ow2.petals.component.framework.util.MtomUtil;
import org.ow2.petals.component.framework.util.SourceUtil;
import com.ebmwebsourcing.easycommons.lang.StringHelper;
import org.ow2.petals.component.framework.util.MtomUtil.MtomMapping;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.ebmwebsourcing.easycommons.xml.XMLHelper;

/**
 * The FTP JBI listener. Writes a file to FTP server on incoming JBI message.
 * 
 * @author alouis
 */
public class FTPService {

    private final Logger logger;

    public FTPService(final Logger logger) {
        super();
        this.logger = logger;
    }

    /***
     * FTP DEL Operation <br>
     * 
     * @param exchange
     * @param connectionInfo
     * @param inMessage
     * @throws MissingElementException
     * @throws Exception
     */
    public void processDel(final Exchange exchange, final Document inMessage,
            final WrappedFTPClient client) throws MessagingException, MissingElementException {
        if (this.logger.isLoggable(Level.INFO)) {
            this.logger.info("JBI exchange received - Process DEL operation");
        }
        // Checks the MEP of the incoming message
        if (!exchange.isInOnlyPattern()) {
            throw new MessagingException(FTPUtil.getOperationValidMep(FTPConstants.DEL_OPERATION,
                    Message.MEPConstants.IN_ONLY_PATTERN));
        }
        // retrieve the filename to get
        final Node filePatternNode = XMLHelper.findChild(inMessage.getFirstChild(),
                null, FTPConstants.FTP_FILENAME, false);
        String filePattern = null;
        if (filePatternNode != null) {
            filePattern = filePatternNode.getTextContent();
        } else {
            throw new MissingElementException(FTPConstants.FTP_FILENAME);
        }

        if (filePattern != null) {
            try {
                // do the FTP DEL operation
                client.del(filePattern);
            } catch (final IOException e) {
                final StringBuilder errorSB = new StringBuilder();
                errorSB.append("Can not delete file on FTP server [");
                errorSB.append(client.getConnectionInfo().getServer());
                errorSB.append("] : IOException : ");
                errorSB.append(e.getMessage());
                throw new MessagingException(errorSB.toString());
            }
            if (this.logger.isLoggable(Level.FINEST)) {
                this.logger.finest("DEL operation ended");
            }
        }
    }

    /**
     * FTP DIR operation. <br>
     * a Fault is returned for bad request or IO problems
     * 
     * @param exchange
     *            the message exchange, MUST be non null
     * @throws MessagingException
     *             bad MEP
     */

    public void processDir(final Exchange exchange, final WrappedFTPClient client)
            throws MessagingException {
        if (this.logger.isLoggable(Level.INFO)) {
            this.logger.info("JBI exchange received - Process DIR operation");
        }
        if (!exchange.isInOutPattern()) {
            throw new MessagingException(FTPUtil.getOperationValidMep(FTPConstants.DIR_OPERATION,
                    MEPConstants.IN_OUT_PATTERN));
        }
        List<String> fileNames;
        try {
            // do the FTP DIR operation
            fileNames = client.listFolderContent();

            // generate filename list
            final String result = FTPUtil.generateFileNameList(fileNames, exchange.getOperation());
            exchange.getOutMessage().setContent(SourceUtil.createSource(result));
        } catch (IOException e) {
            FTPUtil.setIOFaultOnExchange(exchange, "Can not list files on FTP server ["
                    + client.getConnectionInfo().getServer() + "]. Cause : " + e.getMessage());
        } catch (PEtALSCDKException e) {
            throw new MessagingException(e);
        }
        if (this.logger.isLoggable(Level.FINEST)) {
            this.logger.finest("DIR operation ended");
        }
    }

    /**
     * 
     * FTP GET operation <br>
     * get the file as a source message a Fault is returned for bad request or
     * IO problems
     * 
     * @param exchange
     *            the message exchange, MUST be non null
     * @param connectionInfo
     * @param inMessage
     * @throws MessagingException
     *             bad MEP
     */
    public void processGet(final Exchange exchange, final Document inMessage,
            final WrappedFTPClient client) throws MessagingException {
        if (this.logger.isLoggable(Level.INFO)) {
            this.logger.info("JBI exchange received - Process GET operation");
        }
        // Checks the MEP of the incoming message
        if (!exchange.isInOutPattern()) {
            throw new MessagingException(FTPUtil.getOperationValidMep(FTPConstants.GET_OPERATION,
                    Message.MEPConstants.IN_OUT_PATTERN));
        }
        // retrieve the filename to get
        final Node filePatternNode = XMLHelper.findChild(inMessage.getFirstChild(),
                null, FTPConstants.FTP_FILENAME, false);
        String filePattern = null;
        if (filePatternNode != null) {
            filePattern = filePatternNode.getTextContent();
        }
        if (StringHelper.isNullOrEmpty(filePattern)) {
            FTPUtil.setMissingElementFaultOnExchange(exchange, FTPConstants.FTP_FILENAME);
        } else {
            try {
                // do the FTP GET operation
                final Document doc = client.get(filePattern);
                final Element rootElt = doc.createElementNS(exchange.getOperation()
                        .getNamespaceURI(), "ver:getResponse");
                rootElt.appendChild(doc.getDocumentElement());
                doc.appendChild(rootElt);
                exchange.setOutMessageContent(doc);
                if (client.getConnectionInfo().getDeleteProcessedFile()) {
                    client.del(filePattern);
                }
            } catch (final IOException e) {
                FTPUtil.setIOFaultOnExchange(exchange, "Can not get file [" + filePattern
                        + "] on FTP server [" + client.getConnectionInfo().getServer()
                        + "]. Cause : " + e.getMessage());
            }
        }
        if (this.logger.isLoggable(Level.FINEST)) {
            this.logger.finest("GET operation ended");
        }
    }

    /**
     * 
     * FTP GET operation <br>
     * get the file as a source message a Fault is returned for bad request or
     * IO problems
     * 
     * @param exchange
     *            the message exchange, MUST be non null
     * @param connectionInfo
     * @param inMessage
     * @throws MessagingException
     *             bad MEP
     */
    public void processGetAsAttachment(final Exchange exchange, final Document inMessage,
            final WrappedFTPClient client) throws MessagingException {
        this.logger.info("JBI exchange received - Process GET AS ATTACHMENT operation");
        // Checks the MEP of the incoming message
        if (!exchange.isInOutPattern()) {
            throw new MessagingException(FTPUtil.getOperationValidMep(
                    FTPConstants.GET_ATTACHMENT_OPERATION, Message.MEPConstants.IN_OUT_PATTERN));
        }

        // retrieve the filename to get
        final Node filePatternNode = XMLHelper.findChild(inMessage.getFirstChild(),
                null, FTPConstants.FTP_FILENAME, true);
        String filePattern = null;
        if (filePatternNode != null) {
            filePattern = filePatternNode.getTextContent();
        }
        if (StringHelper.isNullOrEmpty(filePattern)) {
            FTPUtil.setMissingElementFaultOnExchange(exchange, FTPConstants.FTP_FILENAME);
        } else {
            try {
                // do the FTP GET operation
                final DataHandler file = client.getFileAsAttachment(filePattern);
                exchange.setOutMessageAttachment(file.getName(), file);
                final Document outDoc = FTPUtil.generateMTOMResponse(file.getName(),
                        exchange.getOperation());
                exchange.setOutMessageContent(outDoc);
                if (client.getConnectionInfo().getDeleteProcessedFile()) {
                    client.del(file.getName());
                }
            } catch (final IOException e) {
                FTPUtil.setIOFaultOnExchange(exchange, "Can not get file [" + filePattern
                        + "] on FTP server [" + client.getConnectionInfo().getServer()
                        + "]. Cause : " + e.getMessage());
            }
        }
        this.logger.info("GET AS ATTACHMENT operation ended");
    }

    /**
     * FTP MGET operation <br>
     * All files are getted as attachments
     * 
     * @param exchange
     * @param connectionInfo
     * @param inMessage
     * @throws Exception
     */
    public void processMGet(final Exchange exchange, final Document inMessage,
            final WrappedFTPClient client) throws MessagingException {
        if (this.logger.isLoggable(Level.INFO)) {
            this.logger.info("JBI exchange received - Process MGET operation");
        }
        if (!exchange.isInOutPattern()) {
            throw new MessagingException(FTPUtil.getOperationValidMep(FTPConstants.MGET_OPERATION,
                    MEPConstants.IN_OUT_PATTERN));
        }
        // retrieve filenames to get
        final NodeList list = inMessage.getFirstChild().getChildNodes();
        final List<String> filenames = new LinkedList<String>();
        for (int index = 0; index < list.getLength(); index++) {
            if (FTPConstants.FTP_FILENAME.equalsIgnoreCase(list.item(index).getLocalName())) {
                filenames.add(list.item(index).getTextContent());
            }
        }
        if (filenames.size() == 0) {
            FTPUtil.setMissingElementFaultOnExchange(exchange, FTPConstants.FTP_FILENAME);
        } else {
            try {
                // do the FTP MGET operation
                final Map<String, DataHandler> files = client.mGet(filenames);
                // add attachments on the OUT message
                for (Entry<String, DataHandler> entry : files.entrySet()) {
                    exchange.getOutMessage().addAttachment(entry.getKey(), entry.getValue());
                }
                final List<String> retrievedFilenames = new LinkedList<String>(files.keySet());
                final Document outDoc = FTPUtil.generateMTOMListResponse(retrievedFilenames,
                        exchange.getOperation());
                exchange.setOutMessageContent(outDoc);
                if (client.getConnectionInfo().getDeleteProcessedFile()) {
                    client.mDel(retrievedFilenames);
                }
            } catch (final IOException e) {
                FTPUtil.setIOFaultOnExchange(exchange, "Can not get files [" + filenames
                        + "] on FTP server [" + client.getConnectionInfo().getServer()
                        + "]. Cause : " + e.getMessage());
            }
        }
        if (this.logger.isLoggable(Level.FINEST)) {
            this.logger.finest("MGET operation ended");
        }
    }

    /**
     * Put attachments on the ftp server TODO an FTP connection is done for each
     * file...
     * 
     * @param exchange
     * @param connectionInfo
     * @throws MessagingException
     */
    public void processMPut(final Exchange exchange, final Document inputDocument,
            final WrappedFTPClient client) throws MessagingException {
        if (this.logger.isLoggable(Level.INFO)) {
            this.logger.info("JBI exchange received - Process MPUT operation");
        }
        // Checks the MEP of the incoming message
        if (!exchange.isInOnlyPattern()) {
            throw new MessagingException(FTPUtil.getOperationValidMep(FTPConstants.MPUT_OPERATION,
                    MEPConstants.IN_ONLY_PATTERN));
        }

        // Check if the message has attachements
        if (exchange.getInMessageAttachments().isEmpty()) {
            final StringBuilder errorSB = new StringBuilder();
            errorSB.append("One or more attachments expected.");
            throw new MessagingException(errorSB.toString());
        }
        try {
            // rebuild a hashmap containing files and their names
            MtomMapping mapping = MtomUtil.getMtomMapping(exchange,
                    inputDocument.getDocumentElement());
            final Map<String, DataHandler> attachments = mapping.getContentIdToDataHandler();
            if (this.logger.isLoggable(Level.FINE)) {
                this.logger.fine(mapping.toString());
            }
            client.mPut(attachments);
        } catch (IOException e) {
            final StringBuilder errorSB = new StringBuilder();
            errorSB.append("Can not put files on FTP server [");
            errorSB.append(client.getConnectionInfo().getServer());
            errorSB.append("] : IOException : ");
            errorSB.append(e.getMessage());
            throw new MessagingException(errorSB.toString());
        }
        this.logger.info("MPUT operation ended");
    }

    /**
     * Put the source on the ftp server with the given filename a Fault is
     * returned on IO problems TODO the content to send is a String, which is
     * not optimal
     * 
     * @param exchange
     * @param fileName
     * @param connectionInfo
     * @param inMessage
     * @throws MessagingException
     * @throws MissingElementException
     */
    public void processPut(final Exchange exchange, final Document inMessage,
            final WrappedFTPClient client) throws MessagingException, MissingElementException {
        if (exchange == null) {
            throw new MessagingException("exchange parameter is null");
        }

        if (this.logger.isLoggable(Level.INFO)) {
            this.logger.info("JBI exchange received - Process PUT operation");
        }

        // Checks the MEP of the incoming message
        if (!exchange.isInOnlyPattern()) {
            throw new MessagingException(FTPUtil.getOperationValidMep(FTPConstants.PUT_OPERATION,
                    MEPConstants.IN_ONLY_PATTERN));
        }
        // Check the root node
        final Node root = inMessage.getFirstChild();
        if (root == null || !FTPConstants.PUT_OPERATION.equalsIgnoreCase(root.getLocalName())) {
            throw new MissingElementException(FTPConstants.PUT_OPERATION);
        }

        final Node fileNameNode = XMLHelper.findChild(root, null, FTPConstants.FTP_FILENAME, false);
        if (fileNameNode == null
                || !FTPConstants.FTP_FILENAME.equalsIgnoreCase(fileNameNode.getLocalName())) {
            throw new MissingElementException(FTPConstants.FTP_FILENAME);
        }
        final Node bodyNode = XMLHelper.findChild(root, null, FTPConstants.BODY, false);
        if (bodyNode == null) {
            throw new MissingElementException(FTPConstants.BODY);
        }
        try {
            String body = XMLHelper.toString(bodyNode.getChildNodes());
            client.putString(fileNameNode.getTextContent(), body);
        } catch (final IOException e) {
            final StringBuilder errorSB = new StringBuilder();
            errorSB.append("Can not put XML content on FTP server [");
            errorSB.append(client.getConnectionInfo().getServer());
            errorSB.append("] : IOException : ");
            errorSB.append(e.getMessage());
            throw new MessagingException(errorSB.toString());
        } catch (final DOMException e) {
            final StringBuilder errorSB = new StringBuilder();
            errorSB.append("Can not put XML content on FTP server [");
            errorSB.append(client.getConnectionInfo().getServer());
            errorSB.append("] : DOMException : ");
            errorSB.append(e.getMessage());
            throw new MessagingException(errorSB.toString());
        }
        if (this.logger.isLoggable(Level.FINEST)) {
            this.logger.finest("PUT operation ended");
        }
    }
}
