/**
 * PETALS - PETALS Services Platform.
 * Copyright (c) 2007 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * $Id$
 * -------------------------------------------------------------------------
 */

package org.ow2.petals.component.framework.api;

import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.ow2.petals.component.framework.api.exception.PEtALSCDKException;
import org.ow2.petals.component.framework.api.message.Exchange;
import org.ow2.petals.component.framework.jbidescriptor.generated.Consumes;
import org.ow2.petals.component.framework.jbidescriptor.generated.Provides;

/**
 * An abstract class for JBI message interceptor.
 * 
 * @author chamerling - EBM WebSourcing
 * @since 1.0
 * 
 */
public abstract class Interceptor {

    /**
     * The JBI component of the interceptor (to enable access to all the
     * component context)
     */
    protected PetalsComponent component;

    /**
     * The name of the interceptor (defined in the component descriptor). This
     * name will be used in the service units as reference;
     */
    protected String name;

    /**
     * The interceptor attributes (defined in the component descriptor). These
     * parameters are the initial ones and are not mutable.
     */
    protected Map<String, String> componentParameters;

    /**
     * Interceptor is always activated ie it is activated for all service units
     * on all phases (send and accept)?
     */
    protected boolean active = false;

    /**
     * The component logger
     */
    protected Logger logger;

    /**
     * 
     * Creates a new instance of {@link Interceptor}
     * 
     */
    public Interceptor() {
    }

    /**
     * Initialize the interceptor from its parameters. The parameters are
     * created from the interceptor parameters defined in the JBI descriptor.
     * 
     * The interceptor developer must implement this method if he want to
     * initialize its interceptor from the defined parameters.
     * 
     * @param parameters
     *      could not be null. The parameters are loaded by the CDK. If no
     *      parameter has been found, the map is empty but not null.
     */
    public final void init(final PetalsComponent component, final String name,
            final boolean active,
            final Map<String, String> componentParameters, final Logger logger) throws PEtALSCDKException {
        this.component = component;
        this.componentParameters = componentParameters;
        this.logger = logger;
        this.name = name;
        this.active = active;

        try {
            this.onInit();
        } catch (final Exception e) {
            logger.log(Level.WARNING, "Problem during interceptor initialization");
            throw new PEtALSCDKException("Interceptor initialization failure");
        }
    }

    /**
     * Initialization callback. The developer can do extra interceptor
     * configuration here (create configuration objects from parameters,
     * register MBeans...)
     * 
     * @throws Exception
     *      an exception can be raised by the interceptor developer. If an
     *      exception is raised, the interceptor will not be available in the
     *      component.
     */
    protected void onInit() throws Exception {
    }

    /**
     * Method called before the consumer of service send a request.
     * 
     * @param exchange
     * @param sonsumes
     *      The {@code Consumes} configuration
     * 
     * @param additionalParameters
     *      the parameters defined in the SU for this interceptor. The user can
     *      use this map to retrieve the parameters defined in the Consume or
     *      Provide extensions.
     */
    public abstract void handleMessageSent(Exchange exchange, Consumes consumes,
            Map<String, String> additionalParameters);

    /**
     * Method called before the consumer of service receive its response.
     * 
     * @param exchange
     * @param consumes
     *      The {@code Consumes} configuration
     * 
     * @param additionalParameters
     *      the parameters defined in the SU for this interceptor. The user can
     *      use this map to retrieve the parameters defined in the Consume or
     *      Provide extensions.
     */
    public abstract void handleMessageSentResponse(Exchange exchange, Consumes consumes,
            Map<String, String> additionalParameters);

    /**
     * Method called before the provider of service receive its request.
     * 
     * @param exchange
     * @param provides
     *      The Provides configuration
     * @param additionalParameters
     *      the parameters defined in the SU for this interceptor. The user can
     *      use this map to retrieve the parameters defined in the Consume or
     *      Provide extensions.
     */
    public abstract void handleMessageAccept(Exchange exchange, Provides provides,
            Map<String, String> additionalParameters);

    /**
     * Method called before the provider of service send its response.
     * 
     * @param exchange
     * @param provides
     *      The Provides configuration
     * @param additionalParameters
     *      the parameters defined in the SU for this interceptor. The user can
     *      use this map to retrieve the parameters defined in the Consume or
     *      Provide extensions.
     */
    public abstract void handleMessageAcceptResponse(Exchange exchange, Provides provides,
            Map<String, String> additionalParameters);

    /**
     * @return the name
     */
    public final String getName() {
        return this.name;
    }

    /**
     * @return the active
     */
    public boolean isActive() {
        return this.active;
    }

    /**
     * @param active
     *      the active to set
     */
    public void setActive(final boolean active) {
        this.active = active;
    }

    /**
     * @return the parameters
     */
    public Map<String, String> getComponentParameters() {
        return this.componentParameters;
    }
}
