package org.ow2.petals.component.framework.api.util;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.activation.DataHandler;
import javax.jbi.messaging.MessagingException;
import javax.jbi.messaging.NormalizedMessage;
import javax.xml.namespace.QName;

import org.ow2.petals.component.framework.api.exception.PEtALSCDKException;
import org.ow2.petals.component.framework.api.message.Exchange;
import org.w3c.dom.Document;
import org.w3c.dom.DocumentFragment;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/***
 * Contains utility to manipulate MTOM attachments
 * 
 * @author ”Mathieu CARROLLE - mathieu.carrolle@petalslink.com”
 * @author Vincent Zurczak - EBM WebSourcing
 */
public final class MtomUtil {

    /**
     * A bean that describes a mapping of attachments associated with a Petals
     * message.
     * <p>
     * An instance of this class aims at:
     * </p>
     * <ul>
     * <li>Associating a content ID with a data handler.</li>
     * <li>Indicating content IDs that could be mapped to a data handlers
     * (missing data-handler).</li>
     * <li>Indicating data handlers that are not referenced from the pay-load
     * (data handlers that may be ignored).</li>
     * </ul>
     */
    public static class MtomMapping {

        private final Map<String, DataHandler> contentIdToDataHandler;

        private final Set<String> orphanContentIds;

        private final List<DataHandler> orphanDataHandlers;

        /**
         * Constructor.
         */
        public MtomMapping() {
            this.contentIdToDataHandler = new HashMap<String, DataHandler>();
            this.orphanContentIds = new HashSet<String>();
            this.orphanDataHandlers = new ArrayList<DataHandler>();
        }

        /**
         * @return a map associating a content ID (as specified in the XOP's
         *         cid) and a data handler.
         *         <p>
         *         This map is never null, but can be empty.
         *         </p>
         */
        public Map<String, DataHandler> getContentIdToDataHandler() {
            return this.contentIdToDataHandler;
        }

        /**
         * @return the list of content IDs for which no data handler was found.
         *         <p>
         *         This list is never null, but can be empty.<br />
         *         Every missing data handler implies there is an error,
         *         something is missing in the pay-load.
         *         </p>
         */
        public Set<String> getOrphanContentIds() {
            return this.orphanContentIds;
        }

        /**
         * @return the list of data handlers that are not associated with a
         *         content ID.
         *         <p>
         *         This list is never null, but can be empty.<br />
         *         Every orphan data handler implies there is an error on the
         *         client side (SwA attachments are part of the pay-load). But
         *         this should not prevent the processing from being done. This
         *         method aims at describing the coherence of the pay-load with
         *         the attachments (and can be used for logging reasons).
         *         </p>
         */
        public List<DataHandler> getOrphanDataHandlers() {
            return this.orphanDataHandlers;
        }

        /**
         * @param o
         *            a data handler which is associated with no content ID
         * @return true if the data handler was not already in the list
         * @see java.util.List#add(java.lang.Object)
         */
        boolean add(DataHandler o) {
            return this.orphanDataHandlers.add(o);
        }

        /**
         * @param o
         *            a content ID, as specified by the XOP's cid, which is
         *            associated with no data handler.
         * @return true, if this content ID was not already stored
         * @see java.util.Set#add(java.lang.Object)
         */
        boolean add(String o) {
            return this.orphanContentIds.add(o);
        }

        /**
         * @param c
         *            a collection of data handlers which are associated with no
         *            content ID
         * @return if this list changed as a result of the call
         * @see java.util.List#add(java.lang.Object)
         */
        boolean addAll(Collection<? extends DataHandler> c) {
            return this.orphanDataHandlers.addAll(c);
        }

        /**
         * @param key
         *            a content ID, as specified by the XOP's cid
         * @param value
         *            the associated data handler
         * @return the previous value that was associated with this content ID
         * @see java.util.Map#put(java.lang.Object, java.lang.Object)
         */
        DataHandler put(String key, DataHandler value) {
            return this.contentIdToDataHandler.put(key, value);
        }

        /**
         * This method build a message indicating the listing of orphan data
         * handler's name, orphan content ID and the map associating content ID
         * to DataHandler.<br/>
         */
        @Override
        public String toString() {
            final StringBuilder result = new StringBuilder();
            if (this.orphanContentIds.size() != 0) {
                result.append("Orphan Content Id : ");
                result.append(this.orphanContentIds.toString());
                result.append("\n");
            }
            if (this.orphanDataHandlers.size() != 0) {
                result.append("Orphan data handler name : [");
                for (DataHandler dtHandler : this.orphanDataHandlers) {
                    result.append(dtHandler.getName() + ",");
                }
                result.replace(result.length() - 1, result.length(), "]");
                result.append("\n");
            }
            if (this.contentIdToDataHandler.size() != 0) {
                result.append("Map associating content ID to DataHandler :");
                result.append(this.contentIdToDataHandler.toString());
            }
            return result.toString();
        }
    }

    /**
     * Include element namespace
     */
    public static final String MTOM_NSURI = "http://www.w3.org/2004/08/xop/include";

    /**
     * Element : "attachments"
     */
    public static final String NODE_ATTACHMENTS = "attachments";

    /**
     * Compares two strings, the first one being encoded as an URL and not the
     * second.
     * <p>
     * This method is used to compare attachment names.<br />
     * </p>
     * 
     * @param attachmentElementValue
     *            the value of the MTOM attachment, as referenced in the href
     *            attribute
     * @param attachmentName
     *            the attachment name, as specified in the exchange
     * @return true if the arguments are the same (despite the encoding), false
     *         otherwise
     * @throws UnsupportedEncodingException
     */
    public static boolean compare(String attachmentElementValue, String attachmentName) {
        if (attachmentElementValue.equals(attachmentName))
            return true;
        try {
            // In some cases, the attachment ID is encoded as an URL
            String encodedText2 = URLEncoder.encode(attachmentName, "UTF-8");
            if (attachmentElementValue.equals(encodedText2))
                return true;

            // Last attempt
            String encodedText1 = attachmentElementValue.replace("@", "%40");
            if (encodedText1.equals(encodedText2))
                return true;
        } catch (UnsupportedEncodingException e) {

        }
        return false;
    }

    /***
     * Return a string representation of the given {@link QName}. Name with
     * prefix is returned if the given {@link QName} contains prefix otherwise
     * return just the localpart.
     * 
     * @param qname
     * @return prefix:localpart or localpart
     */
    public final static String convertQNameToString(final QName qname) {
        return qname.getPrefix().length() > 0 ? qname.getPrefix() + ":" + qname.getLocalPart()
                : qname.getLocalPart();
    }

    /***
     * Extract element containing xop element.
     * 
     * @param inputDocument
     *            Input xml message.
     * @return The xml element "attachments".
     * @deprecated useless method
     */
    @Deprecated
    public static Element extractAttachmentRootElement(Document inputDocument)
            throws MessagingException {
        final Element attachmentsRootElement;
        final NodeList attachmentsCandidatesList = inputDocument.getElementsByTagNameNS("*",
                MtomUtil.NODE_ATTACHMENTS);
        if (attachmentsCandidatesList.getLength() != 1)
            throw new MessagingException("xml element : '" + NODE_ATTACHMENTS + "' not found.");
        attachmentsRootElement = (Element) attachmentsCandidatesList.item(0);
        return attachmentsRootElement;
    }

    /**
     * In the message payload, extract the identifier(cid) associated with each
     * xop:Include element. <br/>
     * Document need to be namespaceaware.
     * 
     * @param inputDocument
     *            Input xml message.
     * @param exchange
     * @return List of all cid found in the message
     */
    public static Set<String> extractAttachmentsIdFromPayload(Element attachmentRootElement) {
        final Set<String> attachmentIds = new HashSet<String>();
        String attachmentId = null;
        // MTOM
        final NodeList includeNodes = attachmentRootElement.getElementsByTagNameNS(MTOM_NSURI, "*");
        for (int i = 0; i < includeNodes.getLength(); i++) {
            final Node n = includeNodes.item(i);
            if ("include".equalsIgnoreCase(getNodeName(n))) {
                attachmentId = ((Element) n).getAttribute("href").replaceFirst("cid:", "");
                attachmentIds.add(attachmentId);
            }
        }
        return attachmentIds;
    }

    /**
     * Build the xml structure according to xop standard.
     * 
     * @param document
     *            Document used to create node.
     * @param contentIds
     *            List of attachment id.
     * @param element
     *            QName representing the element that encapsulate the
     *            xop:Include element.
     * @return Return an xml structure according to xop standard. <br/>
     *         Exemple : <br/>
     *         &lt;tns:file&gt;<br/>
     *         <div style="margin-left:30px;"> &lt;xop:Include
     *         xmlns:xop="http://www.w3.org/2004/08/xop/include"
     *         href="cid:MyAttachmentId1"/&gt;</div> &lt;/tns:file&gt;<br/>
     *         &lt;tns:file&gt;<br/>
     *         <div style="margin-left:30px;">&lt;xop:Include
     *         xmlns:xop="http://www.w3.org/2004/08/xop/include"
     *         href="cid:MyAttachmentId2"/&gt;</div> &lt;/tns:file&gt;
     */
    public static DocumentFragment generateMtomStructure(final Document document,
            final List<String> contentIds, final QName element) {
        final DocumentFragment fragment = document.createDocumentFragment();
        final String qnameAsString = convertQNameToString(element);
        Element elt = null;
        for (final String cId : contentIds) {
            elt = document.createElementNS(element.getNamespaceURI(), qnameAsString);
            elt.appendChild(generateMtomXopElement(document, cId));
            fragment.appendChild(elt);
        }
        return fragment;
    }

    /**
     * Build the xml structure according to xop standard.
     * 
     * @param document
     *            Document used to create node.
     * @param contentIds
     *            List of attachment id.
     * @param rootNode
     *            QName representing the root element.
     * @param customNode
     *            QName representing an additional xml element parent to include
     *            for each xop:Include element
     * @return Return an xml structure according to xop standard. <br/>
     *         Exemple<br/>
     *         &lt;tns:<b>rootNode</b>&gt;<br>
     *         <div style="margin-left:20px;">&lt;tns:<b>customNode</b>&gt; <div
     *         style="margin-left:20px;">&lt;xop:Include
     *         xmlns:xop="http://www.w3.org/2004/08/xop/include"
     *         href="cid:MyAttachmentId1"
     *         /&gt;</div>&lt;/tns:<b>customNode</b>&gt;</div> <div
     *         style="margin-left:20px;">&lt;tns:<b>customNode</b>&gt; <div
     *         style="margin-left:20px;">&lt;xop:Include
     *         xmlns:xop="http://www.w3.org/2004/08/xop/include"
     *         href="cid:MyAttachmentId2"
     *         /&gt;</div>&lt;/tns:<b>customNode</b>&gt;</div>
     *         &lt;/tns:<b>rootNode</b>&gt;
     */
    public static Node generateMtomStructure(Document document, List<String> contentIds,
            QName rootNode, QName customNode) {
        final Element rootElt = document.createElementNS(rootNode.getNamespaceURI(),
                convertQNameToString(rootNode));
        final DocumentFragment docFragment = generateMtomStructure(document, contentIds, customNode);
        rootElt.appendChild(docFragment);
        return rootElt;
    }

    /**
     * Generate an Include element according to xop standard. <br />
     * Example : &lt;xop:Include
     * xmlns:xop='http://www.w3.org/2004/08/xop/include'
     * href='cid:MYCONTENTID'/>
     * 
     * @param doc
     *            Document used to create node.
     * @param contentId
     *            attachment ID
     * @return
     */
    public static Node generateMtomXopElement(Document doc, String contentId) {
        Element includeElt = doc.createElementNS(MTOM_NSURI, "xop:Include");
        includeElt.setAttribute("href", "cid:" + contentId);
        return includeElt;
    }

    /**
     * Return all attachments by using Xml-binary Optimized Packaging (XOP)
     * standard. All xop:Include element need to be child of an elements named
     * attachments.
     * 
     * @param inputDocument
     * @param exchange
     * @throw {@link PEtALSCDKException} if :<br>
     *        -no attachment id was found in the message payload<br>
     *        -the attachments id defined in the message payload doesn't match
     *        the attachments id defined in the normalized message<br>
     * @return
     * @deprecated use getMtomMapping(Exchange exchange, Element element)
     */
    @Deprecated
    public static Map<String, DataHandler> getAttachments(Document inputDocument,
            NormalizedMessage nmMessage) throws MessagingException {
        final Map<String, DataHandler> mAttachments = new HashMap<String, DataHandler>();

        final Element attachmentRootElement = extractAttachmentRootElement(inputDocument);
        final Set<String> attachmentIds = extractAttachmentsIdFromPayload(attachmentRootElement);
        if (attachmentIds.size() == 0)
            throw new MessagingException("no attachment id was found in the message payload");
        for (String id : attachmentIds) {
            for (Object obj : nmMessage.getAttachmentNames()) {
                final String jbiAttachmentId = (String) obj;
                if (compare(id, jbiAttachmentId))
                    mAttachments.put(jbiAttachmentId, nmMessage.getAttachment(jbiAttachmentId));
            }
        }
        if (mAttachments.size() != attachmentIds.size()) {
            final StringBuilder builder = new StringBuilder();
            builder.append("error when comparing the attachments id" + " | ");
            builder.append("payload attachment id found : " + attachmentIds.toString() + " | ");
            builder.append("normalized message attachment id found : "
                    + nmMessage.getAttachmentNames().toString());
            throw new MessagingException(builder.toString());
        }
        return mAttachments;
    }

    /**
     * Builds a MTOM mapping from both an exchange and a root element.
     * 
     * @param exchange
     *            the message exchange
     * @param element
     *            the element from which MTOM elements must be searched.
     *            <p>
     *            MTOM elements are elements associated with the name space
     *            <code>http://www.w3.org/2004/08/xop/include</code>.<br />
     *            If null, the root element of the pay-load is taken.<br />
     *            If you already have extracted the pay-load as a document, you
     *            can use <code>Document#getDocumentElement()</code>.
     *            </p>
     * 
     * @return null if the element is null and could not be built, a
     *         {@link MtomMapping} instance otherwise
     * @throws MessagingException
     *             if something went wrong while acquiring the message content
     */
    public static MtomMapping getMtomMapping(Exchange exchange, Element element)
            throws MessagingException {

        MtomMapping result = null;

        // Need an element? Take the root of the pay-load.
        if (element == null) {
            Document doc = exchange.getInMessageContentAsDocument(true);
            if (doc != null && doc.getDocumentElement() != null)
                element = doc.getDocumentElement();
        }

        // Get the mapping
        if (element != null) {
            result = new MtomMapping();

            // Get all the content IDs from the pay-load
            Set<String> contentIds = MtomUtil.extractAttachmentsIdFromPayload(element);
            List<DataHandler> allTheDataHandlers = new ArrayList<DataHandler>();
            for (String attName : exchange.getInMessageAttachmentNames()) {
                allTheDataHandlers.add(exchange.getInMessageAttachment(attName));
            }

            // For each content ID, find the associated data handler.
            // Keep a trace of orphan data handlers.
            for (String contentId : contentIds) {
                DataHandler dataHandler = null;
                for (String attName : exchange.getInMessageAttachmentNames()) {
                    if (MtomUtil.compare(contentId, attName)) {
                        dataHandler = exchange.getInMessageAttachment(attName);
                        break;
                    }
                }

                if (dataHandler != null) {
                    allTheDataHandlers.remove(dataHandler);
                    result.put(contentId, dataHandler);
                } else {
                    result.add(contentId);
                }
            }

            // Do not forget the orphan data handlers
            result.addAll(allTheDataHandlers);
        }

        return result;
    }

    /**
     * Return the node name
     * 
     * @param node
     * @return the name of the XML node (with no name space element)
     */
    public static String getNodeName(Node node) {
        String name = node.getNamespaceURI() != null ? node.getLocalName() : node.getNodeName();
        if (name.contains(":")) { //$NON-NLS-1$
            String[] parts = name.split(":"); //$NON-NLS-1$
            name = parts[parts.length - 1];
        }
        return name;
    }
}
