/**
 * PETALS - PETALS Services Platform. Copyright (c) 2007 EBM Websourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * -------------------------------------------------------------------------
 * $Id$
 * -------------------------------------------------------------------------
 */

package org.ow2.petals.component.framework.api.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.util.Properties;

import com.ebmwebsourcing.easycommons.io.IOHelper;
import com.ebmwebsourcing.easycommons.properties.PropertiesException;

/**
 * Utility to handle a CDK properties file.
 * 
 * @author ofabre - EBM Websourcing
 * @since 1.0
 */
public class PropertiesUtil {

    /**
     * <p>
     * Load a {@link Properties} object from a properties file path. The file
     * path can be relative path based on Petals root path or a valid URL.
     * </p>
     * <p>
     * Exemples : <br/>
     * Relative path : /conf/external.properties <br/>
     * Absolute path : file:///C:/conf/external.properties <br/>
     * URL : ftp://user:pwd@host:port/external.properties
     * </p>
     * 
     * @param propertiesFilePath
     *            The file path can be relative path based on Petals root path
     *            or a valid URL
     * @return a {@link Properties} object fill with properties value found in
     *         the targeted file
     * @throws PropertiesException
     *             if an error occurs when reading external properties file
     */
    public static final Properties loadProperties(final String propertiesFilePath)
            throws PropertiesException {
        final Properties properties = new Properties();

        if (propertiesFilePath != null) {
            InputStream propFileStream = null;
            try {
                propFileStream = loadPropertiesFileFromUrl(new URL(propertiesFilePath));
            } catch (final MalformedURLException e) {
                propFileStream = loadPropertiesFileFromRelativePath(propertiesFilePath);
            }
            if (propFileStream != null) {
                try {
                    properties.load(propFileStream);
                } catch (final IOException e) {
                    String message = String.format("Cannot find properties at location : %s",
                            propertiesFilePath);
                    throw new PropertiesException(message);
                } finally {
                    IOHelper.close(propFileStream);
                }
            } else {
                String message = String.format("Cannot find properties at location : %s",
                        propertiesFilePath);
                throw new PropertiesException(message);
            }
        }

        return properties;
    }

    private static final InputStream loadPropertiesFileFromRelativePath(
            final String propertiesFilePath) throws PropertiesException {
        InputStream is = null;

        final String petalsHome = System.getenv("PETALS_HOME");
        if (petalsHome != null) {
            final File propFile = new File(petalsHome, propertiesFilePath);
            try {
                is = new FileInputStream(propFile);
            } catch (final FileNotFoundException fnfe) {
                String message = String.format("Cannot load properties from the file %s",
                        propertiesFilePath);
                throw new PropertiesException(message, fnfe);
            }
        } else {
            throw new PropertiesException("PETALS_HOME is not defined");
        }

        return is;
    }

    private static final InputStream loadPropertiesFileFromUrl(final URL url)
            throws PropertiesException {
        InputStream is = null;

        try {
            final URLConnection connection = url.openConnection();
            is = connection.getInputStream();
        } catch (final IOException ioe) {
            String message = String.format("Cannot load properties from URL %s", url);
            throw new PropertiesException(message, ioe);
        }

        return is;
    }
}
