/****************************************************************************
 *
 * Copyright (c) 2008-2012, EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA 
 *
 *****************************************************************************/
 
package org.ow2.petals.component.framework.api.message;

import java.io.InputStream;
import java.net.URI;
import java.util.Map;
import java.util.Set;

import javax.activation.DataHandler;
import javax.jbi.messaging.ExchangeStatus;
import javax.jbi.messaging.Fault;
import javax.jbi.messaging.FlowAttributes;
import javax.jbi.messaging.MessageExchange;
import javax.jbi.messaging.MessagingException;
import javax.jbi.messaging.NormalizedMessage;
import javax.jbi.servicedesc.ServiceEndpoint;
import javax.security.auth.Subject;
import javax.xml.namespace.QName;
import javax.xml.transform.Source;

import org.ow2.petals.component.framework.api.Constants;
import org.w3c.dom.Document;

/**
 * The MessageExchange wrapper to handle properly MessageExchange against the CDK.
 * @author Roland Naudin - EBM WebSourcing
 * @author Christophe Hamerling - EBM WebSourcing
 */
public interface Exchange {

    static final QName DEFAULT_FAULT_QNAME = new QName("http://petals.ow2.org/cdk/defaultFault",
            "0");

    static final String FAULT_MESSAGE_NAME = "fault";

    static final String IN_MESSAGE_NAME = "in";

    static final String OUT_MESSAGE_NAME = "out";

    static final String PROPERTY_PROVIDER_LINKTYPE = "org.ow2.petals.router.provider.linktype";

    void addInMessageAttachment(String name, DataHandler dataHandler) throws MessagingException;

    void addOutMessageAttachment(String name, DataHandler dataHandler) throws MessagingException;

    Fault createFault() throws MessagingException;

    ServiceEndpoint getEndpoint();

    /**
     * Get the endpoint name specified in the message exchange
     * 
     * @return the endpoint name
     * @throws MessagingException
     *             if no Endpoint is set
     */
    String getEndpointName() throws MessagingException;

    Exception getError();

    String getExchangeId();

    /**
     * Get the message exchange Pattern
     * 
     * @return the URI pattern. Can't be null
     * @throws MessagingException
     *             if no message exchange pattern is set
     */
    URI getExchangePattern() throws MessagingException;

    /**
     * Get the exchange fault
     * 
     * @return the exchange fault
     */
    Fault getFault();

    /**
     * Get the addressing data from the IN message. Key is the addressing name
     * (Values are available in {@link Constants.Addressing}), value is the URL
     * as String...
     * 
     * @return null if there is no addressing information in the input message
     */
    Map<QName, String> getInAddressing();

    /**
     * Extract incoming NormalizedMessage from given MessageExchange. Create the
     * message if it is null.
     * 
     * @return the incoming NormalizedMessage
     * 
     */
    NormalizedMessage getInMessage();

    /**
     * Get the "In" attachment matching the given name
     * 
     * @param name an attachment name
     * @return the "In" attachment matching the given name
     */
    DataHandler getInMessageAttachment(String name) throws MessagingException;

    /**
     * Get all the "In" attachment names
     * 
     * @return the "In" attachment names
     * @throws MessagingException
     */
    Set<String> getInMessageAttachmentNames() throws MessagingException;

    /**
     * Get the "In" attachments in a {@link Set}
     * 
     * @param exchange
     * @return the "In" attachments in a {@link Set}
     */
    Set<DataHandler> getInMessageAttachments() throws MessagingException;

    /**
     * Get the content of the "in" message as a DOM document. <br>
     * If the {@link Source} is a {@link Stream} or {@link Document}, a fork is
     * processed, which makes the {@link Source} still available, but can
     * decreases the performance.
     * 
     * @return the DOM document representation of the "in" message content
     *@throws MessagingException
     *             if error occurred during conversion of the Source or if "in"
     *             message is {@code null}
     */
    Document getInMessageContentAsDocument() throws MessagingException;

    /**
     * Get the content of the "in" message as a DOM document. <br>
     * If the {@link Source} is a {@link Stream} or {@link Document} and {@code
     * forkSource} is {@code true}, a fork is processed, which makes the
     * {@link Source} still available, but can decreases the performance.
     * 
     * @return the DOM document representation of the "in" message content
     *@throws MessagingException
     *             if error occurred during conversion of the Source or if "in"
     *             message is {@code null}
     */
    Document getInMessageContentAsDocument(boolean forkSource) throws MessagingException;

    /**
     * Get the content of the "in" message as a Source
     * 
     * @return the "in" message content
     * @throws MessagingException
     *             if "in" message is {@code null}
     */
    Source getInMessageContentAsSource() throws MessagingException;

    /**
     * 
     * @param name
     * @return
     * @throws MessagingException
     */
    Object getInMessageProperty(String name) throws MessagingException;

    /**
     * 
     * @return
     * @throws MessagingException
     */
    Set<String> getInMessagePropertyNames() throws MessagingException;

    /**
     * Get "In" security subject
     * 
     * @return the subject or {@code null} if no subject
     */
    Subject getInSubject() throws MessagingException;

    QName getInterfaceName();

    QName getOperation();

    /**
     * Get the operation name (local part of the operation QName)
     * 
     * @return the operation name, NULL if no operation is set
     */
    String getOperationName();

    /**
     * Extract outgoing NormalizedMessage from given MessageExchange. Create the
     * output message if message is null.
     * 
     * @return the outgoing NormalizedMessage
     * 
     */
    NormalizedMessage getOutMessage();

    /**
     * Get the "Out" attachment matching the attachment name.
     * 
     * @param exchange
     * @return
     */
    DataHandler getOutMessageAttachment(String name) throws MessagingException;

    /**
     * Get all the "Out" attachment names.
     * 
     * @return
     * @throws MessagingException
     */
    Set<String> getOutMessageAttachmentNames() throws MessagingException;

    /**
     * Get all the "Out" attachments in a {@link Set}.
     * 
     * @param exchange
     * @return
     */
    Set<DataHandler> getOutMessageAttachments() throws MessagingException;

    /**
     * Get the content of the "out" message as a DOM document. <br>
     * If the {@link Source} is a {@link Stream} or {@link Document}, a fork is
     * processed, which makes the {@link Source} still available, but can
     * decreases the performance.
     * 
     * @return the DOM document representation of the "in" message content
     *@throws MessagingException
     *             if error occurred during conversion of the Source or if "in"
     *             message is {@code null}
     */
    Document getOutMessageContentAsDocument() throws MessagingException;

    /**
     * Get the content of the "out" message as a DOM document. <br>
     * If the {@link Source} is a {@link Stream} or {@link Document} and {@code
     * forkSource} is {@code true}, a fork is processed, which makes the
     * {@link Source} still available, but can decreases the performance.
     * 
     * @return the DOM document representation of the "in" message content
     *@throws MessagingException
     *             if error occurred during conversion of the Source or if "in"
     *             message is {@code null}
     */
    Document getOutMessageContentAsDocument(boolean forkSource) throws MessagingException;

    /**
     * Get the content of the "out" message as a Source
     * 
     * @return the "out" message content
     * @throws MessagingException
     *             if "out" message is null
     */
    Source getOutMessageContentAsSource() throws MessagingException;

    Object getOutMessageProperty(String name) throws MessagingException;

    Set<?> getOutMessagePropertyNames() throws MessagingException;

    /**
     * Get "Out" security subject
     * 
     * @return the subject or {@code null} if no subject
     */
    Subject getOutSubject() throws MessagingException;

    URI getPattern();

    Object getProperty(String name);

    Set<?> getPropertyNames();

    MessageExchange.Role getRole();

    QName getService();

    ExchangeStatus getStatus();

    /**
     * Test if the message status is ACTIVE
     * 
     * @return true if the status is ACTIVE, otherwise false
     */
    boolean isActiveStatus();

    /**
     * Test if the message role is consumer
     * 
     * @return true is the message role is consumer
     */
    boolean isConsumerRole();

    /**
     * Test if the message status is DONE
     * 
     * @return true if the status is DONE, otherwise false
     */
    boolean isDoneStatus();

    /**
     * Test if the message status is ERROR
     * 
     * @return true if the status is ERROR, otherwise false
     */
    boolean isErrorStatus();

    /**
     * Test if an In message is set on the MessageExchange
     * 
     * @return true if an In message is set on the MessageExchange otherwise false
     */
    boolean isInMessage();

    boolean isInOnlyPattern();

    boolean isInOptionalOutPattern();

    boolean isInOutPattern();

    /**
     * Test if an out message is set on the MessageExchange
     * 
     * @return true if an out message is set on the MessageExchange otherwise false
     */
    boolean isOutMessage();

    /**
     * Test if the message role is provider
     * 
     * @return true is the message role is provider
     */
    boolean isProviderRole();

    boolean isRobustInOnlyPattern();

    boolean isTransacted();

    /**
     * Set the message exchange status to ACTIVE
     * 
     * @throws MessagingException
     */
    void setActiveStatus() throws MessagingException;

    /**
     * Set the message exchange status to DONE
     * 
     * @throws MessagingException
     */
    void setDoneStatus() throws MessagingException;

    void setEndpoint(ServiceEndpoint endpoint);

    /**
     * Put the given exception as Error part of the message exchange. Allowed
     * exception types are : java.*, org.w3c.* and org.xml.*.
     */
    void setError(Exception error);

    /**
     * Set the message exchange status to Error
     * 
     * @throws MessagingException
     */
    void setErrorStatus() throws MessagingException;

    void setFault(Fault fault) throws MessagingException;

    /**
     * Create a {@link Fault} with the given {@link Exception} and set it in the
     * {@link MessageExchange}
     * 
     * @param e
     *            the {@link Throwable} to transform in {@link Fault}
     * @throws MessagingException
     */
    void setFault(final Throwable e) throws MessagingException;

    /**
     * Set the addressing information in the IN message. An example of map data
     * is organized like this :
     * <ul>
     * <li>{http://www.w3.org/2005/08/addressing}To =
     * http://localhost:80804/petals/ws/ExampleService</li>
     * <li>{http://www.w3.org/2005/08/addressing}ReplyTo =
     * http://localhost:80804/petals/ws/ReplyService</li>
     * </ul>
     * Possible keys can be found in {@link Constants.Addressing} and values are
     * the URL as String. If the {@link NormalizedMessage} already contains the
     * addressing data, they will be overrided by the input map. If No
     * addressing data is available in the message, the input map will be the
     * addressing data.
     * 
     * @param map
     */
    void setInAddressing(Map<QName, String> map);

    /**
     * Set the input message
     * 
     * @param msg the input message
     * @throws MessagingException
     */
    void setInMessage(final NormalizedMessage msg) throws MessagingException;

    /**
     * Put the attachment in the "out" message. If the out normalized message is
     * null, it is created here.
     * 
     * @param name the attachment name
     * @param attachement the attachement data handler
     * @throws MessagingException
     */
    void setInMessageAttachment(String name, DataHandler attachement) throws MessagingException;

    /**
     * Put a Map of attachments in the out message of the message exchange. If
     * the out normalized message is null, it is created here.
     * 
     * @param attachements
     *            the attachments to put in the out message
     * @throws MessagingException
     *             if error occurred during out message creation
     */
    void setInMessageAttachments(Map<String, DataHandler> attachements) throws MessagingException;

    /**
     * Put a Source object in the in message of the message exchange. If the in
     * normalized message is null, it is created here.
     * 
     * @param inContent
     *            the DOM Document content to put in the in message
     * @throws MessagingException
     *             if error occurred during in message creation
     */
    void setInMessageContent(final Document inContent) throws MessagingException;

    /**
     * Put a String object in the in message of the message exchange. If the in
     * normalized message is null, it is created here.
     * 
     * @param inContent
     *            the Source content to put in the in message
     * @throws MessagingException
     *             if error occurred during in message creation
     */
    void setInMessageContent(final InputStream inContent) throws MessagingException;

    /**
     * Put a Source object in the in message of the message exchange. If the in
     * normalized message is null, it is created here.
     * 
     * @param inContent
     *            the Source content to put in the in message
     * @throws MessagingException
     *             if error occurred during in message creation
     */
    void setInMessageContent(final Source inContent) throws MessagingException;

    void setInMessageProperties(Map<String, Object> properties) throws MessagingException;

    void setInMessageProperty(String name, Object value) throws MessagingException;

    /**
     * Set the "In" security subject.
     * 
     * @param  the "In" security subject.
     */
    void setInSubject(Subject subject);

    /**
     * Set the target interface name to invoke the service
     * 
     * @param interfaceName
     */
    void setInterfaceName(QName interfaceName);

    /**
     * Set the message operation
     * 
     * @param name the operation (QName)
     */
    void setOperation(QName name);

    /**
     * Set the output message
     * 
     * @param msg the output message
     * @throws MessagingException
     */
    void setOutMessage(final NormalizedMessage msg) throws MessagingException;

    /**
     * Put the attachment in the "out" message. If the out normalized message is
     * null, it is created here.
     * 
     * @param name
     * @param attachment
     * @throws MessagingException
     */
    void setOutMessageAttachment(String name, DataHandler attachment) throws MessagingException;

    /**
     * Put a Map of attachments in the "out" message. If the out normalized
     * message is null, it is created here.
     * 
     * @param attachements
     *            the attachements to put in the out message
     * @throws MessagingException
     *             if error occured during out message creation
     */
    void setOutMessageAttachments(Map<String, DataHandler> attachements) throws MessagingException;

    /**
     * Put a Source object in the out message of the message exchange. If the
     * out normalized message is null, it is created here.
     * 
     * @param outContent
     *            the DOM document content to put in the out message
     * @throws MessagingException
     *             if error occured during out message creation
     */
    void setOutMessageContent(final Document outContent) throws MessagingException;

    /**
     * Put a String object in the in message of the message exchange. If the in
     * normalized message is null, it is created here.
     * 
     * @param inContent
     *            the Source content to put in the in message
     * @throws MessagingException
     *             if error occurred during in message creation
     */
    void setOutMessageContent(final InputStream inContent) throws MessagingException;

    /**
     * Put a Source object in the out message of the message exchange. If the
     * out normalized message is null, it is created here.
     * 
     * @param outContent
     *            the Source content to put in the out message
     * @throws MessagingException
     *             if error occured during out message creation
     */
    void setOutMessageContent(final Source outContent) throws MessagingException;

    void setOutMessageProperties(Map<String, Object> properties) throws MessagingException;

    void setOutMessageProperty(String name, Object value) throws MessagingException;

    /**
     * Set the out security subject.
     * 
     * @param subject the out security subject
     */
    void setOutSubject(Subject subject);

    void setProperty(String name, Object obj);

    void setService(QName service);

    void setStatus(ExchangeStatus status) throws MessagingException;

    /**
     * Get the flow attributes of the message exchange
     * 
     * @return the flow attributes of the message exchange
     */
    FlowAttributes getFlowAttributes();
}
