/**
 * PETALS - PETALS Services Platform. Copyright (c) 2008 EBM Websourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * -------------------------------------------------------------------------
 * $Id$
 * -------------------------------------------------------------------------
 */
package org.ow2.petals.component.framework.jbidescriptor;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.net.URL;

import javax.xml.XMLConstants;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;

import org.ow2.petals.commons.threadlocal.DocumentBuilders;
import org.ow2.petals.component.framework.jbidescriptor.generated.Jbi;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.xml.sax.SAXException;

/**
 * CDK JBI descriptor Builder. Provide marshalling/unmarshalling features.
 * 
 * Created on 22 mai 2008
 * 
 * @author Christophe HAMERLING, Roland NAUDIN - eBM WebSourcing
 * @since 1.0
 * 
 */
public class JBIDescriptorBuilder {

    /**
     * CDK schema
     */
    public static final String CDKJBI_XSD = "CDKjbi.xsd";

    public static final String ABSTRACT_CDKJBI_XSD = "abstractCDKjbi.xsd";

    /**
     * Extensions schema
     */
    public static final String CDKEXTENSIONS_XSD = "CDKextensions.xsd";

    /**
     * The schema factory
     */
    private static SchemaFactory factory;

    /**
     * The unmarshaller.
     */
    private static Unmarshaller unmarshaller;

    private static Object unmarshallerSync = new Object();

    /**
     * The exception raised during JAXB unique unmarshaller creation.
     */
    private static CDKJBIDescriptorException unmarshCreationEx;

    /**
     * The JAXB unique marshaller.
     */
    private static Marshaller marshaller;

    private static Object marshallerSync = new Object();;

    /**
     * The exception raised during JAXB unique marshaller creation.
     */
    private static CDKJBIDescriptorException marshallerCreationEx;

    /**
     * Static object initialization
     */
    static {

        factory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);

        final URL cdkSchemaUrl = JBIDescriptorBuilder.class.getResource("/" + CDKJBI_XSD);
        final URL cdkExtUrl = JBIDescriptorBuilder.class.getResource("/" + CDKEXTENSIONS_XSD);

        try {
            InputStream cdkSchemaInputStream = cdkExtUrl.openStream();
            InputStream cdkExtInputStream = cdkSchemaUrl.openStream();

            final Schema schema = factory.newSchema(new StreamSource[] {
                    new StreamSource(cdkSchemaInputStream), new StreamSource(cdkExtInputStream) });

            final JAXBContext jaxbContext = JAXBContext.newInstance(new Class[] { Jbi.class });

            // Unmarshaller initialization
            try {
                unmarshaller = jaxbContext.createUnmarshaller();
                unmarshaller.setSchema(schema);
            } catch (final JAXBException e) {
                unmarshCreationEx = new CDKJBIDescriptorException(
                        "Failed to create the JAXB unmarshaller", e);
            }

            // Marshaller initialization
            try {
                marshaller = jaxbContext.createMarshaller();
                marshaller.setSchema(schema);
                marshaller.setProperty(Marshaller.JAXB_ENCODING, "UTF-8");
                marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, Boolean.TRUE);
            } catch (JAXBException e) {
                marshallerCreationEx = new CDKJBIDescriptorException(
                        "Failed to create the JAXB marshaller", e);
            }

            cdkSchemaInputStream.close();
            cdkExtInputStream.close();

        } catch (final Exception e) {
            unmarshCreationEx = new CDKJBIDescriptorException(
                    "Failed to create the JAXB unmarshaller", e);
            marshallerCreationEx = new CDKJBIDescriptorException(
                    "Failed to create the JAXB marshaller and unmarshaller", e);
        }
    }
    
    /**
     * Frees singleton resources. The JBIDescriptorBuilder can not be used after to be freed.
     */
    public final static void close() {
    	marshaller = null;
    	marshallerCreationEx = null;
    	marshallerSync = null;
    	unmarshaller = null;
    	unmarshCreationEx = null;
    	unmarshallerSync = null;
    	factory = null;
    }

    /**
     * Validate the given JBI descriptor against the CDK JBI schema.
     * 
     * @param jbiDescriptorStream
     * @throws CDKJBIDescriptorException
     */
    public static void validateJBIDescriptor(final InputStream jbiDescriptorStream)
            throws CDKJBIDescriptorException {

        // TODO : is the validator thread safe?
        final Validator validator = getUnMarshaller().getSchema().newValidator();

        try {
            validator.validate(new StreamSource(jbiDescriptorStream));
        } catch (final SAXException e) {
            throw new CDKJBIDescriptorException(
                    "Failed to validate JBI descriptor against JBI schema", e);
        } catch (final IOException e) {
            throw new CDKJBIDescriptorException(
                    "Failed to validate JBI descriptor against JBI schema", e);
        }
    }

    /**
     * Validate the given JBI descriptor against the component implementation
     * JBI schemas.
     * 
     * @param jbiDescriptorStream
     * @throws CDKJBIDescriptorException
     * 
     *             FIXME: The structure of XSDs with the redefine feature
     *             doesn't work, maybe a Xerces bug?
     */
    public static void validateJBIDescriptor(final InputStream jbiDescriptorStream,
            final String componentName) throws CDKJBIDescriptorException {
        final URL cdkExtUrl = JBIDescriptorBuilder.class.getResource("/" + CDKEXTENSIONS_XSD);
        final URL componentExtUrl = JBIDescriptorBuilder.class.getResource("/" + componentName
                + "extensions.xsd");
        final URL cdkSchemaUrl = JBIDescriptorBuilder.class.getResource("/" + ABSTRACT_CDKJBI_XSD);
        final URL componentSchemaUrl = JBIDescriptorBuilder.class.getResource("/" + componentName
                + "jbi.xsd");

        try {
            final Schema schema = factory.newSchema(new StreamSource[] {
                    new StreamSource(cdkExtUrl.openStream()),
                    new StreamSource(componentExtUrl.openStream()),
                    new StreamSource(cdkSchemaUrl.openStream()),
                    new StreamSource(componentSchemaUrl.openStream()) });

            schema.newValidator().validate(new StreamSource(jbiDescriptorStream));

        } catch (SAXException e) {
            throw new CDKJBIDescriptorException(
                    "Failed to create the CDK component specific schema validation", e);
        } catch (IOException e) {
            throw new CDKJBIDescriptorException(
                    "Failed to create the CDK component specific schema validation", e);
        }
    }
    
	/**
	 * <p>
	 * Build the Jbi Java binding from the given JBI descriptor in XML character
	 * reader form.
	 * </p>
	 * <p>
	 * <b>Note</b> : <b> MUST </b> be used to parse a JBI descriptor available as
	 * {@link String}.
	 * 
	 * @param jbiDescriptorReader
	 *            the XML JBI descriptor character reader
	 * @return the Jbi instance representing the JBI descriptor
	 * @throws CDKJBIDescriptorException
	 *             impossible to create the Java bindings
	 */
    public static Jbi buildJavaJBIDescriptor(final Reader jbiDescriptorReader)
            throws CDKJBIDescriptorException {
    	try {
            final JAXBElement<Jbi> root;
            // The default Xerces unmarshaller is not thread safe
            synchronized (unmarshallerSync) {
                root = getUnMarshaller()
                        .unmarshal(new StreamSource(jbiDescriptorReader), Jbi.class);
            }

            return root.getValue();
        } catch (final JAXBException e) {
            throw new CDKJBIDescriptorException("Failed to build jbi descriptor", e);
        }
    }

    /**
     * <p>Build the Jbi Java binding from the given JBI descriptor in XML Stream
     * form.</p>
     * <p><b>Note</b> : if the stream is initialized from a {@link String}, prefer to use 
     * 
     * @param jbiDescriptorStream
     *            the XML JBI descriptor stream
     * @return the Jbi instance representing the JBI descriptor
     * @throws CDKJBIDescriptorException
     *             impossible to create the Java bindings
     */
    public static Jbi buildJavaJBIDescriptor(final InputStream jbiDescriptorStream)
            throws CDKJBIDescriptorException {
        try {
            final JAXBElement<Jbi> root;
            // The default Xerces unmarshaller is not thread safe
            synchronized (unmarshallerSync) {
                root = getUnMarshaller()
                        .unmarshal(new StreamSource(jbiDescriptorStream), Jbi.class);
            }

            return root.getValue();
        } catch (final JAXBException e) {
            throw new CDKJBIDescriptorException("Failed to build jbi descriptor", e);
        }
    }

    /**
     * Build the XML nodes tree from the JBI descriptor in Java classes form.
     * 
     * @param jbiDescriptorClass
     *            The JBI Descriptor root class
     * @param jbiDescriptorNode
     *            The XML Node to fill with the JBI descriptor XML nodes
     * @throws JBIDescriptorException
     *             The exception raised during the marshaller creation or the
     *             exception raised during the build of the XML nodes.
     */
    public static void buildXmlJBIdescriptor(final Jbi jbiDescriptorClass, Node jbiDescriptorNode)
            throws CDKJBIDescriptorException {

        try {
            // The default Xerces marshaller is not thread safe
            synchronized (marshallerSync) {
                getMarshaller().marshal(jbiDescriptorClass, jbiDescriptorNode);
            }
        } catch (JAXBException e) {
            throw new CDKJBIDescriptorException(
                    "Failed to build XML binding from JBI descriptor Java classes", e);
        }
    }

    /**
     * Build the XML nodes tree from the JBI descriptor in Java classes form. A
     * default Document id built-in and filled with the created nodes.
     * 
     * @param jbiDescriptorClass
     *            The JBI Descriptor root class
     * @return The built-in @ code Document} filled with the generated nodes
     * @throws JBIDescriptorException
     *             The exception raised during the marshaller creation or the
     *             exception raised during the build of the XML nodes.
     */
    public static Document buildXmlJBIdescriptor(final Jbi jbiDescriptorClass)
            throws CDKJBIDescriptorException {

        Document document = getDocumentBuilder().newDocument();

        try {
            // The default Xerces marshaller is not thread safe
            synchronized (marshallerSync) {
                getMarshaller().marshal(jbiDescriptorClass, document);
            }
        } catch (JAXBException e) {
            throw new CDKJBIDescriptorException(
                    "Failed to build XML binding from JBI descriptor Java classes", e);
        }

        return document;
    }

    /**
     * Build the output Stream from the JBI descriptor in Java classes form.
     * 
     * @param jbiDescriptorClass
     *            The JBI Descriptor root class
     * @param jbiDescriptorNode
     *            The output Stream to fill with the JBI descriptor
     * @throws JBIDescriptorException
     *             The exception raised during the marshaller creation or the
     *             exception raised during the build of the XML nodes.
     */
    public static void buildXmlJBIdescriptor(final Jbi jbiDescriptorClass, OutputStream ouputStream)
            throws CDKJBIDescriptorException {

        try {
            // The default Xerces marshaller is not thread safe
            synchronized (marshallerSync) {
                getMarshaller().marshal(jbiDescriptorClass, ouputStream);
            }
        } catch (JAXBException e) {
            throw new CDKJBIDescriptorException(
                    "Failed to build XML binding from JBI descriptor Java classes", e);
        }
    }

    /**
     * Get the unmarshaller instance.
     * 
     * @return the unmarshaller instance
     * @throws CDKJBIDescriptorException
     *             The exception raised during the unmarshaller creation.
     */
    private static Unmarshaller getUnMarshaller() throws CDKJBIDescriptorException {

        if (unmarshCreationEx != null) {
            throw unmarshCreationEx;
        }

        return unmarshaller;
    }

    /**
     * Get the marshaller instance.
     * 
     * @return the marshaller instance
     * @throws JBIDescriptorException
     *             The exception raised during the marshaller creation.
     */
    private static Marshaller getMarshaller() throws CDKJBIDescriptorException {

        if (marshallerCreationEx != null) {
            throw marshallerCreationEx;
        }

        return marshaller;
    }

    /**
     * Get the document Builder instance.
     * 
     * @return the document Builder instance
     * @throws CDKJBIDescriptorException
     *             The exception raised during the document Builder creation.
     */
    private static DocumentBuilder getDocumentBuilder() throws CDKJBIDescriptorException {

        try {
            return DocumentBuilders.getNamespaceDocumentBuilder();
        } catch (final RuntimeException e) {
            // When initialiazing the thread local document bulder, an exception
            // can occurs.
            throw new CDKJBIDescriptorException("Failed to create the Document Builder", e);
        }
    }
}
