package org.ow2.petals.jbi.messaging.exchange;

import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Monolog logger implementation. This class is used to format various logs. it
 * can be used as a static class or as a wrapper object of a logger. The methods
 * add to the resulting message a "ClassName-MethodName" information. (only for
 * DEBUG and INFO level)
 * 
 * TODO : Rename to MonologLogger and move to kernel implementation
 * 
 * @author alouis
 * 
 */
public class LoggingUtil {

    protected Logger log;

    protected String name;

    /**
     * Creates a new instance of LoggingUtil
     * 
     * @param logger
     */
    public LoggingUtil(Logger logger) {
        this(logger, "");
    }

    /**
     * Creates a new instance of LoggingUtil
     * 
     * @param logger
     * @param loggerName
     */
    public LoggingUtil(Logger logger, String loggerName) {
        this.log = logger;

        if ((loggerName != null) && (loggerName.trim().length() > 0)) {
            loggerName = "[" + loggerName + "] ";
        } else {
            loggerName = "";
        }
        this.name = loggerName;
    }

    /**
     * {@inheritDoc}
     */
    public void call() {
        if ((this.log != null) && this.log.isLoggable(Level.FINEST)) {
            this.log.log(Level.FINEST, this.name + "-CALL-" + classAndMethod());
        }
    }

    /**
     * {@inheritDoc}
     */
    public void call(Object msg) {
        if ((this.log != null) && this.log.isLoggable(Level.FINEST)) {
            this.log.log(Level.FINEST, this.name + "-CALL-" + classAndMethod() + " " + msg);
        }
    }

    /**
     * {@inheritDoc}
     */
    public void start() {
        if ((this.log != null) && this.log.isLoggable(Level.FINEST)) {
            this.log.log(Level.FINEST, this.name + "-START-" + classAndMethod());
        }
    }

    /**
     * {@inheritDoc}
     */
    public void start(Object msg) {
        if ((this.log != null) && this.log.isLoggable(Level.FINEST)) {
            this.log.log(Level.FINEST, this.name + "-START-" + classAndMethod() + " " + msg);
        }
    }

    /**
     * {@inheritDoc}
     */
    public void end() {
        if ((this.log != null) && this.log.isLoggable(Level.FINEST)) {
            this.log.log(Level.FINEST, this.name + "-END-" + classAndMethod());
        }
    }

    /**
     * {@inheritDoc}
     */
    public void end(Object msg) {
        if ((this.log != null) && this.log.isLoggable(Level.FINEST)) {
            this.log.log(Level.FINEST, this.name + "-END-" + classAndMethod() + " " + msg);
        }
    }

    /**
     * {@inheritDoc}
     */
    public void debug(Object message) {
        if ((this.log != null) && this.log.isLoggable(Level.FINEST)) {
            this.log.log(Level.FINEST, this.name + classAndMethod() + " " + message);
        }
    }
    
   
    /**
     * {@inheritDoc}
     */
    public void info(Object message) {
        if ((this.log != null) && this.log.isLoggable(Level.INFO)) {
            if (this.log.isLoggable(Level.FINEST)) {
                this.log.log(Level.INFO, this.name + classAndMethod() + " " + message);
            } else {
                this.log.log(Level.INFO, this.name + message);
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    public void info(Object message, Throwable error) {
        if ((this.log != null) && this.log.isLoggable(Level.INFO)) {
            if (this.log.isLoggable(Level.FINEST)) {
                this.log.log(Level.INFO, this.name + classAndMethod() + " " + message, error);
            } else {
                this.log.log(Level.INFO, this.name + message, error);
            }

        }
    }

    /**
     * {@inheritDoc}
     */
    public void warning(Object message) {
        if ((this.log != null) && this.log.isLoggable(Level.WARNING)) {
            this.log.log(Level.WARNING, this.name + message);
        }
    }

    /**
     * {@inheritDoc}
     */
    public void warning(Object message, Throwable throwable) {
        if ((this.log != null) && this.log.isLoggable(Level.WARNING)) {
            this.log.log(Level.WARNING, this.name + message, throwable);
        }
    }

    /**
     * {@inheritDoc}
     */
    public void error(Object message) {
        if ((this.log != null) && this.log.isLoggable(Level.SEVERE)) {
            this.log.log(Level.SEVERE, this.name + message);
        }
    }

    /**
     * {@inheritDoc}
     */
    public void error(Object message, Throwable throwable) {
        if ((this.log != null) && this.log.isLoggable(Level.SEVERE)) {
            this.log.log(Level.SEVERE, this.name + message, throwable);
        }
    }

    /**
     * {@inheritDoc}
     */
    public boolean isDebugEnabled() {
        return this.isLevelEnabled(Level.FINEST);
    }
    
    /**
     * {@inheritDoc}
     */
    public boolean isInfoEnabled() {
        return this.isLevelEnabled(Level.INFO);
    }

    /**
     * {@inheritDoc}
     */
    public boolean isWarnEnabled() {
        return this.isLevelEnabled(Level.WARNING);
    }

    /**
     * {@inheritDoc}
     */
    public boolean isErrorEnabled() {
        return this.isLevelEnabled(Level.SEVERE);
    }

    private boolean isLevelEnabled(final Level level) {
        return (this.log != null) && this.log.isLoggable(level);
    }

    // //////////////////////////////////////////////////////////
    // Static part
    // //////////////////////////////////////////////////////////

    /**
     * Create an exception and analyse it in order to find the class and method
     * that called the LoggingUtil method.
     * 
     * @return
     */
    private static String classAndMethod() {
        String result = null;

        // throw and Parse an exception to find in the stack
        // the method that called the Loggingutil

        Throwable t = new Throwable();

        StackTraceElement[] ste = t.getStackTrace();

        if ((ste != null) && (ste.length > 2)) {
            StackTraceElement element = ste[2];

            // If the 2nd element in the stack is this class, get the 3rd
            if (element.getClassName().endsWith(LoggingUtil.class.getName())) {
                element = ste[3];
            }
            String className = element.getClassName();

            // remove the package name of the ClassName
            int index = className.lastIndexOf(".");

            if (index > -1) {
                className = className.substring(index + 1, className.length());
            }

            result = className + "." + element.getMethodName() + "()";
        }
        return result;
    }

    /**
     * {@inheritDoc}
     */
    public String getName() {
        return this.log.getName();
    }

	public Logger getLogger() {
		return this.log;
	}

}
