/**
 * PETALS: PETALS Services Platform Copyright (C) 2006 EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 * 
 * Initial developer(s): Adrien LOUIS
 */

package org.ow2.petals.jbi.messaging.exchange;

import java.net.URI;
import java.util.Map;
import java.util.Set;

import javax.jbi.messaging.ExchangeStatus;
import javax.jbi.messaging.Fault;
import javax.jbi.messaging.FlowAttributes;
import javax.jbi.messaging.MessageExchange;
import javax.jbi.messaging.MessagingException;
import javax.jbi.messaging.NormalizedMessage;
import javax.jbi.servicedesc.ServiceEndpoint;
import javax.xml.namespace.QName;

/**
 * 
 * This class wrappers a real <code>MessageExchange</code>. <br>
 * Each actor (consumer or provider) owns an instance of this class. <br>
 * The wrapper checks that the methods which are called by the related
 * component can be used at the current exchange state.
 * 
 * @author Adrien LOUIS - EBM WebSourcing
 */
public class MessageExchangeWrapper implements MessageExchange {

    /**
     * The unique exchange Id of this wrapper.
     */
    protected String exchangeId;

    /**
     * The wrapped Message Exchange. This instance can change during the life
     * of a wrapper instance.
     */
    protected MessageExchangeImpl messageExchange;

    /**
     * Role of the observer (consumer or provider). Used for checking that the
     * ME is in the good role before calling a method.
     */
    private Role observerRole;

    /**
     * A flag to know if the exchange is in a send timeout state.
     */
    private boolean timeout;

    /**
     * Create a MessageExchangeWrapper.
     */
    public MessageExchangeWrapper(MessageExchangeImpl me,
            Role observerRole) {
        assert me != null;
        assert observerRole != null;
        this.messageExchange = me;
        this.exchangeId = this.messageExchange.getExchangeId();
        this.observerRole = observerRole;
    }
    
    /*
     * (non-Javadoc)
     * 
     * @see javax.jbi.messaging.MessageExchange#createFault()
     */
    public Fault createFault() throws MessagingException {
        this.checkComponentOwnsExchange();

        return this.messageExchange.createFault();
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.jbi.messaging.MessageExchange#createMessage()
     */
    public NormalizedMessage createMessage() throws MessagingException {
        this.checkComponentOwnsExchange();

        return this.messageExchange.createMessage();
    }

    public ServiceEndpoint getConsumerEndpoint() {
        return (ServiceEndpoint) this.messageExchange.getConsumerEndpoint();
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.jbi.messaging.MessageExchange#getEndpoint()
     */
    public ServiceEndpoint getEndpoint() {
        this.checkComponentOwnsExchange();

        return this.messageExchange.getEndpoint();
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.jbi.messaging.MessageExchange#getError()
     */
    public Exception getError() {
        checkComponentOwnsExchange();

        return this.messageExchange.getError();
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.jbi.messaging.MessageExchange#getExchangeId()
     */
    public String getExchangeId() {
        return this.exchangeId;
    }

    public FlowAttributes getFlowAttributes() {
        return this.messageExchange.getFlowAttributes();
    }

    public void setFlowAttributes(FlowAttributes flowAttributes) {
        this.messageExchange.setFlowAttributes(flowAttributes);
    }
    
    /*
     * (non-Javadoc)
     * 
     * @see javax.jbi.messaging.MessageExchange#getFault()
     */
    public Fault getFault() {
        this.checkComponentOwnsExchange();

        return this.messageExchange.getFault();
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.jbi.messaging.MessageExchange#getInterfaceName()
     */
    public QName getInterfaceName() {
        this.checkComponentOwnsExchange();

        return this.messageExchange.getInterfaceName();
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.jbi.messaging.MessageExchange#getMessage(java.lang.String)
     */
    public NormalizedMessage getMessage(String name) {
        this.checkComponentOwnsExchange();

        return this.messageExchange.getMessage(name);
    }

    /**
     * Return the real implementation of this MessageExchange.
     * 
     * @return MessageExchangeImpl
     */
    public MessageExchangeImpl getMessageExchange() {
        return this.messageExchange;
    }

    /**
     * Get all the normalized messages
     * 
     * @return
     */
    public Map<String, NormalizedMessage> getMessages() {
        return this.messageExchange.getMessages();
    }

    /**
     * Return the Role that the owner of this object has in the exchange.
     * 
     * @return
     */
    public Role getObserverRole() {
        return this.observerRole;
    }

    /*
     * @see javax.jbi.messaging.MessageExchange#getOperation()
     */
    public QName getOperation() {
        this.checkComponentOwnsExchange();

        return this.messageExchange.getOperation();
    }

    /*
     * @see javax.jbi.messaging.MessageExchange#getPattern()
     */
    public URI getPattern() {
        this.checkComponentOwnsExchange();

        return this.messageExchange.getPattern();
    }

    /*
     * @see javax.jbi.messaging.MessageExchange#getProperty(java.lang.String)
     */
    public Object getProperty(String name) {
        this.checkComponentOwnsExchange();

        return this.messageExchange.getProperty(name);
    }

    /*
     * @see javax.jbi.messaging.MessageExchange#getPropertyNames()
     */
    public Set<?> getPropertyNames() {
        this.checkComponentOwnsExchange();

        return this.messageExchange.getPropertyNames();
    }

    /*
     * @see javax.jbi.messaging.MessageExchange#getRole()
     */
    public Role getRole() {
        return this.messageExchange.getRole();
    }

    /*
     * @see javax.jbi.messaging.MessageExchange#getService()
     */
    public QName getService() {
        this.checkComponentOwnsExchange();

        return this.messageExchange.getService();
    }

    /*
     * @see javax.jbi.messaging.MessageExchange#getStatus()
     */
    public javax.jbi.messaging.ExchangeStatus getStatus() {
        return this.messageExchange.getStatus();
    }

    public boolean isMessageExchangeStored() {
        if (this.messageExchange == null) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Is terminated?
     * 
     * @return
     */
    public boolean isTerminated() {
        return this.messageExchange.isTerminated();
    }

    /**
     * Is send timeout?
     * 
     * @return
     */
    public boolean isTimeout() {
        return timeout;
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.jbi.messaging.MessageExchange#isTransacted()
     */
    public boolean isTransacted() {
        this.checkComponentOwnsExchange();

        return this.messageExchange.isTransacted();
    }

    /*
     * @see javax.jbi.messaging.MessageExchange#setEndpoint(java.lang.Exception)
     */
    public void setEndpoint(ServiceEndpoint endpoint) {
        this.checkComponentOwnsExchange();

        this.messageExchange.setEndpoint(endpoint);
    }

    /*
     * @see javax.jbi.messaging.MessageExchange#setError(java.lang.Exception)
     */
    public void setError(Exception error) {
        this.checkComponentOwnsExchange();

        this.messageExchange.setError(error);
    }

    /*
     * @see
     * javax.jbi.messaging.MessageExchange#setFault(javax.jbi.messaging.Fault)
     */
    public void setFault(Fault fault) throws MessagingException {
        this.checkComponentOwnsExchange();

        this.messageExchange.setFault(fault);
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * javax.jbi.messaging.MessageExchange#setInterfaceName(javax.xml.namespace
     * .QName)
     */
    public void setInterfaceName(QName interfaceName) {
        this.checkComponentOwnsExchange();

        this.messageExchange.setInterfaceName(interfaceName);
    }

    /*
     * @seejavax.jbi.messaging.MessageExchange#setMessage(javax.jbi.messaging.
     * NormalizedMessage, java.lang.String)
     */
    public void setMessage(NormalizedMessage msg, String name) throws MessagingException {
        this.checkComponentOwnsExchange();

        this.messageExchange.setMessage(msg, name);
    }

    /**
     * Reset a Message Exchange instance to the wrapper
     * 
     * @param messageExchange
     */
    public void setMessageExchange(
            org.ow2.petals.jbi.messaging.exchange.MessageExchangeImpl messageExchange)
            throws MessagingException {
        if (!this.exchangeId.equals(messageExchange.getExchangeId())) {
            throw new MessagingException("Wrong Message Exchange Id for wrapper. Expected Id '"
                    + this.exchangeId + "' and get '" + messageExchange.getExchangeId() + "'");
        }
        this.messageExchange = messageExchange;
    }

    /**
     * Set the observer role
     * 
     * @return
     */
    public void setObserverRole(Role role) {
        this.observerRole = role;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * javax.jbi.messaging.MessageExchange#setOperation(javax.xml.namespace.
     * QName)
     */
    public void setOperation(QName name) {
        this.checkComponentOwnsExchange();

        this.messageExchange.setOperation(name);
    }

    /**
     * Set the persisted flag.
     * 
     * @param role
     */
    public void setPersisted(boolean persisted) {
        this.messageExchange.setPersisted(persisted);
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.jbi.messaging.MessageExchange#setProperty(java.lang.String,
     * java.lang.Object)
     */
    public void setProperty(String name, Object obj) {
        this.checkComponentOwnsExchange();

        this.messageExchange.setProperty(name, obj);
    }

    /**
     * Set the role.
     * 
     * @param role
     */
    public void setRole(Role role) {
        this.messageExchange.setRole(role);
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * javax.jbi.messaging.MessageExchange#setService(javax.xml.namespace.QName)
     */
    public void setService(QName service) {
        this.checkComponentOwnsExchange();

        this.messageExchange.setService(service);
    }

    /*
     * (non-Javadoc)
     * 
     * @seejavax.jbi.messaging.MessageExchange#setStatus(javax.jbi.messaging.
     * ExchangeStatus)
     */
    public void setStatus(ExchangeStatus status) throws MessagingException {
        this.checkComponentOwnsExchange();

        this.messageExchange.setStatus(status);
    }

    /**
     * Set the terminated flag.
     * 
     * @param terminated
     */
    public void setTerminated(boolean terminated) {
        this.messageExchange.setTerminated(terminated);
    }

    /**
     * Set the send timeout on the wrapper.
     * 
     * @return
     */
    public void setTimeout(boolean timeout) {
        this.timeout = timeout;
    }

    /**
     * Set the transacted flag.
     * 
     * @param terminated
     */
    public void setTransacted(boolean transacted) {
        this.messageExchange.setTerminated(transacted);
    }

    /**
     * Check that the component owns the message exchange. (Check that the role
     * of the observer is the same than the role of the message exchange)
     * 
     * @throws IllegalStateException
     */
    private void checkComponentOwnsExchange() throws IllegalStateException {
        if (this.observerRole != null && !this.messageExchange.getRole().equals(this.observerRole)) {
            if (this.messageExchange.isTerminated()) {
                throw new IllegalStateException("The Message Exchange has been terminated "
                        + "by another component");
            } else {
                throw new IllegalStateException("The component does not owns the MessageExchange");
            }
        }
    }
}
