/**
 * PETALS: PETALS Services Platform Copyright (C) 2005 EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 * 
 * Initial developer(s): EBM WebSourcing
 * --------------------------------------------------------------------------
 * $Id: MessageExchangeImpl.java,v 1.2 2005/07/22 10:24:27 alouis Exp $
 * --------------------------------------------------------------------------
 */

package org.ow2.petals.jbi.messaging.exchange;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.net.URI;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import javax.jbi.messaging.ExchangeStatus;
import javax.jbi.messaging.Fault;
import javax.jbi.messaging.FlowAttributes;
import javax.jbi.messaging.MessageExchange;
import javax.jbi.messaging.MessagingException;
import javax.jbi.messaging.NormalizedMessage;
import javax.jbi.servicedesc.ServiceEndpoint;
import javax.xml.namespace.QName;

/**
 * @author Adrien LOUIS - EBM WebSourcing
 * @author Gael BLONDELLE - EBM WebSourcing
 * @author Anass OUAZZANI - EBM WebSourcing
 */
public class MessageExchangeImpl implements MessageExchange, Cloneable, Serializable {

    @Deprecated
    public static final URI IN_ONLY_PATTERN = URI.create("http://www.w3.org/2004/08/wsdl/in-only");

    @Deprecated
    public static final URI IN_OPTIONAL_OUT_PATTERN = URI
            .create("http://www.w3.org/2004/08/wsdl/in-opt-out");

    @Deprecated
    public static final URI IN_OUT_PATTERN = URI.create("http://www.w3.org/2004/08/wsdl/in-out");

    @Deprecated
    public static final URI ROBUST_IN_ONLY_PATTERN = URI
            .create("http://www.w3.org/2004/08/wsdl/robust-in-only");

    public static final String IN_MSG = "in";

    public static final String OUT_MSG = "out";

    public static final String FAULT_MSG = "fault";

    public static final short SERIALIZE_ROLE_CONSUMER = 0;

    public static final short SERIALIZE_ROLE_PROVIDER = 1;

    public static final short SERIALIZE_STATUS_ACTIVE = 0;

    public static final short SERIALIZE_STATUS_DONE = 1;

    public static final short SERIALIZE_STATUS_ERROR = 2;

    /**
     * Serial UID
     */
    private static final long serialVersionUID = -4354354L;

    private ServiceEndpoint consumerEndpoint;

    private ServiceEndpoint endpoint;

    private Exception error;

    private String exchangeId;

    private FlowAttributes flowAttributes;

    private QName interfaceName;

    private Map<String, NormalizedMessage> messages = new HashMap<String, NormalizedMessage>();

    private QName operation;

    private URI pattern;

    private Map<String, Object> properties = new HashMap<String, Object>();

    private QName service;

    private boolean terminated;

    private boolean transacted;

    private boolean persisted;

    private boolean monitored;

    private transient Role role = Role.CONSUMER;

    private transient ExchangeStatus status = ExchangeStatus.ACTIVE;

    /**
     * Depending on the state of the Exchange, clean IN and/or OUT normalized
     * messages to reduce exchange size. Status DONE, ERROR : all messages are
     * cleared Fault message : all other messages are cleared OUT message set :
     * IN message is cleared
     */
    public void cleanMessages() {
        // if status is DONE or ERROR, do not send Normalized messages
        if (isDoneOrError()) {
            this.messages.clear();
        }
        // if there is a fault, no other messages are kept
        if (this.messages.containsKey(FAULT_MSG)) {
            if (this.messages.remove(IN_MSG) == null) {
                this.messages.remove(OUT_MSG);
            }
        }
        // if OUT response, IN message is not kept
        if (this.messages.containsKey(OUT_MSG)) {
            this.messages.remove(IN_MSG);
        }
    }

    /**
     * @see javax.jbi.messaging.MessageExchange#createFault()
     */
    public Fault createFault() throws MessagingException {
        return new FaultImpl();
    }

    /**
     * @see javax.jbi.messaging.MessageExchange#createMessage()
     */
    public NormalizedMessage createMessage() throws MessagingException {
        return new NormalizedMessageImpl();
    }

    public ServiceEndpoint getConsumerEndpoint() {
        return this.consumerEndpoint;
    }

    /**
     * @see javax.jbi.messaging.MessageExchange#getEndpoint()
     */
    public ServiceEndpoint getEndpoint() {
        return this.endpoint;
    }

    /**
     * @see javax.jbi.messaging.MessageExchange#getError()
     */
    public Exception getError() {
        return this.error;
    }

    /**
     * @see javax.jbi.messaging.MessageExchange#getExchangeId()
     */
    public String getExchangeId() {
        return this.exchangeId;
    }

    /**
     * @see javax.jbi.messaging.MessageExchange#getFault()
     */
    public Fault getFault() {
        return (Fault) this.getMessage(FAULT_MSG);
    }

    public QName getInterfaceName() {
        return this.interfaceName;
    }

    /**
     * get the message associated to the reference. It is <b>Not</b> Case
     * Sensitive.
     * 
     * @see javax.jbi.messaging.MessageExchange#getMessage(java.lang.String)
     */
    public NormalizedMessage getMessage(String name) {
        if (name != null) {
            return this.messages.get(name.toLowerCase());
        } else {
            return null;
        }
    }

    public Map<String, NormalizedMessage> getMessages() {
        return this.messages;
    }

    /**
     * @see javax.jbi.messaging.MessageExchange#getOperation()
     */
    public QName getOperation() {
        return this.operation;
    }

    /**
     * @see javax.jbi.messaging.MessageExchange#getPattern()
     */
    public URI getPattern() {
        // TODO : remove this test when the deprecated MEP constants are removed
        if (this.pattern != null) {
            String version = null;
            final Package _package = Package.getPackage("org.ow2.petals.cdk");
            if (_package != null) {
                version = _package.getImplementationVersion();
            }
            if (version != null && "5.1".compareTo(version) <= 0) {
                return this.pattern;
            } else {
                if (MEPPatternConstants.IN_ONLY.equals(this.pattern)) {
                    return IN_ONLY_PATTERN;
                } else if (MEPPatternConstants.ROBUST_IN_ONLY.equals(this.pattern)) {
                    return ROBUST_IN_ONLY_PATTERN;
                } else if (MEPPatternConstants.IN_OUT.equals(this.pattern)) {
                    return IN_OUT_PATTERN;
                } else if (MEPPatternConstants.IN_OPTIONAL_OUT.equals(this.pattern)) {
                    return IN_OPTIONAL_OUT_PATTERN;
                }
            }
        }
        return null;
    }

    /**
     * @see javax.jbi.messaging.MessageExchange#getProperty(java.lang.String)
     */
    public Object getProperty(String name) {
        return this.properties.get(name);
    }

    /**
     * @see javax.jbi.messaging.MessageExchange#getPropertyNames()
     */
    public Set<?> getPropertyNames() {
        return this.properties.keySet();
    }

    /**
     * (non-Javadoc)
     * 
     * @see javax.jbi.messaging.MessageExchange#getRole()
     */
    public Role getRole() {
        return this.role;
    }

    /**
     * @see javax.jbi.messaging.MessageExchange#getService()
     */
    public QName getService() {
        return this.service;
    }

    /**
     * {@inheritDoc}
     */
    public ServiceEndpoint getServiceEndpoint() {
        return this.endpoint;
    }

    /**
     * @see javax.jbi.messaging.MessageExchange#getStatus()
     */
    public javax.jbi.messaging.ExchangeStatus getStatus() {
        return this.status;
    }

    public boolean isMonitored() {
        return this.monitored;
    }

    public boolean isPersisted() {
        return this.persisted;
    }

    public boolean isTerminated() {
        return this.terminated;
    }

    /**
     * @see javax.jbi.messaging.MessageExchange#isTransacted()
     */
    public boolean isTransacted() {
        return this.transacted;
    }

    /**
     * {@inheritDoc}
     */
    public void setConsumerEndpoint(ServiceEndpoint serviceEndpoint) {
        this.consumerEndpoint = serviceEndpoint;
    }

    /**
     * {@inheritDoc}
     */
    public void setEndpoint(ServiceEndpoint endpoint) {
        this.endpoint = endpoint;
    }

    // //////////////////////////////////////////
    // specific methods
    // //////////////////////////////////////////

    /**
     * @see javax.jbi.messaging.MessageExchange#setError(java.lang.Exception)
     */
    public void setError(Exception error) {
        if (error != null) {
            this.error = error;
            try {
                this.setStatus(ExchangeStatus.ERROR);
            } catch (final MessagingException e) {
                // the status can not be changed at this transition
            }
        }
    }

    /**
     * @param exchangeId
     *            The exchangeId to set.
     */
    public void setExchangeId(String exchangeId) {
        this.exchangeId = exchangeId;
    }

    /**
     * (non-Javadoc)
     * 
     * @see javax.jbi.messaging.MessageExchange#setFault(javax.jbi.messaging.Fault)
     */
    public void setFault(Fault fault) throws MessagingException {
        this.checkIsNotTerminated();
        this.checkIsNotDoneOrError();
        this.checkCallerIsAllowedToSetFault();

        this.messages.put(FAULT_MSG, fault);
    }

    private void checkCallerIsAllowedToSetFault() throws MessagingException {
        if (isCallerActingAsConsumer() && !isOutResponseOfAnInOptOutExchange()) {
            throw new MessagingException(
                    "The MessageExchange state and the current Role do not allow this operation.");
        }
    }

    private boolean isCallerActingAsConsumer() {
        return MessageExchange.Role.CONSUMER.equals(this.role);
    }

    private boolean isOutResponseOfAnInOptOutExchange() {
        return (IN_OPTIONAL_OUT_PATTERN.equals(this.pattern) && this.messages.containsKey(OUT_MSG));
    }

    private void checkIsNotDoneOrError() throws MessagingException {
        if (isDoneOrError()) {
            throw new MessagingException(
                    "The MessageExchange state and the current Role do not allow this operation.");
        }
    }

    private boolean isDoneOrError() {
        return ExchangeStatus.DONE.equals(this.status) || ExchangeStatus.ERROR.equals(this.status);
    }

    public void setInterfaceName(QName interfaceName) {
        this.interfaceName = interfaceName;
    }

    /**
     * Set the specified message. Check the availability to set the message with
     * the following rules: <br>
     * <li>CONSUMER Role <-> in reference or PROVIDER Role <-> out reference</li>
     * <li>JBI predefined MEP <-> in or out reference</li> <li>setMessage(XXX
     * ,reference) has not already been called.<br>
     * This is <b>not</b> case sensitive
     * 
     * @see javax.jbi.messaging.MessageExchange#setMessage(javax.jbi.messaging.
     *      NormalizedMessage, java.lang.String)
     */
    public void setMessage(NormalizedMessage msg, String name) throws MessagingException {
        this.checkIsNotTerminated();

        if (msg == null) {
            throw new MessagingException("NormalizedMessage must be non null.");
        }
        if ((name == null) || (name.trim().length() == 0)) {
            throw new MessagingException("The message reference must be non null and non empty.");
        }

        this.checkPatternMatching(name);

        this.checkRoleMatching(name);

        // check that the message has not already been set
        if (this.messages.containsKey(name.toLowerCase())) {
            throw new MessagingException("A message has already been set with the '" + name
                    + "' reference.");
        }

        // no exception was thrown, set the message
        this.messages.put(name.toLowerCase(), msg);
    }

    /**
     * Set the monitored flag.
     * 
     * @param persisted
     */
    public void setMonitored(boolean monitored) {
        this.monitored = monitored;
    }

    /**
     * @see javax.jbi.messaging.MessageExchange#setOperation(javax.xml.namespace.
     *      QName)
     */
    public void setOperation(QName name) {
        this.operation = name;
    }

    /**
     * @param pattern
     *            The pattern to set.
     */
    public void setPattern(URI pattern) {
        this.pattern = pattern;
    }

    /**
     * Set the persisted flag.
     * 
     * @param persisted
     */
    public void setPersisted(boolean persisted) {
        this.persisted = persisted;
    }

    /**
     * @see javax.jbi.messaging.MessageExchange#setProperty(java.lang.String,
     *      java.lang.Object)
     */
    public void setProperty(String name, Object obj) {
        this.properties.put(name, obj);
    }

    public void setRole(Role role) {
        this.role = role;
    }

    /**
     * set Location ServiceName. A destination Endpoint reference is created
     * with only "serviceName" attribute
     */
    public void setService(QName service) {
        this.service = service;
    }

    /**
     * @see javax.jbi.messaging.MessageExchange#setStatus(javax.jbi.messaging.
     *      ExchangeStatus)
     */
    public void setStatus(ExchangeStatus status) throws MessagingException {
        this.checkIsNotTerminated();

        // check that DONE can be set.
        if (ExchangeStatus.DONE.equals(status)) {
            if (Role.CONSUMER.equals(this.role)) {
                if (IN_ONLY_PATTERN.equals(this.pattern)
                        || MEPPatternConstants.IN_ONLY.equals(this.pattern)) {
                    throw new MessagingException(
                            "The MessageExchange state does not allow this operation.");
                } else if ((IN_OUT_PATTERN.equals(this.pattern) || MEPPatternConstants.IN_OUT
                        .equals(this.pattern))
                        && !this.messages.containsKey(FAULT_MSG)
                        && !this.messages.containsKey(OUT_MSG)) {
                    throw new MessagingException(
                            "The MessageExchange state does not allow this operation.");
                } else if ((IN_OPTIONAL_OUT_PATTERN.equals(this.pattern) || MEPPatternConstants.IN_OPTIONAL_OUT
                        .equals(this.pattern))
                        && !this.messages.containsKey(FAULT_MSG)
                        && !this.messages.containsKey(OUT_MSG)) {
                    throw new MessagingException(
                            "The MessageExchange state does not allow this operation.");
                }
            } else {
                if ((ROBUST_IN_ONLY_PATTERN.equals(this.pattern) || MEPPatternConstants.IN_ONLY
                        .equals(this.pattern)) && this.messages.containsKey(FAULT_MSG)) {
                    throw new MessagingException(
                            "The MessageExchange state does not allow this operation.");
                } else if (IN_OUT_PATTERN.equals(this.pattern)
                        || MEPPatternConstants.IN_OUT.equals(this.pattern)) {
                    throw new MessagingException(
                            "The MessageExchange state does not allow this operation.");
                } else if (IN_OPTIONAL_OUT_PATTERN.equals(this.pattern)
                        || MEPPatternConstants.IN_OPTIONAL_OUT.equals(this.pattern)) {
                    if (this.messages.containsKey(FAULT_MSG) && !this.messages.containsKey(OUT_MSG)) {
                        throw new MessagingException(
                                "The MessageExchange state does not allow this operation.");
                    } else if (!this.messages.containsKey(FAULT_MSG)
                            && this.messages.containsKey(OUT_MSG)) {
                        throw new MessagingException(
                                "The MessageExchange state does not allow this operation.");
                    }
                }
            }
        } else if (ExchangeStatus.ERROR.equals(status)) {
            if (Role.PROVIDER.equals(this.role)
                    && (IN_OUT_PATTERN.equals(this.pattern) || MEPPatternConstants.IN_OUT
                            .equals(this.pattern))) {
                throw new MessagingException(
                        "The MessageExchange state does not allow this operation.");
            }
        }
        this.status = status;
    }

    /**
     * Set the terminated flag.
     * 
     * @param terminated
     */
    public void setTerminated(boolean terminated) {
        this.terminated = terminated;
    }

    /**
     * @param transacted
     *            {@code true} is this exchange is transacted
     * 
     */
    public void setTransacted(boolean transacted) {
        this.transacted = transacted;
    }

    @Override
    public String toString() {
        return "MessageExchange@" + this.getExchangeId();
    }

    /**
     * Check if the exchange is still valid.
     * 
     * @throws MessagingException
     */
    private void checkIsNotTerminated() throws MessagingException {
        if (this.isTerminated()) {
            throw new MessagingException("The Exchange is terminated.");
        }
    }

    /**
     * Check that the exchange MEP allows the specified reference. This only
     * works for the JBI predefined MEPs. It check that reference is IN or OUT.
     * Other reference or MEP will cause an exception
     * 
     * @param name
     * @throws MessagingException
     *             the MEP does not allow the reference
     */
    private void checkPatternMatching(String name) throws MessagingException {
        if (this.pattern == null) {
            throw new MessagingException("The MEP is not defined.");
        }

        if (name == null) {
            throw new MessagingException("The reference name is not defined.");
        }

        if (IN_MSG.equalsIgnoreCase(name)
                && (IN_ONLY_PATTERN.equals(this.pattern)
                        || ROBUST_IN_ONLY_PATTERN.equals(this.pattern)
                        || IN_OUT_PATTERN.equals(this.pattern)
                        || IN_OPTIONAL_OUT_PATTERN.equals(this.pattern)
                        || MEPPatternConstants.IN_ONLY.equals(this.pattern)
                        || MEPPatternConstants.ROBUST_IN_ONLY.equals(this.pattern)
                        || MEPPatternConstants.IN_OUT.equals(this.pattern) || MEPPatternConstants.IN_OPTIONAL_OUT
                        .equals(this.pattern))) {
            return;
        } else if (OUT_MSG.equalsIgnoreCase(name)
                && (IN_OUT_PATTERN.equals(this.pattern)
                        || IN_OPTIONAL_OUT_PATTERN.equals(this.pattern)
                        || MEPPatternConstants.IN_OUT.equals(this.pattern) || MEPPatternConstants.IN_OPTIONAL_OUT
                        .equals(this.pattern))) {
            return;
        }
        throw new MessagingException("the MessageExchange state does not allow this operation.");
    }

    /**
     * Check that the Role allows the specified reference. This only works for
     * the JBI predefined MEPs. It check that reference is IN or OUT and the
     * Role is CONSUMER or PROVIDER. Other reference will cause an exception
     * 
     * @param name
     * @throws MessagingException
     *             the Role does not allow the reference
     */
    private void checkRoleMatching(String name) throws MessagingException {
        if (this.role == null) {
            throw new MessagingException("the Role is not defined.");
        }

        if (name == null) {
            throw new MessagingException("The reference name is not defined.");
        }

        if (isCallerActingAsConsumer()) {
            if (IN_MSG.equalsIgnoreCase(name)) {
                return;
            }
        } else if (MessageExchange.Role.PROVIDER.equals(this.role)
                && OUT_MSG.equalsIgnoreCase(name)) {
            return;
        }
        throw new MessagingException("The Role does not allow this operation.");
    }

    /**
     * Used to deserialize Status and Role objects
     * 
     * @param s
     * @throws IOException
     */
    private void readObject(ObjectInputStream s) throws IOException {
        // Role deserialization
        switch (s.readShort()) {
            case 0:
                this.role = MessageExchange.Role.CONSUMER;
                break;
            case 1:
                this.role = MessageExchange.Role.PROVIDER;
                break;
            default:
                break;
        }

        // Status deserialization
        switch (s.readShort()) {
            case 0:
                this.status = ExchangeStatus.ACTIVE;
                break;
            case 1:
                this.status = ExchangeStatus.DONE;
                break;
            case 2:
                this.status = ExchangeStatus.ERROR;
                break;
            default:
                break;
        }

        try {
            s.defaultReadObject();
        } catch (final ClassNotFoundException e) {
            throw new IOException(e.getClass() + ":" + e.getMessage());
        }
    }

    /**
     * Used to serialize Status and Role objects
     * 
     * @param s
     * @throws IOException
     */
    private void writeObject(ObjectOutputStream s) throws IOException {
        // Role serialization
        if (isCallerActingAsConsumer()) {
            s.writeShort(SERIALIZE_ROLE_CONSUMER);
        } else {
            s.writeShort(SERIALIZE_ROLE_PROVIDER);
        }

        // Status serialization
        if (ExchangeStatus.ACTIVE.equals(this.status)) {
            s.writeShort(SERIALIZE_STATUS_ACTIVE);
        } else if (ExchangeStatus.DONE.equals(this.status)) {
            s.writeShort(SERIALIZE_STATUS_DONE);
        } else {
            s.writeShort(SERIALIZE_STATUS_ERROR);
        }
        s.defaultWriteObject();
    }

    public final FlowAttributes getFlowAttributes() {
        return flowAttributes;
    }

    public final void setFlowAttributes(FlowAttributes flowAttributes) {
        this.flowAttributes = flowAttributes;
    }

}
