/**
 * PETALS - PETALS Services Platform. Copyright (c) 2005 EBM Websourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * -------------------------------------------------------------------------
 * $Id: MessageExchangeSerializer.java 154 2006-03-27 15:30:10Z alouis $
 * -------------------------------------------------------------------------
 */

package org.ow2.petals.jbi.messaging.exchange;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.SocketOptions;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import javax.activation.DataHandler;
import javax.mail.util.ByteArrayDataSource;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import com.ebmwebsourcing.easycommons.xml.Transformers;

/**
 * 
 * @author alouis - EBM Websourcing
 * 
 */
public class MessageExchangeSerializer {

    private MessageExchangeSerializer() {
        super();
        // NOP
    }

    public static final void serializeContent(Source content, ObjectOutputStream outStream,
            boolean compress) throws Exception {

        if (content == null) {
            // set a flag which indicate that the source is null
            outStream.writeBoolean(false);
        } else {
            outStream.writeBoolean(true);

            synchronized (content) {
                // Compress content if specified
                StreamResult sresult = null;
                GZIPOutputStream gz = null;
                if (compress) {
                    gz = new GZIPOutputStream(outStream);
                    sresult = new StreamResult(gz);
                } else {
                    sresult = new StreamResult(outStream);
                }
                
                Transformer transformer = Transformers.takeTransformer();
                try {
                    transformer.transform(content, sresult);
                } finally {
                    Transformers.releaseTransformer(transformer);
                }

                if (compress) {
                    gz.finish();
                }
            }
        }
    }

    public static final Source deserializeContent(ObjectInputStream inStream, boolean compress)
            throws IOException {
        StreamSource source = null;

        if (inStream.readBoolean()) {
            // the source is not null
            source = new StreamSource();
            InputStream in = inStream;

            if (compress) {
                in = new GZIPInputStream(inStream);
            }

            byte[] buffer = new byte[SocketOptions.SO_RCVBUF];
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
            int n = 0;
            while (-1 != (n = in.read(buffer))) {
                outputStream.write(buffer, 0, n);
            }
            ByteArrayInputStream result = new ByteArrayInputStream(outputStream.toByteArray());

            source.setInputStream(result);
        }
        
        return source;
    }

    /**
     * Serialize all datahandler in the Map. <br>
     * Serialized information is : <br>
     * -attachmentName <br>
     * -attachmentContentType <br>
     * -name <br>
     * -inputStream available bytes count <br>
     * -inputStream as a byteArray
     * 
     * @param attachments
     * @param s
     * @throws Exception
     */
    public static final void serializeAttachments(Map<String, DataHandler> attachments,
            ObjectOutputStream s) throws Exception {
        s.writeInt(attachments.size());

        for (Entry<String, DataHandler> attachment : attachments.entrySet()) {
            final InputStream binary = attachment.getValue().getInputStream();
            try {
                int available = binary.available();

                // write information about the attachment
                s.writeUTF(attachment.getKey());
                s.writeUTF(attachment.getValue().getContentType());
                s.writeUTF(attachment.getValue().getName());
                s.writeInt(available);

                // write attachment content with the default socket output
                // buffer size
                byte[] buff = new byte[SocketOptions.SO_SNDBUF];
                int n = 0;
                while (-1 != (n = binary.read(buff))) {
                    s.write(buff, 0, n);
                    s.flush();
                }
            } finally {
                binary.close();
            }
        }
    }

    /**
     * Deserialize all datahandlers in the stream and return a Map. <br>
     * Deserialized information is : <br>
     * -attachmentName <br>
     * -attachmentContentType <br>
     * -name <br>
     * -Stream available bytes count <br>
     * -Stream as a byteArray
     * 
     * @param s
     * @return
     * @throws IOException
     */
    public static final Map<String, DataHandler> deserializeAttachments(ObjectInputStream s)
            throws IOException {
        HashMap<String, DataHandler> attachments = new HashMap<String, DataHandler>();

        int attachmentsCount = s.readInt();

        for (int i = 0; i < attachmentsCount; i++) {
            // read information
            String attachmentName = s.readUTF();
            String contentType = s.readUTF();
            String name = s.readUTF();
            int available = s.readInt();

            // read content
            byte[] content = new byte[available];
            s.readFully(content);
            ByteArrayDataSource dataSource = new ByteArrayDataSource(content, contentType);
            dataSource.setName(name);

            // create the dataHandler
            DataHandler dataHandler = new DataHandler(dataSource);

            attachments.put(attachmentName, dataHandler);
        }

        return attachments;
    }

}
