/**
 * PETALS - PETALS Services Platform.
 * Copyright (c) 2005 EBM WebSourcing
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * $Id: NormalizedMessageImpl.java 22099 2010-04-21 15:33:25Z rnaudin $
 * -------------------------------------------------------------------------
 */

package org.ow2.petals.jbi.messaging.exchange;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import javax.activation.DataHandler;
import javax.jbi.messaging.MessagingException;
import javax.jbi.messaging.NormalizedMessage;
import javax.security.auth.Subject;
import javax.xml.transform.Source;

/**
 * JBI <code>NormalizedMessage</code> implementation.
 * 
 * @version $Rev: 22099 $ $Date: 2010-02-16 17:39:03 +0100 (mar. 16 févr. 2010)
 *          $
 * @since Petals 1.0
 * @author Adrien LOUIS, <a href="mailto:rmarins@fossilec.com">Rafael Marins</a>
 */
public class NormalizedMessageImpl implements NormalizedMessage, Serializable {

    /**
     * The message exchange property name to set the compression of the exchange
     * payload.
     */
    public final static String PROPERTY_COMPRESSION = "org.ow2.petals.transport.compress";

    /**
     * The unique serial UID.
     */
    private static final long serialVersionUID = 45438732L;

    /**
     * Attachments to the main message: id + content.
     */
    protected transient Map<String, DataHandler> attachments;

    /**
     * The XML "payload" message.
     */
    protected transient Source content;

    /**
     * Metadata associated with the message content. Map: property + value.
     */
    protected Map<String, Object> properties;

    /**
     * JAAS Security Subject.
     */
    protected Subject subject;

    /**
     * This default constructor creates a new empty
     * <code>NormalizedMessage</code>.
     */
    public NormalizedMessageImpl() {
        super();

        content = null;
        properties = new HashMap<String, Object>();
        attachments = new HashMap<String, DataHandler>();
    }

    /*
     * ---------------------------------------------------------------------- N
     * N o r m a l i z e d M e s s a g e M e t h o d s
     * ----------------------------------------------------------------------
     */

    /**
     * @inheritDoc
     */
    public void addAttachment(String id, DataHandler attachment) throws MessagingException {
        attachments.put(id, attachment);
    }

    /**
     * @inheritDoc
     */
    public DataHandler getAttachment(String id) {
        return attachments.get(id);
    }

    /**
     * @inheritDoc
     */
    public Set<?> getAttachmentNames() {
        return attachments.keySet();
    }

    /**
     * @inheritDoc
     */
    public Source getContent() {
        return content;
    }

    /**
     * @inheritDoc
     */
    public Object getProperty(String name) {
        return properties.get(name);
    }

    /**
     * @inheritDoc
     */
    public Set<?> getPropertyNames() {
        return properties.keySet();
    }

    /**
     * @inheritDoc
     */
    public Subject getSecuritySubject() {
        return subject;
    }

    /**
     * @inheritDoc
     */
    public void removeAttachment(String id) throws MessagingException {
        Object att = attachments.remove(id);

        if (att == null) {
            throw new MessagingException(id + " attachment does not exist.");
        }
    }

    /**
     * @inheritDoc
     */
    public void setContent(Source content) throws MessagingException {
        this.content = content;
    }

    /**
     * @inheritDoc
     */
    public void setProperty(String name, Object value) {
        if (value == null) {
            // case to remove properties not defined (null)
            properties.remove(name);

        } else {
            // define this property name with a new value
            properties.put(name, value);
        }
    }

    /**
     * @inheritDoc
     */
    public void setSecuritySubject(Subject securitySubject) {
        this.subject = securitySubject;
    }

    /**
     * Used to deserialize Content object
     * 
     * @param s
     * @throws IOException
     */
    private void readObject(ObjectInputStream s) throws IOException {
        try {
            s.defaultReadObject();

            attachments = MessageExchangeSerializer.deserializeAttachments(s);
            String compression = (String) getProperty(PROPERTY_COMPRESSION);
            content = MessageExchangeSerializer.deserializeContent(s, Boolean.valueOf(compression));
        } catch (ClassNotFoundException e) {
            throw new IOException(e.getClass() + ":" + e.getMessage());
        }
    }

    /**
     * <p>
     * Serialization process :
     * <ul>
     * <li>Serialization of all serializable fields of the NormalizedMessage</li>
     * <li>Serialization of all attachment</li>
     * <li>Finally, serialization of the content.</li>
     * </ul>
     * </p>
     * <p>
     * <b>Note</b>: Content must always be serialized at the end of the
     * serialization process. Because in the deserialization process, a new
     * content will be created with all ending bytes of the Stream.
     * </p>
     * 
     * @param s
     * @throws IOException
     */
    private void writeObject(ObjectOutputStream s) throws IOException {
        s.defaultWriteObject();
        try {
            MessageExchangeSerializer.serializeAttachments(attachments, s);
            String compression = (String) getProperty(PROPERTY_COMPRESSION);
            MessageExchangeSerializer.serializeContent(content, s, Boolean
                    .parseBoolean(compression));
        } catch (final Exception e) {
            throw (IOException) (new IOException(e.getMessage()).initCause(e));
        }
    }
}
