/****************************************************************************
 *
 * Copyright (c) 2006-2012, EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA 
 *
 *****************************************************************************/
 
package org.ow2.petals.jbi.messaging.exchange;

import java.net.URI;
import java.util.logging.Logger;

import javax.jbi.messaging.InOnly;
import javax.jbi.messaging.InOptionalOut;
import javax.jbi.messaging.InOut;
import javax.jbi.messaging.MessageExchange;
import javax.jbi.messaging.MessageExchangeFactory;
import javax.jbi.messaging.MessagingException;
import javax.jbi.messaging.RobustInOnly;
import javax.jbi.servicedesc.ServiceEndpoint;
import javax.xml.namespace.QName;

/**
 * This implementation creates {@link MessageExchangeWrapper} instances of {@link MessageExchange}.
 * @author Olivier Fabre - EBM WebSourcing
 */
public abstract class AbstractContextualMessageExchangeFactory implements MessageExchangeFactory {

    private LoggingUtil log;

    private ContextualMessageExchangeImplBuilder messageExchangeImplBuilder;

    protected AbstractContextualMessageExchangeFactory(final ServiceEndpoint consumerEndpoint,
            final ContextualMessageExchangeImplBuilder messageExchangeImplBuilder, final Logger logger) {
        assert consumerEndpoint != null;
        this.messageExchangeImplBuilder = messageExchangeImplBuilder;
        this.log = new LoggingUtil(logger);
    }

    /**
     * Return a <code>MessageExchangeDecorator</code> for Consumer, as
     * <code>InOnly</code>.
     * 
     * @see javax.jbi.messaging.MessageExchangeFactory#createInOnlyExchange()
     */
    public InOnly createInOnlyExchange() throws MessagingException {
        this.log.call();

        return (InOnly) buildExchangeDecorator(MEPPatternConstants.IN_ONLY.value());
    }

    /**
     * Return a <code>MessageExchangeDecorator</code> for Consumer, as
     * <code>InOptionalOut</code>.
     * 
     * @see javax.jbi.messaging.MessageExchangeFactory#createInOptionalOutExchange()
     */
    public InOptionalOut createInOptionalOutExchange() throws MessagingException {
        this.log.call();

        return (InOptionalOut) buildExchangeDecorator(MEPPatternConstants.IN_OPTIONAL_OUT.value());
    }

    /**
     * Return a <code>MessageExchangeDecorator</code> for Consumer, as
     * <code>InOut</code>.
     * 
     * @see javax.jbi.messaging.MessageExchangeFactory#createInOutExchange()
     */
    public InOut createInOutExchange() throws MessagingException {
        this.log.call();

        return (InOut) buildExchangeDecorator(MEPPatternConstants.IN_OUT.value());
    }

    /**
     * Return a <code>MessageExchangeDecorator</code> for Consumer, as
     * <code>RobustInOnly</code>.
     * 
     * @see javax.jbi.messaging.MessageExchangeFactory#createRobustInOnlyExchange()
     */
    public RobustInOnly createRobustInOnlyExchange() throws MessagingException {
        this.log.call();

        return (RobustInOnly) buildExchangeDecorator(MEPPatternConstants.ROBUST_IN_ONLY.value());
    }

    /**
     * @see javax.jbi.messaging.MessageExchangeFactory#createExchange(QName,
     *      QName)
     */
    public MessageExchange createExchange(QName serviceName, QName operationName)
            throws MessagingException {
        this.log.call("Service Name : " + serviceName + " - operationName : " + operationName);
        assert serviceName != null;
        assert operationName != null;

        MessageExchangeWrapper messageExchangeDecorator = buildExchangeDecorator(null);

        messageExchangeDecorator.setService(serviceName);
        messageExchangeDecorator.setOperation(operationName);

        return messageExchangeDecorator;
    }

    /**
     * Return a <code>MessageExchangeDecorator</code> for Consumer, as
     * <code>InOnlyExchange, InOutExchange,...</code>
     * 
     * @see javax.jbi.messaging.MessageExchangeFactory#createExchange(java.net.URI)
     */
    public MessageExchange createExchange(URI pattern) throws MessagingException {
        this.log.call(pattern);
        assert pattern != null;

        checkPatternIsJBISupportedPattern(pattern);

        return buildExchangeDecorator(pattern);
    }

    private void checkPatternIsJBISupportedPattern(URI pattern) throws MessagingException {
        if (!MessageExchangeImpl.IN_ONLY_PATTERN.equals(pattern)
                && !MessageExchangeImpl.ROBUST_IN_ONLY_PATTERN.equals(pattern)
                && !MessageExchangeImpl.IN_OUT_PATTERN.equals(pattern)
                && !MessageExchangeImpl.IN_OPTIONAL_OUT_PATTERN.equals(pattern)
                && !MEPPatternConstants.IN_ONLY.equals(pattern)
                && !MEPPatternConstants.ROBUST_IN_ONLY.equals(pattern)
                && !MEPPatternConstants.IN_OUT.equals(pattern)
                && !MEPPatternConstants.IN_OPTIONAL_OUT.equals(pattern)) {
            throw new MessagingException("This Message Exchange Pattern is not recognized by JBI.");
        }
    }

    private MessageExchangeWrapper buildExchangeDecorator(URI pattern) throws MessagingException {
        this.log.call(pattern);

        MessageExchangeImpl messageExchangeImpl = this.messageExchangeImplBuilder.build();

        setFactorySpecificTarget(messageExchangeImpl);

        return createMessageExchangeDecoratorForPattern(messageExchangeImpl, pattern);

    }

    protected void setFactorySpecificTarget(MessageExchangeImpl exchange) {
    }

    protected abstract MessageExchangeWrapper createMessageExchangeDecoratorForPattern(
            MessageExchangeImpl exchange, URI pattern);

    protected boolean isInOptionalOutPattern(URI pattern) {
        return MessageExchangeImpl.IN_OPTIONAL_OUT_PATTERN.equals(pattern)
                || MEPPatternConstants.IN_OPTIONAL_OUT.equals(pattern);
    }

    protected boolean isInOutPattern(URI pattern) {
        return MessageExchangeImpl.IN_OUT_PATTERN.equals(pattern)
                || MEPPatternConstants.IN_OUT.equals(pattern);
    }

    protected boolean isRobustInOnlyPattern(URI pattern) {
        return MessageExchangeImpl.ROBUST_IN_ONLY_PATTERN.equals(pattern)
                || MEPPatternConstants.ROBUST_IN_ONLY.equals(pattern);
    }

    protected boolean isInOnlyPattern(URI pattern) {
        return MessageExchangeImpl.IN_ONLY_PATTERN.equals(pattern)
                || MEPPatternConstants.IN_ONLY.equals(pattern);
    }

}
