/*
 * Copyright (c) 2009 EBM Websourcing, http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package org.ow2.petals.registry.api;

import java.util.List;

import org.ow2.petals.registry.api.exception.RegistryException;

/**
 * The operations allowed by the registry. This is an internal API to not be
 * used by clients since they do not care about the resources but only wnat to
 * get endpoints...
 * 
 * @author Christophe HAMERLING - eBM WebSourcing
 * 
 */
public interface Operations {

    /**
     * Put a resource in the registry.
     * 
     * @param path
     *            The path to store the resource into. Contains the parent
     *            directory and the resource identifier (ie : in
     *            /path/to/resource/r1 r1 is the resource identifier) if the
     *            resource already exists override, create a new one if not.
     * @param resource
     *            The resource to put
     */
    boolean put(String path, Resource resource, boolean propagate) throws RegistryException;

    /**
     * Get the element from its path only if the path refers to an element. If
     * the path is a directory, it will returns null, please refer to
     * {@link #getAll(String)}
     * 
     * @param path
     * @return the resource if found, otherwise it will throw an exception
     * @throws an
     *             exception there is no resource at the given path or if a
     *             problem occurs.
     */
    Resource get(String path, boolean remote) throws RegistryException;

    /**
     * Get all the resources in the given path only if the given path is a
     * directory. If it is, returns all the children, if not, return null... For
     * single resource retrieval, refer to {@link #get(String)}. Be carefull on
     * the implementation side when trying to get all the resources of all the
     * registries. The best way is that the getAll implementation just returns
     * the resources of the local registry ie its vision of the resources of the
     * network at the current time.
     * 
     * @param path
     * @return
     * @throws RegistryException
     *             if the path is not a directory path...
     */
    List<Resource> getAll(String path, boolean remote) throws RegistryException;

    /**
     * Delete the element from the registry. If the path is a directory, delete
     * all the children of the given path (and the path too...)
     * 
     * @param key
     * @return true if the resource has been deleted, false if not of if not
     *         found
     * 
     * @throws RegistryException
     */
    boolean delete(String path, boolean propagate) throws RegistryException;

    /**
     * Get a list of resources under the given path which assert that the
     * endpoint properties match the given ones.
     * 
     * @param query
     * @return
     * @throws RegistryException
     */
    List<Resource> query(Query query, boolean remote) throws RegistryException;

}
