/**
 * PETALS - PETALS Services Platform. Copyright (c) 2008 EBM Websourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * -------------------------------------------------------------------------
 * $Id$
 * -------------------------------------------------------------------------
 */
package org.ow2.petals.registry.api.config;

import java.io.IOException;
import java.io.InputStream;
import java.util.Iterator;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.petals.registry.api.exception.RegistryException;

import static org.ow2.petals.registry.api.config.Constants.DEFAULT_DB_C3P0_ENABLED;
import static org.ow2.petals.registry.api.config.Constants.DEFAULT_DB_CACHING_ENABLED;
import static org.ow2.petals.registry.api.config.Constants.DEFAULT_DB_DIALECT;
import static org.ow2.petals.registry.api.config.Constants.DEFAULT_DB_DRIVER;
import static org.ow2.petals.registry.api.config.Constants.DEFAULT_DB_PASSWORD;
import static org.ow2.petals.registry.api.config.Constants.DEFAULT_DB_URL;
import static org.ow2.petals.registry.api.config.Constants.DEFAULT_DB_USER;
import static org.ow2.petals.registry.api.config.Constants.DEFAULT_INCOMING_MANAGER;
import static org.ow2.petals.registry.api.config.Constants.DEFAULT_MESSAGE_RECEIVER;
import static org.ow2.petals.registry.api.config.Constants.DEFAULT_MESSAGE_SENDER;
import static org.ow2.petals.registry.api.config.Constants.DEFAULT_OUTGOING_MANAGER;
import static org.ow2.petals.registry.api.config.Constants.DEFAULT_REGISTRY_MODE;
import static org.ow2.petals.registry.api.config.Constants.DEFAULT_REPOSITORY;

/**
 * A loader from a properties file
 * 
 * @author Christophe HAMERLING - eBM WebSourcing
 * @author Charles CASADEI - eBM WebSourcing
 */
public class PropertiesConfigurationLoader implements ConfigurationLoader {

    /**
     * 
     */
    public static final String REGISTRY_TRANSPORT_TIMEOUT = "registry.transport.timeout";

    /**
     * 
     */
    public static final String REGISTRY_SYNCHRO_PERIOD = "registry.synchro.period";

    /**
     * 
     */
    public static final String REGISTRY_MODE = "registry.mode";

    /**
     * 
     */
    public static final String REGISTRY_DB_DIALECT = "registry.db.dialect";

    /**
     * 
     */
    public static final String REGISTRY_DB_USER = "registry.db.user";

    /**
     * 
     */
    public static final String REGISTRY_DB_PASSWORD = "registry.db.password";

    /**
     * 
     */
    public static final String REGISTRY_DB_URL = "registry.db.url";

    /**
     * 
     */
    public static final String REGISTRY_DB_DRIVER = "registry.db.driver";
    
    /**
     * 
     */
    public static final String REGISTRY_DB_CACHING_ENABLED = "registry.db.caching.enabled";

    /**
     * 
     */
    public static final String REGISTRY_DB_C3P0_ENABLED = "registry.db.c3p0.enabled";

    /**
     * 
     */
    public static final String REGISTRY_TOPOLOGY_FILE = "registry.topology.file";

    /**
     * 
     */
    public static final String REGISTRY_ROOTPATH = "registry.rootpath";

    /**
     * 
     */
    public static final String REGISTRY_CLIENT_MANAGER_URL = "registry.client.manager.url";

    /**
     * 
     */
    public static final String REGISTRY_CLIENT_MANAGER = "registry.client.manager";

    /**
     * 
     */
    public static final String REGISTRY_MESSAGE_RECEIVER_URL = "registry.message.receiver.url";

    /**
     * 
     */
    public static final String REGISTRY_MESSAGE_RECEIVER = "registry.message.receiver";

    /**
     * 
     */
    public static final String REGISTRY_MESSAGE_SENDER = "registry.message.sender";

    /**
     * 
     */
    public static final String REGISTRY_REPOSITORY = "registry.repository";

    /**
     * 
     */
    public static final String REGISTRY_OUTGOING_MANAGER = "registry.outgoing.manager";

    /**
     * 
     */
    public static final String REGISTRY_INCOMING_MANAGER = "registry.incoming.manager";

    /**
     * The class logger
     */
    private final Log log = LogFactory.getLog(PropertiesConfigurationLoader.class);

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.petals.registry.core.config.ConfigurationLoader#load(java.io.
     * InputStream)
     */
    public Configuration loadLocal(InputStream is) throws RegistryException {
        if (is == null) {
            if (this.log.isDebugEnabled()) {
                this.log.debug("Stream is null, loading from the default resource");
            }
            is = Thread.currentThread().getContextClassLoader().getResourceAsStream(
                    Constants.DEFAULT_LOCAL_CONFIGURATION);
        }

        if (is == null) {
            throw new RegistryException("Can not load configuration from a null input stream");
        }

        Properties props = new Properties();
        try {
            props.load(is);
        } catch (IOException e) {
            String msg = "Can not load the resource file";
            if (this.log.isErrorEnabled()) {
                this.log.error(msg, e);
            }
            throw new RegistryException(msg, e);
        }

        // get the registry name
        String name = props.getProperty("registry.name");

        Configuration configuration = new Configuration(name);
        // get the local registry implementation

        configuration.setRegistryTransportTimeout(props.getProperty(REGISTRY_TRANSPORT_TIMEOUT));
        configuration.setRegistrySynchroPeriod(props.getProperty(REGISTRY_SYNCHRO_PERIOD));

        configuration.setIncomingManagerClassName(props.getProperty(REGISTRY_INCOMING_MANAGER, DEFAULT_INCOMING_MANAGER));
        configuration.setOutgoingManagerClassName(props.getProperty(REGISTRY_OUTGOING_MANAGER, DEFAULT_OUTGOING_MANAGER));
        configuration.setRepositoryClassName(props.getProperty(REGISTRY_REPOSITORY, DEFAULT_REPOSITORY));

        configuration.setMessageReceiverClassName(props.getProperty(REGISTRY_MESSAGE_RECEIVER, DEFAULT_MESSAGE_RECEIVER));
        configuration.setMessageReceiverURL(props.getProperty(REGISTRY_MESSAGE_RECEIVER_URL));

        configuration.setMessageSenderClassName(props.getProperty(REGISTRY_MESSAGE_SENDER, DEFAULT_MESSAGE_SENDER));

        configuration.setClientManagerClassName(props.getProperty(REGISTRY_CLIENT_MANAGER));
        configuration.setClientManagerURL(props.getProperty(REGISTRY_CLIENT_MANAGER_URL));

        configuration.setRegistryMode(props.getProperty(REGISTRY_MODE, DEFAULT_REGISTRY_MODE));

        // the root path
        configuration.setRootPath(props.getProperty(REGISTRY_ROOTPATH));

        // the topology path
        configuration.setTopologyFile(props.getProperty(REGISTRY_TOPOLOGY_FILE));

        DBConfiguration dbConfiguration = new DBConfiguration();
        dbConfiguration.setDriver(props.getProperty(REGISTRY_DB_DRIVER, DEFAULT_DB_DRIVER));
        dbConfiguration.setUrl(props.getProperty(REGISTRY_DB_URL, DEFAULT_DB_URL));
        dbConfiguration.setPassword(props.getProperty(REGISTRY_DB_PASSWORD, DEFAULT_DB_PASSWORD));
        dbConfiguration.setUsername(props.getProperty(REGISTRY_DB_USER, DEFAULT_DB_USER));
        dbConfiguration.setDialect(props.getProperty(REGISTRY_DB_DIALECT, DEFAULT_DB_DIALECT));
        dbConfiguration.setCachingEnabled(props.getProperty(REGISTRY_DB_CACHING_ENABLED, DEFAULT_DB_CACHING_ENABLED).trim().equals("true"));
        dbConfiguration.setc3p0Enabled(props.getProperty(REGISTRY_DB_C3P0_ENABLED, DEFAULT_DB_C3P0_ENABLED).trim().equals("true"));
        configuration.setDbConfiguration(dbConfiguration);

        // extensions are not filtered, add all the data from the properties
        Iterator<?> keys = props.keySet().iterator();
        while (keys.hasNext()) {
            String elem = keys.next().toString();
            configuration.getExtensions().put(elem, props.getProperty(elem));
        }

        return configuration;
    }
}
