/**
 * PETALS - PETALS Services Platform. Copyright (c) 2009 EBM Websourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * -------------------------------------------------------------------------
 * $Id$
 * -------------------------------------------------------------------------
 */
package org.ow2.petals.registry.api.context;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.petals.registry.api.Constants;
import org.ow2.petals.registry.api.config.Configuration;
import org.ow2.petals.registry.api.config.RemoteConfiguration;
import org.ow2.petals.registry.api.config.TopologyLoader;
import org.ow2.petals.registry.api.config.TopologyLoaderFactory;
import org.ow2.petals.registry.api.exception.LifeCycleException;
import org.ow2.petals.registry.api.exception.RegistryException;
import org.ow2.petals.registry.api.lifecycle.Initializable;
import org.ow2.petals.registry.api.util.Assert;

/**
 * The context contains all the related things needed by the registry to work...
 * It is built from the {@link Configuration} data by the {@link ContextFactory}
 * .
 * 
 * @author Christophe HAMERLING - eBM WebSourcing
 * 
 */
public class Context implements Initializable {

    private static Log log = LogFactory.getLog(Context.class);

    /**
     * The configuration bean
     */
    private final Configuration configuration;

    /**
     * This is the initial topology loaded during bootstrap. This list is not
     * updated when remote registries are added at runtime to the remote
     * manager.
     */
    private List<RemoteConfiguration> topology;

    /**
     * The workspace
     */
    private final Workspace workspace;

    private boolean initialized;

    /**
     * Store the resgistry context instances. Since there can be multiple
     * registry in the same JVM, store them in a map. The key is the registry
     * name which is unique.
     */
    private static Map<String, Context> instances = new HashMap<String, Context>();

    private final Map<String, Object> properties;

    /**
     * Store the the context in order to potentially retrieve it everywhere in
     * the framework.
     * 
     * @param name
     * @param context
     */
    public static synchronized void store(Context context) {
        Assert.notNull(context);
        Assert.notNull(context.getConfiguration());
        Assert.notNull(context.getConfiguration().getName());
        instances.put(context.getConfiguration().getName(), context);
    }

    /**
     * 
     * @param registryName
     * @return
     */
    public static Context getContext(String registryName) {
        return instances.get(registryName);
    }

    /**
     * Creates a new instance of Context from the {@link Configuration}
     * 
     * @param configuration
     */
    public Context(Configuration configuration) throws RegistryException {
        Assert.notNull(configuration);
        this.properties = new HashMap<String, Object>();
        this.topology = new ArrayList<RemoteConfiguration>();
        // at this step the configuration name field can not be null !
        this.configuration = configuration;

        // Fix the root path...
        // workspace
        File home = null;
        String rootPath = configuration.getRootPath();
        if ((rootPath == null) || (rootPath.trim().length() == 0)) {
            // try to retrieve a home folder from system properties
            rootPath = System.getProperty(Constants.REGISTRY_HOME);
            if (rootPath == null) {
                String pwd = null;
                try {
                    pwd = new File(".").getCanonicalPath();
                } catch (IOException e1) {
                    pwd = new File(".").getAbsolutePath();
                }
                home = new File(pwd);
            } else {
                home = new File(rootPath);
            }
        } else {
            home = new File(rootPath);
        }
        configuration.setRootPath(home.getAbsolutePath());
        if (log.isInfoEnabled()) {
            log.debug("Loading with config : " + configuration.toString());
        }

        // TODO : put it elsewhere and handle exceptions
        if (configuration.getTopologyFile() != null) {
            File config = new File(configuration.getTopologyFile());
            if (config.exists()) {
                TopologyLoader loader = TopologyLoaderFactory.getLoader(config);
                if (loader != null) {
                    try {
                        this.topology.addAll(loader.load(new FileInputStream(config)));
                    } catch (FileNotFoundException e) {
                        log.warn(e.getMessage());
                    } catch (RegistryException e) {
                        log.warn(e.getMessage());
                    }
                }
            } else {
                throw new RegistryException("Can not find specified topology file "
                        + configuration.getTopologyFile());
            }
        }

        this.workspace = new Workspace(new File(configuration.getRootPath()), configuration
                .getName());

        Context.store(this);
    }

    /**
     * @return the configuration
     */
    public Configuration getConfiguration() {
        return this.configuration;
    }

    /**
     * @return the workspace
     */
    public Workspace getWorkspace() {
        return this.workspace;
    }

    public boolean isLoaded() {
        return this.initialized;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.ow2.petals.registry.api.lifecycle.Initializable#init()
     */
    public void init() throws LifeCycleException {
        // load the workspace
        this.workspace.init();
        Context.store(this);
        this.initialized = true;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.ow2.petals.registry.api.lifecycle.Initializable#isInitialized()
     */
    public boolean isInitialized() {
        return this.initialized;
    }

    /**
     * @return the properties
     */
    public Map<String, Object> getProperties() {
        return this.properties;
    }

    /**
     * @return the topology
     */
    public List<RemoteConfiguration> getTopology() {
        return this.topology;
    }

    /**
     * @param topology
     *            the topology to set
     */
    public void setTopology(List<RemoteConfiguration> topology) {
        this.topology = topology;
    }

}
