/**
 * PETALS - PETALS Services Platform. Copyright (c) 2008 EBM Websourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * -------------------------------------------------------------------------
 * $Id$
 * -------------------------------------------------------------------------
 */
package org.ow2.petals.registry.api.lifecycle;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.petals.registry.api.context.Context;
import org.ow2.petals.registry.api.exception.LifeCycleException;

/**
 * The abstract life cycle
 * 
 * @author Christophe HAMERLING - eBM WebSourcing
 * 
 */
public abstract class AbstractLifeCycle implements LifeCycle {

    /**
     * The initial state is equals not not initialized
     */
    private transient STATE state = STATE.NOT_INITIALIZED;

    /**
     * The class logger
     */
    protected Log log = LogFactory.getLog(AbstractLifeCycle.class);
    
    /**
     * Creates a new instance of AbstractLifeCycle
     */
    public AbstractLifeCycle() {
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.ow2.petals.registry.core.LifeCycle#init()
     */
    public final void init(Context context) throws LifeCycleException {
        if (this.log.isDebugEnabled()) {
            this.log.debug("Initializing " + this.getClass().getName());
        }

        try {

            if (this.state != STATE.NOT_INITIALIZED) {
                throw new LifeCycleException("Can not initialize in state " + this.state);
            }

            if (this.isInitialized()) {
                return;
            }

            this.state = STATE.INITIALIZING;
            this.doInit(context);
            if (this.log.isDebugEnabled()) {
                this.log.debug("Initialized");
            }
            this.state = STATE.INITIALIZED;
        } catch (LifeCycleException e) {
            if (this.log.isErrorEnabled()) {
                this.log.error("failed", e);
            }
            this.state = STATE.FAILED;
            throw e;
        } catch (Error e) {
            if (this.log.isErrorEnabled()) {
                this.log.error("Failed", e);
            }
            this.state = STATE.FAILED;
            throw e;
        }
    }

    /**
     * Do the initialization
     * 
     * @param params the initialization parameters
     * @throws LifeCycleException
     */
    protected void doInit(Context context) throws LifeCycleException {
        
    }

    /**
     * 
     * @throws LifeCycleException
     */
    protected void doStart() throws LifeCycleException {
        
    }
    /**
     * 
     * @throws LifeCycleException
     */
    protected void doStop() throws LifeCycleException {
        
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.ow2.petals.registry.core.LifeCycle#start()
     */
    public final void start() throws LifeCycleException {
        if (this.log.isInfoEnabled()) {
            this.log.debug("Starting " + this.getClass().getName());
        }

        if (!this.isInitialized()) {
            throw new LifeCycleException("Can not start if not initialized");
        }

        try {
            if (this.state == STATE.STARTED) {
                return;
            }
            this.state = STATE.STARTING;
            this.doStart();
            this.log.debug("Started");
            this.state = STATE.STARTED;
        } catch (LifeCycleException e) {
            if (this.log.isErrorEnabled()) {
                this.log.error("Fail to start", e);
            }
            this.state = STATE.FAILED;
            throw e;
        } catch (Error e) {
            if (this.log.isErrorEnabled()) {
                this.log.error("Failed to start", e);
            }
            this.state = STATE.FAILED;
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.ow2.petals.registry.core.LifeCycle#stop()
     */
    public final void stop() throws LifeCycleException {
        if (this.log.isInfoEnabled()) {
            this.log.debug("Stopping " + this.getClass().getName());
        }

        if (!this.isRunning()) {
            throw new LifeCycleException("Can not stop if not running...");
        }

        try {
            if ((this.state == STATE.STOPPING) || (this.state == STATE.STOPPED)) {
                return;
            }
            this.state = STATE.STOPPING;
            this.doStop();
            if (this.log.isDebugEnabled()) {
                this.log.debug("Stopped");
            }
            this.state = STATE.STOPPED;
        } catch (LifeCycleException e) {
            if (this.log.isErrorEnabled()) {
                this.log.error("failed", e);
            }
            this.state = STATE.FAILED;
            throw e;
        } catch (Error e) {
            if (this.log.isErrorEnabled()) {
                this.log.error("Failed", e);
            }
            this.state = STATE.FAILED;
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.ow2.petals.registry.core.LifeCycle#isRunning()
     */
    public boolean isRunning() {
        return (this.state == STATE.STARTED) || (this.state == STATE.STARTING);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.ow2.petals.registry.core.LifeCycle#isStarted()
     */
    public boolean isStarted() {
        return this.state == STATE.STARTED;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.ow2.petals.registry.core.LifeCycle#isStopped()
     */
    public boolean isStopped() {
        return this.state == STATE.STOPPED;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.ow2.petals.registry.core.LifeCycle#isInitialized()
     */
    public boolean isInitialized() {
        return this.state == STATE.INITIALIZED;
    }
    
    public boolean isNotInitialized() {
        return this.state == STATE.NOT_INITIALIZED;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.ow2.petals.registry.core.LifeCycle#getState()
     */
    public STATE getState() {
        return this.state;
    }
    
    public void setState(STATE state) {
        this.state = state;
    }
}
