/**
 * PETALS - PETALS Services Platform. Copyright (c) 2008 EBM Websourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * -------------------------------------------------------------------------
 * $Id$
 * -------------------------------------------------------------------------
 */
package org.ow2.petals.registry.client;

import java.util.ArrayList;
import java.util.List;

import org.ow2.petals.registry.api.Endpoint;
import org.ow2.petals.registry.api.LocalRegistry;
import org.ow2.petals.registry.api.Query;
import org.ow2.petals.registry.api.Resource;
import org.ow2.petals.registry.api.exception.RegistryException;
import org.ow2.petals.registry.api.factory.ResourceFactory;
import org.ow2.petals.registry.api.util.ResourceUtil;
import org.ow2.petals.registry.client.api.RegistryClient;

/**
 * This is just a facade to the registry in order to avoid Resource manipulation
 * on the client side.
 * 
 * @author Christophe HAMERLING - eBM WebSourcing
 * 
 */
public final class RegistryClientImpl implements RegistryClient {

    private final LocalRegistry localRegistry;

    /**
     * Creates a new instance of RegistryClientImpl
     */
    public RegistryClientImpl(LocalRegistry localRegistry) {
        this.localRegistry = localRegistry;
    }

    public void delete(String path, boolean b) throws RegistryException {
        if (path == null) {
            throw new RegistryException("Path is null");
        }
        this.localRegistry.delete(path, b);
    }

    public Endpoint get(String path, boolean b) throws RegistryException {
        if ((path == null) || ResourceUtil.isDirectory(path)) {
            throw new RegistryException("Bad path (null or directory)");
        }

        Endpoint result = null;
        Resource r = this.localRegistry.get(path, b);
        if (r != null) {
            result = r.getEndpoint();
        }
        return result;
    }

    public boolean put(String path, Endpoint endpoint, boolean b) throws RegistryException {
        if ((path == null) || ResourceUtil.isDirectory(path)) {
            throw new RegistryException("Bad path (null or directory)");
        }

        Resource r = ResourceFactory.createResource(endpoint);
        return this.localRegistry.put(path, r, b);
    }

    public List<Endpoint> getAll(String path, boolean b) throws RegistryException {
        if ((path == null) || !ResourceUtil.isDirectory(path)) {
            throw new RegistryException("Bad path (null or not directory)");
        }
        List<Endpoint> result = new ArrayList<Endpoint>();
        List<Resource> resources = this.localRegistry.getAll(path, b);
        if (resources != null) {
            result = new ArrayList<Endpoint>(resources.size());
            for (Resource resource : resources) {
                org.ow2.petals.registry.api.Endpoint ep = resource.getEndpoint();
                if (ep != null) {
                    result.add(ep);
                }
            }
        }
        return result;
    }

    public List<Endpoint> lookup(Query query, boolean b) throws RegistryException {
        if (query == null) {
            throw new RegistryException("Query can not be null");
        }
        List<Endpoint> result = new ArrayList<Endpoint>();
        List<Resource> r = this.localRegistry.query(query, b);
        if (r != null) {
            for (Resource resource : r) {
                org.ow2.petals.registry.api.Endpoint ep = resource.getEndpoint();
                if (ep != null) {
                    result.add(ep);
                }
            }
        }
        return result;
    }

    public LocalRegistry getLocalRegistry() {
        return this.localRegistry;
    }
}
