/**
 * PETALS - PETALS Services Platform. Copyright (c) 2009 EBM WebSourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 */

package org.ow2.petals.regulation;

import java.io.File;
import java.io.InputStream;
import java.io.StringWriter;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;

import org.ow2.petals.regulation.generated.Regulation;
import org.xml.sax.SAXException;

/**
 * The Regulation builder allows to marshall/unmarshall the
 * petalsRegulation.xml file.
 * 
 * @author rnaudin - EBM WebSourcing
 * 
 */
public final class RegulationBuilder {

    /**
     * XSD schema
     */
    public static final String NAMESPACE_XSD = "http://www.w3.org/2001/XMLSchema";

    /**
     * Topology schema
     */
    public static final String REGULATION_XSD = "petalsRegulation.xsd";

    /**
     * The unmarshaller.
     */
    private static Unmarshaller unmarshaller;

    /**
     * The exception raised during JAXB unique unmarshaller creation.
     */
    private static RegulationException unmarshCreationEx;

    /**
     * The JAXB unique marshaller.
     */
    private static Marshaller marshaller = null;

    /**
     * The exception raised during JAXB unique marshaller creation.
     */
    private static RegulationException marshallerCreationEx = null;

    /**
     * Static object initialization
     */
    static {

        final InputStream schemaInputStream = RegulationBuilder.class.getResourceAsStream("/"
                + REGULATION_XSD);

        try {
            final SchemaFactory factory = SchemaFactory.newInstance(NAMESPACE_XSD);
            final Source[] schemas = new StreamSource[] { new StreamSource(schemaInputStream) };
            final Schema schema = factory.newSchema(schemas);

            try {
                final JAXBContext jaxbContext = JAXBContext
                        .newInstance(new Class[] { Regulation.class });
                unmarshaller = jaxbContext.createUnmarshaller();
                unmarshaller.setSchema(schema);
            } catch (final JAXBException e) {
                unmarshCreationEx = new RegulationException(
                        "Failed to create the JAXB unmarshaller", e);
            }

            // Marshaller initialization
            try {
                final JAXBContext jaxbContext = JAXBContext
                        .newInstance(new Class[] { Regulation.class });
                marshaller = jaxbContext.createMarshaller();
                marshaller.setSchema(schema);
            } catch (final JAXBException e) {
                marshallerCreationEx = new RegulationException(
                        "Failed to create the JAXB marshaller", e);
            }

        } catch (final SAXException e) {
            final RegulationException schemaCreationEx = new RegulationException(
                    "Failed to parse resource '" + REGULATION_XSD + "'", e);
            unmarshCreationEx = new RegulationException("Failed to create the JAXB unmarshaller",
                    schemaCreationEx);
            marshallerCreationEx = new RegulationException("Failed to create the JAXB marshaller",
                    schemaCreationEx);
        }
    }

    /**
     * No constructor available.
     */
    private RegulationBuilder() {
        // NOP
    }

    /**
     * Create the system state object.
     * 
     * @param xmlFileName
     *            the xml file name
     * @return the topology
     * @throws RegulationException
     *             impossible to create the system state
     */
    public static Regulation createRegulation(final String xmlFileName) throws RegulationException {
        return RegulationBuilder.createRegulation(new File(xmlFileName));
    }

    /**
     * Create the system state object.
     * 
     * @param xmlFile
     *            the xml file
     * @return the system state
     * @throws RegulationException
     *             impossible to create the system state
     */
    public static Regulation createRegulation(final File xmlFile) throws RegulationException {

        if (unmarshaller == null) {
            throw unmarshCreationEx;
        } else {
            try {
                final JAXBElement<Regulation> root;
                // The default Xerces unmarshaller is not thread safe
                synchronized (unmarshaller) {
                    root = unmarshaller.unmarshal(new StreamSource(xmlFile), Regulation.class);
                }

                return root.getValue();
            } catch (final JAXBException e) {
                throw new RegulationException("Failed to build Regulation", e);
            }
        }
    }

    /**
     * Build the XML String from the System State in Java classes form.
     * 
     * @param regulationClass
     * @return
     * @throws RegulationException
     *             The exception raised during the marshaller creation or the
     *             exception raised during the build of the XML string.
     */
    public static String buildXmlStringRegulation(final Regulation regulationClass)
            throws RegulationException {

        if (marshaller == null) {
            throw marshallerCreationEx;
        } else {
            try {
                final StringWriter stringWriter = new StringWriter();
                // The default Xerces marshaller is not thread safe
                synchronized (marshaller) {
                    marshaller.marshal(regulationClass, stringWriter);
                }

                return stringWriter.toString();
            } catch (JAXBException e) {
                throw new RegulationException(
                        "Failed to build XML binding from Regulation Java classes", e);
            }
        }
    }

    /**
     * Build the XML File from the Regulation in Java classes form.
     * 
     * @param regulationClass
     * @param regulationFile
     * @throws RegulationException
     *             The exception raised during the marshaller creation or the
     *             exception raised during the build of the XML file.
     */
    public static void buildXmlFileRegulation(final Regulation regulationClass,
            final File regulationFile) throws RegulationException {

        if (marshaller == null) {
            throw marshallerCreationEx;
        } else {
            try {
                // The default Xerces marshaller is not thread safe
                synchronized (marshaller) {
                    marshaller.marshal(regulationClass, regulationFile);
                }
            } catch (JAXBException e) {
                throw new RegulationException(
                        "Failed to build XML binding from Regulation Java classes", e);
            }
        }
    }
}
