/****************************************************************************
 *
 * Copyright (c) 2009-2012, EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA 
 *
 *****************************************************************************/
 
package com.ebmwebsourcing.easybpel.model.bpel.impl.runtime;

import java.util.List;
import java.util.logging.Logger;

import org.jdom.Attribute;
import org.jdom.Content;
import org.jdom.Element;
import org.jdom.IllegalAddException;
import org.jdom.Namespace;
import org.jdom.Text;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;
import org.ow2.easywsdl.schema.api.SchemaElement;

import com.ebmwebsourcing.easybpel.model.bpel.api.BPELProcess;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.element.elements4assign.Copy;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.element.elements4assign.From;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.element.elements4assign.To;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.element.elements4assign.ToImpl;
import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.Execution;
import com.ebmwebsourcing.easyviper.core.api.engine.expression.Assigner;

/**
 * @author Adrien Louis - EBM WebSourcing
 * @author Nicolas Salatge - EBM WebSourcing
 */
public class BPELAssigner implements Assigner {

    private Logger log = Logger.getLogger(BPELAssigner.class.getName());

    public BPELAssigner(final BPELProcess bpeldefinition) {
    }

    public void setLog(final Logger logger) {
        this.log = logger;
    }

    @Override
    public void affect(Execution execution, final Object left, final Object right)
            throws CoreException {
        assert left instanceof To;
        assert right instanceof From;
        From from = (From) right;
        To to = (To) left;

        try {
            // below is implem of replacement rules (spec section 8.4.2)
            Object destination = to.resolve(execution);
            Object source = from.resolve(execution);

            if (isEII(source) && isEII(destination)) {
                SchemaElement parent = ((ToImpl) to).getParent();
                boolean keepSrcElementName = false;
                if (parent instanceof Copy) {
                    Copy copyParent = (Copy) parent;
                    keepSrcElementName = copyParent.getKeepSrcElementName();
                }
                replaceElement((Element) source, (Element) destination, keepSrcElementName);
                log.fine("Assignement resultat:  set source (" + ((Element)source).getName() + ") in destination (" +  ((Element)destination).getName() + ")).\nNew value of destination:\n" + new XMLOutputter(Format.getPrettyFormat()).outputString((Element) destination));
            } else {
                replaceContent(source, destination);
                log.fine("Assignement resultat: destination = " + (isEII(destination) ? "New value of destination:\n" + new XMLOutputter(Format.getPrettyFormat()).outputString((Element) destination) : destination));
            }            
        } catch (Throwable t) {
            t.printStackTrace();
        }

    }

    private final void replaceContent(Object source, Object destination) {
        if (isEII(destination)) {
            Element destinationEII = (Element) destination;
            destinationEII.removeContent();
            if (source instanceof String) {
                destinationEII.addContent((String) source);
            } else if (source instanceof Text) {
                Text textSource = (Text) source;
                destinationEII.addContent((Text) textSource.clone());
            } else if (source instanceof Number) {
                destinationEII.addContent(String.valueOf(source));
            } else {
                assert false;
            }
        } else if (isAII(destination)) {
            String sourceTII = asStringTII(source);
            // TODO : sourceTII should be normalized (see spec).
            Attribute destinationAII = (Attribute) destination;
            destinationAII.setValue(sourceTII);
        } else if (isTII(destination)) {
            String sourceTII = asStringTII(source);
            Text destinationTII = (Text) destination;
            destinationTII.setText(sourceTII);
        } else {
            assert false;
        }
    }

    private String asStringTII(Object source) {
        if (source instanceof String) {
            return (String) source;
        } else if (source instanceof Content) {
            return ((Content) source).getValue();
        } else if (source instanceof Number) {
            return String.valueOf(source);
        } else {
            assert false;
        }
        return null;
    }


    @SuppressWarnings("unchecked")
    private final void replaceElement(Element source, Element destination, 
            boolean keepSrcElementName) {
        if (keepSrcElementName) {
            destination.setName(source.getName());
            destination.setNamespace(source.getNamespace());
        }
        // remove all attributes and all content from existing destination
        destination.removeContent();
        destination.getAttributes().clear();
        
        // add all cloned source attributes, content, and additional namespaces 
        for (Content c : (List<Content>) source.getContent()) {
            destination.addContent(((Content) c.clone()).detach());
        }
        for (Attribute a : (List<Attribute>) source.getAttributes()) {
            destination.setAttribute(((Attribute) a.clone()).detach());
        }
        for (Namespace ns : (List<Namespace>) source.getAdditionalNamespaces()) {
            if (ns.getPrefix().isEmpty()) continue;
            destination.addNamespaceDeclaration(ns);
        }
    }

    private final boolean isEII(Object obj) {
        return obj instanceof Element;
    }

    private final boolean isAII(Object obj) {
        return obj instanceof Attribute;
    }

    private final boolean isTII(Object obj) {
        return (obj instanceof String) || (obj instanceof Text);
    }
    
}
