/****************************************************************************
 *
 * Copyright (c) 2009-2012, EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA 
 *
 *****************************************************************************/

package com.ebmwebsourcing.easybpel.model.bpel.impl.runtime;

import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import org.jaxen.JaxenException;
import org.jaxen.SimpleNamespaceContext;
import org.jaxen.SimpleVariableContext;
import org.jaxen.XPath;
import org.jaxen.XPathFunctionContext;
import org.jaxen.jdom.JDOMXPath;
import org.jdom.Element;
import org.jdom.Namespace;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;
import org.ow2.easywsdl.schema.api.abstractElmt.AbstractSchemaElementImpl;
import org.petalslink.abslayer.service.api.Description;
import org.petalslink.abslayer.service.api.Message;
import org.petalslink.abslayer.service.api.Part;

import com.ebmwebsourcing.easybpel.model.bpel.api.BPELProcess;
import com.ebmwebsourcing.easybpel.model.bpel.api.Constants;
import com.ebmwebsourcing.easybpel.model.bpel.api.variable.ElementVariable;
import com.ebmwebsourcing.easybpel.model.bpel.api.variable.MessageTypeVariable;
import com.ebmwebsourcing.easybpel.model.bpel.api.variable.TypeVariable;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TVariable;
import com.ebmwebsourcing.easybpel.model.bpel.impl.BPELProcessImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.exception.SelectionFailureException;
import com.ebmwebsourcing.easybpel.model.bpel.impl.variable.AbstractVariableImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.DoXslTransformFunction;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.GetVariablePropertyFunction;
import com.ebmwebsourcing.easyviper.core.api.engine.Execution;
import com.ebmwebsourcing.easyviper.core.api.engine.ExpressionEvaluator;
import com.ebmwebsourcing.easyviper.core.api.engine.Scope;
import com.ebmwebsourcing.easyviper.core.api.engine.expression.Expression;
import com.ebmwebsourcing.easyviper.core.api.engine.variable.Variable;

public class ExpressionEvaluatorImpl implements ExpressionEvaluator {

	private final BPELProcess bpelProcess;
	private final Scope scope;

	private Logger log = Logger.getLogger(this.getClass().getCanonicalName());

	public ExpressionEvaluatorImpl(BPELProcess bpelProcess, Scope scope) {
		this.bpelProcess = bpelProcess;
		this.scope = scope;

	}

	@Override
	public boolean evaluateAsBoolean(Execution execution, Expression expression) {
		try {

			XPath xpath = createXPath(execution, expression);
			return xpath.booleanValueOf(null);
		} catch (JaxenException e) {
			e.printStackTrace();
			throw new SelectionFailureException(expression);
		}
	}

	@Override
	public int evaluateAsInteger(Execution execution, Expression expression) {
		try {
			XPath xpath = createXPath(execution, expression);
			return xpath.numberValueOf(null).intValue();
		} catch (JaxenException e) {
			throw new SelectionFailureException(expression);
		}
	}

	@Override
	public String evaluateAsString(Execution execution, Expression expression) {
		try {
			XPath xpath = createXPath(execution, expression);
			return xpath.stringValueOf(null);
		} catch (JaxenException e) {
			throw new SelectionFailureException(expression);
		}
	}

	@Override
	public Object evaluateAsNode(Execution execution, Expression expression) {
		return evaluateAsNode(execution, expression, null);
	}


	@Override
	public Object evaluateAsNode(Execution execution, Expression expression, Object contextNode) {
		try {
			log.finest("XPATH = " + expression.getContent());
			XPath xpath = createXPath(execution, expression);
			Object result = xpath.selectSingleNode(contextNode);
			log.finest("XPATH RESULT = " + result);
			return result;
		} catch (JaxenException e) {
			e.printStackTrace();
			throw new SelectionFailureException(expression);
		}
	}


	@SuppressWarnings("unchecked")
	private final XPath createXPath(Execution execution, Expression expression) {

		XPathFunctionContext functionContext = new XPathFunctionContext();
		functionContext.registerFunction(Constants.BPEL_20_EXECUTABLE_NAMESPACE, 
				"doXslTransform", new DoXslTransformFunction(bpelProcess.getDocumentBaseURI()));
		functionContext.registerFunction(Constants.BPEL_20_EXECUTABLE_NAMESPACE, 
				"getVariableProperty", new GetVariablePropertyFunction(execution));

		SimpleVariableContext variableContext = new SimpleVariableContext();
		XPath xpath;
		try {
			xpath = new JDOMXPath(expression.getContent());
		} catch (JaxenException e) {
			throw new SelectionFailureException(expression);
		}

		// TODO : add custom functions
		SimpleNamespaceContext namespaceContext = new SimpleNamespaceContext();
		BPELProcessImpl bpelProcess = findBpelProcess(expression);
		if (bpelProcess != null) {
			for (Description desc : bpelProcess.getImports().getDescriptions()) {
				for (Map.Entry<String, String> namespaceBinding :
					desc.getNamespaces().entrySet()) {
					//                    log.finest("::" + namespaceBinding.getKey() + "=" + namespaceBinding.getValue());
					namespaceContext.addNamespace(namespaceBinding.getKey(), namespaceBinding.getValue());
				}
			}
			Map<String, String> namespaceBindings = bpelProcess.getNamespaceContext().getNamespaces();
			for (Map.Entry<String, String> namespaceBinding : namespaceBindings.entrySet()) {
				//                log.finest("!!" + namespaceBinding.getKey() + "=" + namespaceBinding.getValue());
				namespaceContext.addNamespace(namespaceBinding.getKey(), namespaceBinding.getValue());
			}
		}

		if (scope != null) {

			Map<String, Variable> variables = scope.getInScopeVariables();
			for (Map.Entry<String, Variable> entry : variables.entrySet()) {
				String variableName = entry.getKey();
				assert variableName != null;
				Variable bpelVariable = entry.getValue();
				//log.finest("-*-*- DECL var " + bpelVariable.getName());
				//log.finest("-*-*- DECL value " + bpelVariable);

				if (bpelVariable instanceof MessageTypeVariable) {
					//log.finest("     type MessageType Variable");
					// if variable is a message, each part of message must be
					// declared in context...
					MessageTypeVariable messageTypeVariable = (MessageTypeVariable) bpelVariable;

					Element rootMessageElement = messageTypeVariable.getValue(execution);
					Part[] parts = retrievePartsFromMessageElement((AbstractVariableImpl) bpelVariable);
					List<Element> children = (List<Element>) rootMessageElement.getChildren();
					assert children.size() == parts.length;

					int i = 0;
					for (Element partElement : children) {

						// declare element namespace
						if (!partElement.getNamespacePrefix().isEmpty()) {
							namespaceContext.addNamespace(partElement.getNamespacePrefix(),
									partElement.getNamespaceURI());
						}
						// declare additional namespaces in namespace context...
						for (Namespace additionalNamespace : (List<Namespace>) partElement
								.getAdditionalNamespaces()) {
							namespaceContext.addNamespace(additionalNamespace.getPrefix(),
									additionalNamespace.getURI());
						}

						String partName = parts[i].getQName().getLocalPart();
						String partVariableName = variableName + "." + partName;
						log.finest("var as element : "+partVariableName+"  "+partElement+"  "+variableContext);
						String nms = partElement.getNamespaceURI();
						
						declareVariableInVariableContext(partVariableName, partElement, variableContext,nms );
					//	declareVariableInVariableContext(partVariableName, partElement, variableContext);
						++i;
					}
				} else if (bpelVariable instanceof ElementVariable) {
					//log.finest("     type Element Variable");
					// if variable is a element, variable is directly declared in context
					Element variableValue = ((ElementVariable) bpelVariable).getValue(execution);

					declareVariableInVariableContext(variableName, variableValue, variableContext, variableValue.getNamespace().getURI());
				} else if (bpelVariable instanceof TypeVariable) {
					//log.finest("     type Type Variable");
					Object variableValue = ((TypeVariable) bpelVariable).getValue(execution);

					//log.finest("Value of "+variableName+" "+variableValue+" "+variableValue.getClass().getName()+ "context : "+variableContext);
					declareVariableInVariableContext(variableName, variableValue, variableContext);

				} else {
					//log.finest("     type Variable");
					assert false;
				}
			}
		}
		xpath.setNamespaceContext(namespaceContext);
		xpath.setVariableContext(variableContext);
		xpath.setFunctionContext(functionContext);
		return xpath;

	}


	private void declareVariableInVariableContext(String variableName, Object variableValue,
			SimpleVariableContext variableContext, String namespace) {

		variableContext.setVariableValue(variableName, variableValue);
		variableContext.setVariableValue(namespace, variableName, variableValue);
	}

	private void declareVariableInVariableContext(String variableName, Object variableValue,
			SimpleVariableContext variableContext) {
		//        log.finest("************ " + variableName);
		//        if (variableValue instanceof Element) {
		//            try {
		//                new XMLOutputter(Format.getPrettyFormat()).output((Element) variableValue, System.err);
		//                log.finest("");
		//            } catch (IOException e) {
		//                throw new RuntimeException(e);
		//            }
		//            
		//        } else if (variableValue instanceof Text){
		//            log.finest(variableValue);
		//            
		//        }
		variableContext.setVariableValue(variableName, variableValue);
		//variableContext.setVariableValue("http://petals.ow2.org/crisis", variableName, variableValue);
	}

	@SuppressWarnings("rawtypes")
	private final BPELProcessImpl findBpelProcess(Expression expression) {
		assert expression != null;
		AbstractSchemaElementImpl<?> elementImpl = (AbstractSchemaElementImpl) expression;
		if (!(elementImpl.getTopParent() instanceof BPELProcessImpl)) return null;
		BPELProcessImpl bpelProcess = (BPELProcessImpl) elementImpl.getTopParent();
		return bpelProcess;
	}

	private static final Part[] retrievePartsFromMessageElement(AbstractVariableImpl bpelVariable) {
		TVariable varDef = ((AbstractSchemaElementImpl<TVariable>) bpelVariable).getModel();
		// find message definition in process imported WSDLs
		BPELProcessImpl bpelProcess = (BPELProcessImpl) bpelVariable.getTopParent();
		Message message = bpelProcess.getImports().findMessage(varDef.getMessageType());
		return message.getParts();
	}

}
