
package com.ebmwebsourcing.easycommons.io;

import java.io.File;
import java.io.IOException;
import java.util.Random;

import static java.io.File.separator;

public final class FileSystemHelper {

    private static final Random random = new Random();

    private FileSystemHelper() {
    }

    private static final File getSystemTempDir() {
        String tmpDirPath = System.getProperty("java.io.tmpdir");
        assert tmpDirPath != null;
        File tmpDir = new File(tmpDirPath);
        assert tmpDir != null;
        return tmpDir;
    }

    public static File createTempDir() throws IOException {
        return createTempDir(getSystemTempDir(), "tmp");
    }

    public static File createTempDir(String prefix) throws IOException {
        return createTempDir(getSystemTempDir(), prefix);
    }

    public static File createTempDir(File parentDir, String prefix) throws IOException {
        File tempDir = null;
        while (true) {
            tempDir = new File(parentDir, prefix + random.nextLong());
            if (!tempDir.exists())
                break;
        }
        if (!tempDir.mkdir()) {
            throw new IOException(String.format("Impossible to create temp dir '%s'",
                    tempDir.getAbsolutePath()));
        }
        tempDir.deleteOnExit();
        return tempDir;
    }

    /**
     * Cleans a directory without deleting it.
     * 
     * @param directory
     *            directory to clean
     * @throws IOException
     *             in case cleaning is unsuccessful
     */
    public static void cleanDirectory(File directory) throws IOException {
        assert directory != null;
        if (!directory.exists()) {
            String message = directory + " does not exist";
            throw new IOException(message);
        }
        if (!directory.isDirectory()) {
            String message = directory + " is not a directory";
            throw new IOException(message);
        }

        File[] files = directory.listFiles();
        if (files == null) { // null if security restricted
            throw new IOException("Failed to list contents of " + directory);
        }

        IOException exception = null;
        for (int i = 0; i < files.length; i++) {
            File file = files[i];
            try {
                forceDelete(file);
            } catch (IOException ioe) {
                exception = ioe;
            }
        }

        if (null != exception) {
            throw exception;
        }
    }

    /**
     * Deletes a file. If file is a directory, delete it and all
     * sub-directories.
     * <p>
     * The difference between File.delete() and this method are:
     * <ul>
     * <li>A directory to be deleted does not have to be empty.</li>
     * <li>You get exceptions when a file or directory cannot be deleted.
     * (java.io.File methods returns a boolean)</li>
     * </ul>
     * 
     * @param file
     *            file or directory to delete, must not be <code>null</code>
     * @throws NullPointerException
     *             if the directory is <code>null</code>
     * @throws IOException
     *             in case deletion is unsuccessful
     */
    public static void forceDelete(File file) throws IOException {
        assert file != null;
        if (file.isDirectory()) {
            deleteDirectory(file);
        } else {
            if (!file.exists()) {
                return;
            }
            if (!file.delete()) {
                String message = "Unable to delete file: " + file;
                throw new IOException(message);
            }
        }
    }

    /**
     * Deletes a directory recursively.
     * 
     * @param directory
     *            directory to delete
     * @throws IOException
     *             in case deletion is unsuccessful
     */
    private static void deleteDirectory(File directory) throws IOException {
        if (!directory.exists()) {
            return;
        }

        cleanDirectory(directory);
        if (!directory.delete()) {
            String message = "Unable to delete directory " + directory + ".";
            throw new IOException(message);
        }
    }

    /**
     * Return relative path.
     * <p>
     * <b>WARNING:</b></br> Folder path must finished with the character:
     * {@link File.separator}
     * </p>
     * <ul>
     * Example:
     * <table border="1px">
     * <tr>
     * <td>source</td>
     * <td>target</td>
     * <td>relative path</td>
     * </tr>
     * <tr>
     * <td>/root/dir1/dir2/</td>
     * <td>/root/dir1/</td>
     * <td>../</td>
     * </tr>
     * <tr>
     * <td>/root/</td>
     * <td>/root/dir1/dir2/</td>
     * <td>dir1/dir2/</td>
     * </tr>
     * <tr>
     * <td>/root/dir1/dir2/</td>
     * <td>/root/test.xml</td>
     * <td>../../test.xml</td>
     * </tr>
     * </table>
     * 
     * 
     * @param source
     *            path of the source folder
     * @param target
     *            path of the target folder/file
     * @return the relative path
     */
    public static String getRelativePath(String source, String target) {
        // Determine if the target is a file
        String targetFile = null;
        if (!source.endsWith(separator)) {
            throw new IllegalArgumentException();
        }
        if (!target.endsWith(separator)) {
            int lastFolderIndex = target.lastIndexOf(separator) + 1;
            targetFile = target.substring(lastFolderIndex);
            target = target.substring(0, lastFolderIndex);
        }
        String[] sources = source.split(separator);
        String[] targets = target.split(separator);

        // Get the shortest of the two paths
        int length = sources.length < targets.length ? sources.length : targets.length;
        // Get the common part
        int common = 0;
        while (common < length) {
            if (sources[common].equals(targets[common])) {
                common++;
            } else {
                break;
            }
        }
        StringBuilder relativePathBuilder = new StringBuilder();
        for (int i = common; i < sources.length; i++) {
            relativePathBuilder.append(".." + separator);
        }

        for (int i = common; i < targets.length; i++) {
            relativePathBuilder.append(targets[i] + separator);
        }
        if (targetFile != null) {
            relativePathBuilder.append(targetFile);
        }
        return relativePathBuilder.toString();
    }
}
