/**
 * PETALS - PETALS Services Platform.
 * Copyright (c) 2008 OW2 consortium (www.ow2.org)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * $Id: XMLUtil.java 98 2006-02-24 16:18:48Z alouis $
 * -------------------------------------------------------------------------
 */

package com.ebmwebsourcing.easycommons.xml;

import javax.xml.transform.Transformer;

import com.ebmwebsourcing.easycommons.pooling.PoolException;
import com.ebmwebsourcing.easycommons.pooling.PoolPolicy;

/**
 * A static pool of {@link Transformer} to improve {@link Transformer} creation efficiency. 
 * A {@link Transformer} is initialized the first time this class is used.
 * An infinite number (integer maximum value) of {@link Transformer} may be created. They are never evicted.
 * 
 * @ author noddoux
 */
public final class Transformers {

    private static final TransformerResourcePool transformerPool = new TransformerResourcePool(1, Integer.MAX_VALUE, PoolPolicy.WAIT);

    /**
     * Take a {@link Transformer} from the pool
     * 
     * @return one {@link Transformer}
     * 
     * @throws PoolException
     *             if the current thread waits for a {@link Transformer}
     *             of the pool and is interrupted
     */
    public final static Transformer takeTransformer() throws PoolException {
       return transformerPool.take();
    }

    /**
     * Release a {@link Transformer} to the pool
     * 
     * @param transformer a {@link Transformer} to release
     */
    public final static void releaseTransformer(Transformer transformer) {
        transformerPool.release(transformer);
    }
}
