/****************************************************************************
 *
 * Copyright (c) 2009-2012, EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA 
 *
 *****************************************************************************/

package com.ebmwebsourcing.easyviper.core.api.engine.behaviour;

import java.util.List;
import java.util.logging.Logger;

import org.oasisopen.sca.ServiceReference;
import org.oasisopen.sca.annotation.PolicySets;
import org.oasisopen.sca.annotation.Scope;
import org.oasisopen.sca.annotation.Service;
import org.objectweb.fractal.api.Component;

import com.ebmwebsourcing.easycommons.sca.helper.api.SCAException;
import com.ebmwebsourcing.easycommons.sca.helper.impl.SCAHelper;
import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.Execution;
import com.ebmwebsourcing.easyviper.core.api.engine.Node;

/**
 * @author Nicolas Salatge - EBM WebSourcing
 */
@Scope("COMPOSITE")
@Service(value = Behaviour.class, names = "abstract_service")
@PolicySets("frascati:scaEasyPrimitive")
public abstract class AbstractComplexBehaviourImpl extends AbstractBehaviourImpl {

	protected Logger log = Logger.getLogger(AbstractComplexBehaviourImpl.class.getSimpleName());


	protected State executeOnInactive(Execution execution) {
		selectNextNodeToExecute(execution, getNode());
		return State.ACTIVITY_STARTED;
	}

	protected State executeOnStarted(Execution execution) {
		selectNextNodeToExecute(execution, getNode());
		return State.ACTIVITY_ENDED;
	}

	protected State executeOnEnded(Execution execution) {
		selectNextNodeToExecute(execution, null);        
		return State.ACTIVITY_ENDED;
	}


	@Override
	public final Node execute(Execution execution) throws CoreException {
		assert execution != null;
		this.currentExecution = execution;
		this.log.fine(String.format("executing '%s' behaviour on node '%s' (state = '%s')...",
				getName(), getNode().getName(), String.valueOf(state)));
		nextTarget = null;


		if (state == State.ACTIVITY_INACTIVE) {
			this.state = executeOnInactive(execution);
		} else if (state == State.ACTIVITY_STARTED) {
			this.state = executeOnStarted(execution);
		} else if (state == State.ACTIVITY_ENDED) {
			this.state = executeOnEnded(execution);
			currentExecution = null;
		}

		this.log.fine(String.format("executed '%s' behaviour on node '%s' (state = '%s')...",
				getName(), getNode().getName(), String.valueOf(state)));
		if(nextTarget == null) {
			selectNextNodeToExecute(execution, null);
		}
		return nextTarget;
	}


	@Override
	public void reset() {
		System.err.println("^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^ reset " + getName());
		this.state = State.ACTIVITY_INACTIVE;
		List<Node> childNodes = getNode().getChildNodes();  
		for (Node childNode : childNodes) {
			try {
				Component targetComp = SCAHelper.getSCAHelper().getComponentByInterface(
						SCAHelper.getSCAHelper().getParent(this.getComponent()),
						(ServiceReference) childNode, "service");
				if(SCAHelper.getSCAHelper().isStarted(targetComp)) {
					childNode.getBehaviour().reset();
				}
			} catch (SCAException e) {
				throw new CoreException(e);
			}
		}
	}
}
