/****************************************************************************
 *
 * Copyright (c) 2009-2012, EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA 
 *
 *****************************************************************************/
 
package com.ebmwebsourcing.easyviper.core.api.model.registry.definition;

import java.net.URI;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.logging.Logger;

import javax.xml.namespace.QName;

import org.oasisopen.sca.annotation.PolicySets;
import org.oasisopen.sca.annotation.Scope;
import org.oasisopen.sca.annotation.Service;
import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.NoSuchInterfaceException;

import com.ebmwebsourcing.easycommons.sca.helper.api.SCAException;
import com.ebmwebsourcing.easycommons.sca.helper.impl.SCAComponentImpl;
import com.ebmwebsourcing.easycommons.sca.helper.impl.SCAHelper;
import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.Execution.State;
import com.ebmwebsourcing.easyviper.core.api.engine.Process;
import com.ebmwebsourcing.easyviper.core.api.model.Model;
import com.ebmwebsourcing.easyviper.core.api.model.compiler.validation.StaticAnalysis;
import com.ebmwebsourcing.easyviper.core.api.model.compiler.validation.StaticAnalysisPrinter;
import com.ebmwebsourcing.easyviper.core.api.model.registry.ProcessKey;
import com.ebmwebsourcing.easyviper.core.api.soa.message.Message;
import com.ebmwebsourcing.easyviper.core.api.tools.ProcessForker;

/**
 * @author Nicolas Salatge - EBM WebSourcing
 */
//@org.objectweb.fractal.fraclet.annotations.Component
//@Membrane(controller = "primitive")
@Scope("COMPOSITE")
@Service(value=ProcessDefinitionRegistry.class,names="service")
@PolicySets("frascati:scaEasyPrimitive")
public abstract class AbstractProcessDefinitionRegistry<Def extends ProcessDefinition>
extends SCAComponentImpl implements ProcessDefinitionRegistry<Def> {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	private static Logger log = Logger
	.getLogger(AbstractProcessDefinitionRegistry.class.getName());


	/**
	 * The Registry of Process Definitions as a Map
	 */
	protected Map<ProcessKey, Def> registry = new HashMap<ProcessKey, Def>();

	/**
	 * The StaticAnalysis to set after creating the registry
	 */
	private StaticAnalysis staticAnalysis;

	/**
	 * The StaticAnalysisPrinter to set after creating the staticAnalysis
	 */
	private StaticAnalysisPrinter staticAnalysisPrinter;


	public ProcessKey findProcessKey(QName processDefinitionName){
		ProcessKey pk = null;
		for(Entry<ProcessKey, Def> entry : this.registry.entrySet()){
			if(entry.getValue().getQName().equals(processDefinitionName)){
				pk = entry.getKey();
				break;
			}
		}
		return pk;
	}


	public Model getModel() throws CoreException {
		Model res = null;
		try {
			final Component modelComp = SCAHelper.getSCAHelper().getParent(
					this.getComponent());
			res = (Model) modelComp.getFcInterface("service");
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		} catch (SCAException e) {
			throw new CoreException(e);
		}
		return res;
	}


	public StaticAnalysis getStaticAnalysis() {
		return staticAnalysis;
	}

	public void setStaticAnalysis(StaticAnalysis staticAnalysis) {
		this.staticAnalysis = staticAnalysis;
	}

	public StaticAnalysisPrinter getStaticAnalysisPrinter() {
		return staticAnalysisPrinter;
	}

	public void setStaticAnalysisPrinter(StaticAnalysisPrinter staticAnalysisPrinter) {
		this.staticAnalysisPrinter = staticAnalysisPrinter;
	}

	public synchronized ProcessDefinition storeProcessDefinition(final URI uri,
			final ProcessContextDefinition context) throws CoreException {

		Def processDefinition = null;
		processDefinition = (Def) this.getModel().getCompiler().validate(uri);
		AbstractProcessDefinitionRegistry.log.finest("Number of process instances : "
				+ this.getModel().getCore().getEngine()
				.getProcessInstanceRegistry().getProcessInstances()
				.size());
		//TODO Check StaticAnalysis
		if(this.getStaticAnalysis()!=null && this.getStaticAnalysisPrinter() != null){
			if (this.getStaticAnalysis().getErrors().size() > 0) {
				final String msg = this.getStaticAnalysisPrinter().printAll(
						this.getStaticAnalysis().getErrors(),
						this.getStaticAnalysis().getWarnings(),
						this.getStaticAnalysis().getInfos());
				throw new CoreException("Compilation Error: " + msg,
						this.getStaticAnalysis().getErrors().get(0)
						.getError());
			} else if (this.getStaticAnalysis().getWarnings().size() > 0) {
				final String msg = this.getStaticAnalysisPrinter().printAll(
						this.getStaticAnalysis().getErrors(),
						this.getStaticAnalysis().getWarnings(),
						this.getStaticAnalysis().getInfos());
				AbstractProcessDefinitionRegistry.log.finest("Compilation warning: " + msg);
			}

			// create Keys
			List<ProcessKey> keys = null;
			keys = this.createKeys(processDefinition);
			if ((keys == null) || (keys.size() == 0)) {
				throw new CoreException("The keys cannot be null");
			}


			String logMsg = "\nList of key :\n";
			for (final ProcessKey key : keys) {
				logMsg = logMsg + key;
				this.registry.put(key, processDefinition);
			}
			AbstractProcessDefinitionRegistry.log.finest(logMsg);

			if (context.getPoolSize() > 0) {
				final Process processInstance = this.getModel().getCompiler().compile(processDefinition);
				getModel().getCore().getEngine().run(processInstance);
				AbstractProcessDefinitionRegistry.log.info("First process started");
				
				// wait that the first process is ready
				while(processInstance.hasExecution() == false || processInstance.getExecution().getState() != State.SUSPENDED) {
					try {
						Thread.sleep(200);
					} catch (InterruptedException e) {
						// do nothing
					}
				}

				// Pool of instances
				int cpt = 1;
				while (cpt < context.getPoolSize()) {
					ProcessForker forker = new ProcessForker(cpt,
							processDefinition, this, keys);

					this.getModel().getCore().getEngine().getPoolOfProcessForkers().addProcessForkers(forker);
					forker.start();
					AbstractProcessDefinitionRegistry.log.finest("process number" + cpt
							+ " is going to be stored ...");
					cpt++;
				}
			} else {
				throw new CoreException("PoolSize must greater than 0");
			}

		} else {
			AbstractProcessDefinitionRegistry.log.severe("StaticAnalysis and StaticAnalysisPrinter MUST be set for this definition model.");
		}
		return processDefinition;
	}

	public void unStoreProcessDefinition(final URI uri,
			final boolean forceUninstall) throws CoreException {

		Def processDefinition = null;

		processDefinition = (Def) this.getModel().getCompiler().validate(uri);

		// create Keys
		List<ProcessKey> keys = null;
		keys = this.createKeys(processDefinition);
		if (keys != null) {
			for (final ProcessKey key : keys) {
				this.removeProcessDefinition(key, forceUninstall);
			}
		}
	}

	public void setLog(final Logger logger) {
		this.log = logger;
	}



	abstract public List<ProcessKey> createKeys(ProcessDefinition processDefinition) throws CoreException;

	abstract public <D extends ProcessDefinition > boolean isCreateInstance(D def, Message mess) throws CoreException;


}
