/****************************************************************************
 *
 * Copyright (c) 2009-2012, EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA 
 *
 *****************************************************************************/
 
package com.ebmwebsourcing.easyviper.core.impl.engine.behaviour.functionnal;

import java.util.Calendar;
import java.util.Date;

import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.Duration;
import javax.xml.datatype.XMLGregorianCalendar;

import org.osoa.sca.annotations.Property;

import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.Execution;
import com.ebmwebsourcing.easyviper.core.api.engine.behaviour.AbstractComplexBehaviourImpl;
import com.ebmwebsourcing.easyviper.core.api.engine.behaviour.functionnal.WaitBehaviour;
import com.ebmwebsourcing.easyviper.core.api.engine.expression.Expression;
import com.ebmwebsourcing.easyviper.core.impl.engine.time.WaitingThreadPool;

/**
 * @author Nicolas Salatge - EBM WebSourcing
 */
@org.oasisopen.sca.annotation.Scope("COMPOSITE")
@org.oasisopen.sca.annotation.Service(value = WaitBehaviour.class, names = "service")
public class WaitBehaviourImpl extends AbstractComplexBehaviourImpl implements WaitBehaviour {

    @Property(name = "time", required = false)
    private Expression time;

    @Property(name = "date", required = false)
    private Expression date;

    private boolean alreadyWaited;

    public WaitBehaviourImpl() {
    }

    public Expression getDuration() {
        return this.time;
    }

    public void setDuration(final Expression time) {
        this.time = time;
    }

    public Expression getDate() {
        return this.date;
    }

    public void setDate(Expression date) {
        this.date = date;
    }

    @Override
    public void terminateWaiting(Execution execution) throws CoreException {
        alreadyWaited = true;
        selectNextNodeToExecute(execution, null);
        notifyParentBehaviour(execution, this.nextTarget);
        
    }

    @Override
    protected State executeOnInactive(Execution execution) throws CoreException {
        if (alreadyWaited) {
            return State.ACTIVITY_STARTED;
        }

        if (this.time != null) {
            String durationString = getNode().getScope().getExpressionEvaluator()
                    .evaluateAsString(execution, this.time);

            Duration duration;
            try {
                duration = DatatypeFactory.newInstance().newDuration(durationString);
                Long time = duration.getTimeInMillis(Calendar.getInstance());

                return doWait(execution, time);

            } catch (DatatypeConfigurationException e) {
                throw new CoreException("Impossible to evaluate time expression: "
                        + this.time.getContent());
            }

        } else {
            assert date != null;
            String dateString = getNode().getScope().getExpressionEvaluator()
                    .evaluateAsString(execution, this.date);
            XMLGregorianCalendar xmlCalendar;
            try {
                xmlCalendar = DatatypeFactory.newInstance().newXMLGregorianCalendar(dateString);
                Date date = xmlCalendar.toGregorianCalendar().getTime();
                this.log.fine("execute wait behaviour on node: " + getNode().getName()
                        + " => wait until " + date);

                return doWait(execution, date.getTime() - System.currentTimeMillis());

            } catch (DatatypeConfigurationException e) {
                throw new CoreException("Impossible to evaluate date expression: "
                        + this.date.getContent());
            }
        }
    }

    private final State doWait(Execution execution, long time) {
        if (time >= 0) {
            this.log.fine("wait during " + time + " ms");
            execution.setStateRecursively(Execution.State.SUSPENDED);
            WaitingThreadPool.addWaitingThread(execution, this, time);
            selectNextNodeToExecute(execution, null);
            return State.ACTIVITY_STARTED;
        }
        return State.ACTIVITY_ENDED;
    }

    @Override
    protected State executeOnStarted(Execution execution) throws CoreException {
        if (alreadyWaited) {
            alreadyWaited = false;
            return State.ACTIVITY_ENDED;
        }
        return State.ACTIVITY_STARTED;
    }
    

    @Override
    protected State executeOnEnded(Execution execution) throws CoreException {
        alreadyWaited = false;
        return super.executeOnEnded(execution);
    }
    

}
