/****************************************************************************
 *
 * Copyright (c) 2009-2012, EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA 
 *
 *****************************************************************************/

package com.ebmwebsourcing.easyviper.core.impl.engine;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.logging.Logger;

import org.oasisopen.sca.ServiceReference;
import org.oasisopen.sca.annotation.PolicySets;
import org.oasisopen.sca.annotation.Reference;
import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.Interface;
import org.objectweb.fractal.api.NoSuchInterfaceException;
import org.objectweb.fractal.fraclet.annotations.Lifecycle;
import org.objectweb.fractal.fraclet.types.Step;
import org.objectweb.fractal.julia.ComponentInterface;
import org.objectweb.fractal.julia.Interceptor;
import org.ow2.frascati.tinfi.oasis.ServiceReferenceImpl;

import com.ebmwebsourcing.easycommons.sca.helper.api.SCAException;
import com.ebmwebsourcing.easycommons.sca.helper.impl.SCAComponentImpl;
import com.ebmwebsourcing.easycommons.sca.helper.impl.SCAHelper;
import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.Execution;
import com.ebmwebsourcing.easyviper.core.api.engine.Node;
import com.ebmwebsourcing.easyviper.core.api.engine.Process;
import com.ebmwebsourcing.easyviper.core.api.engine.Scope;
import com.ebmwebsourcing.easyviper.core.api.engine.behaviour.Behaviour;

/**
 * @author Nicolas Salatge - EBM WebSourcing
 */
@org.oasisopen.sca.annotation.Scope("COMPOSITE")
@org.oasisopen.sca.annotation.Service(value = Node.class, names = "service")
@PolicySets("frascati:scaEasyPrimitive")
public class NodeImpl extends SCAComponentImpl implements Node {

	private Logger log = Logger.getLogger(NodeImpl.class.getSimpleName());

	/**
	 * child childNodes
	 */
	@Reference(name = "childNodes", required = false)
	protected List<Node> childNodes = Collections.synchronizedList(new ArrayList<Node>());

	@Reference(name = "incomingNodes", required = false)
	protected List<Node> incomingNodes = Collections.synchronizedList(new ArrayList<Node>());

	@Reference(name = "outgoingNodes", required = false)
	protected List<Node> outgoingNodes = Collections.synchronizedList(new ArrayList<Node>());

	@Reference(name = "behaviour", required = false)
	protected Behaviour behaviour;

	@Reference(name = "parentNode", required = false)
	protected Node parentNode;

	/**
	 * Default constructor
	 * 
	 * @throws CoreException
	 */
	public NodeImpl() {
		super();
	}

	/**
	 * Stop the NodeImpl behaviour
	 * 
	 * @throws CoreException
	 */
	@Lifecycle(step = Step.STOP)
	public void stopSCAComponent() throws SCAException {
		this.log.fine("node " + this.getName() + " is stopping... ");
		if ((this.childNodes != null) && (this.childNodes.size() > 0)) {
			final Component parentComp = SCAHelper.getSCAHelper().getParent(this.getComponent());
			for (final Node child : this.getChildNodes()) {
				this.log.fine("stop child node: " + this.getName());

				final Component nodeComp = SCAHelper.getSCAHelper().getComponentByInterface(
						parentComp, (ServiceReference<?>) child, "service");
				if ((nodeComp != null) && (SCAHelper.getSCAHelper().isStarted(nodeComp))) {
					SCAHelper.getSCAHelper().stopComponent(nodeComp);
				}
			}
		}
		this.log.fine("node stopped: " + this.getName());
	}

	public void setActivity(final Behaviour behaviour) {

		this.behaviour = behaviour;
	}

	public Behaviour getActivity() {
		return this.behaviour;
	}

	@Override
	public final Node execute(Execution execution) throws CoreException {
		try {
			if (this.behaviour != null) {

				// start the behaviour
				Component activityComp = null;
				if (this.behaviour instanceof ServiceReference<?>) {
					activityComp = SCAHelper.getSCAHelper().getComponentByInterface(
							getScope().getComponent(), (ServiceReference<?>) this.behaviour,
							"service");
				} else {
					if (getScope() != null) {
						activityComp = SCAHelper.getSCAHelper().getComponentByInterface(
								getScope().getComponent(),
								(org.objectweb.fractal.api.Interface) this.behaviour, "service");
					}
				}
				if ((activityComp != null) && (!SCAHelper.getSCAHelper().isStarted(activityComp))) {
					SCAHelper.getSCAHelper().startComponent(activityComp);
				}

				Node node = this.behaviour.execute(execution);
				return node;
			}

		} catch (SCAException e1) {
			throw new CoreException(e1);
		}
		return null;

	}

	public Node getParentNode() {
		return this.parentNode;
	}

	public void setParentNode(Node parentNode) {
		this.parentNode = parentNode;
	}

	public List<Node> getChildNodes() {
		return new ArrayList<Node>(this.childNodes);
	}

	public Behaviour getBehaviour() {

		if (this.behaviour instanceof Interface) {
			return this.behaviour;
		}

		/*
		 * The solution implemented below is linked to a difference between
		 * Fraclet/Julia and Tinfi.
		 * 
		 * With Fraclet/Julia, when injecting a reference in a @Required
		 * annotated field, the reference of the target bound interface is
		 * injected.
		 * 
		 * With Tinfi, the ServiceReference of the source component is injected.
		 * 
		 * Let getBehaviour return the target bound interface.
		 */
		if (this.behaviour != null) {
			ServiceReference<?> sr = (ServiceReference<?>) this.behaviour;
			ServiceReferenceImpl<?> cri = (ServiceReferenceImpl<?>) sr;
			Object service = cri._getDelegate();
			ComponentInterface ci = (ComponentInterface) service;
			Interceptor intercep = (Interceptor) ci.getFcItfImpl();
			Behaviour itf = (Behaviour) intercep.getFcItfDelegate();
			return itf;
		} else {
			return null;
		}
	}

	@Override
	public String toString() {
		String res = "";
		res = "node " + this.getName();
		return res;
	}

	public List<Node> getIncomingNodes() {
		return this.incomingNodes;
	}

	public List<Node> getOutgoingNodes() {
		return this.outgoingNodes;
	}

	public void setLog(final Logger logger) {
		this.log = logger;
	}

	public Logger getLogger() {
		return this.log;
	}

	@Override
	public Scope getScope() {
		Scope res = null;
		try {
			final Component scopeComponent = SCAHelper.getSCAHelper().getParent(getComponent());
			if (!(scopeComponent.getFcInterface("service") instanceof Scope)) {
				return null;
			}
			res = (Scope) scopeComponent.getFcInterface("service");
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		} catch (final SCAException e) {
			throw new CoreException(e);
		}
		return res;
	}

	@Override
	public com.ebmwebsourcing.easyviper.core.api.engine.Process getProcess() {
		Scope scope = getScope();
		while (true) {
			assert scope != null;
			if (scope instanceof Process)
				break;
			scope = scope.getScope();
		}
		return (Process) scope;
	}

}
